/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

#ifndef COM_ROTATE_H
#define COM_ROTATE_H

#include <stdlib.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include <math.h> //??

/**
 * \ingroup com_resize_image
 * \defgroup rotate Rotate an image
 */

/**
 * \ingroup rotate
 * \file com_rotate.h
 * \author Felicity Lodge
 *   
 * \brief Rotate an EFIDIR image ANTICLOCKWISE through a given angle.
 * 
 *
  \verbatim
 	Parameters:
 		--angle float 		: Input angle of rotation (degrees)
 		--inputImage string	: Input image file name 
 		--outputImage string	: Output image file name 
 		[--desc string]		: Output imagefile description ??
  \endverbatim
 *
 * \return Rotated image in an enlarged space.
 *
 */

#ifndef PI
	#define PI 4.0*atan(1.0);
#endif

#ifndef max
	#define max( a, b ) ( ((a) > (b)) ? (a) : (b) )
#endif

#ifndef min
	#define min( a, b ) ( ((a) < (b)) ? (a) : (b) )
#endif

/**
 * \ingroup rotate
 * \struct rotate_param
 * \brief structure that contains the parameters for the rotation functions
 *
 */

typedef struct rotate_param
{
  double angle;
}rotate_param,*Rotate_param;

/**
 * \ingroup rotate
 * \fn void define_rotation_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_rotation_param(char *extra_description);

/**
 * \ingroup rotate
 * \fn Rotate_param get_rotation_param();
 * \brief Default efidir function that gets parameters
 *  
 * \return A Rotate_param that contains the parameters
 *
 */
Rotate_param get_rotation_param();

/**
 * \ingroup rotate
 * \fn Rotate_param new_rotation_param();
 * \brief Create a new Rotate_param reference
 * \return A new reference on allocated rotate_param structure
 *
 */
Rotate_param new_rotation_param();

/**
 * \ingroup rotate
 * \fn void free_rotation_param(Rotate_param p);
 * \brief Free an allocated Rotate_param reference
 * \param p A reference on a rotate_param structure
 *
 */
void free_rotation_param(Rotate_param p);

/**
 * \ingroup rotate
 * \fn void rotate(Rotate_param param, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief Rotate an image through a given angle
 * 
 * The rotate operator rotates the image anticlockwise about the origin then translates the enlarged box containing the rotated image back onto positive axes. In practice, inverse mapping is used to reduce image degradation.
 *
 * \param param is the structure of rotate parameters
 * \param inputImage is the image to be rotated
 * \param outputImage is the resulting rotated image
 *
 */
void rotate(Rotate_param param, EFIDIRImage inputImage, EFIDIRImage outputImage);

#endif /* COM_ROTATE_H */


