/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */


/**
 * \ingroup statistic
 * \defgroup local_stat Basic local statistics estimated on a sliding window
 */
/**
 * \ingroup local_stat
 * \file com_local_stat.h 
 * \author Emmanuel Trouve
 * 
 * \brief basic local statistics estimated on a sliding window  
 * 
 * Input image : amplitude of a SAR image (data type : Unsigned short or float) 
 * Output image : statistical parameter computed on a sliding window (data type : float). 
 */

#ifndef COM_LOCAL_STAT_H
#define COM_LOCAL_STAT_H

#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup local_stat
 * \typedef pt_local_stat_param
 *
 * \brief reference on local_stat structure
 *
 */
/**
 * \ingroup local_stat
 * \struct local_stat_param
 *
 * \brief structure that contains the parameters of the operator local_stat
 *
 */
typedef struct struct_local_stat_param{
  int ws_dimX;  /*!< window size : number of columns */
  int ws_dimY;  /*!< window size : number of rows */
  int stat_id;  /*!< statistical feature ID (1:mean=E[x], 2:E[x^2], 3:sigma^2, 4:variation_coefficient=3/1^2, 5:Equivalent_Nb_Look=1/4)  */
}local_stat_param, *pt_local_stat_param;

/**
 * \ingroup local_stat
 * \fn new_local_stat_param()
 *
 * \return A new reference on allocated local_stat_param structure
 *
 * \brief Create a new pt_local_stat_param reference 
 *
 */
static pt_local_stat_param new_local_stat_param();
/**
 * \ingroup local_stat
 * \fn free_local_stat_param(pt_local_stat_param p)
 *
 * \param p A reference on a local_stat_param structure
 *
 * \brief Free an allocated pt_local_stat_param reference 
 *
 */
void free_local_stat_param(pt_local_stat_param p);


/**
 * \ingroup local_stat
 * \fn void define_local_stat_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_local_stat_param(char *extra_description);

/**
 * \ingroup local_stat
 * \fn pt_local_stat_param get_local_stat_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A pt_local_stat_param structure that contains the parameters.
 */
pt_local_stat_param get_local_stat_param();

/**
 * \ingroup local_stat
 * \fn void local_stat(pt_local_stat_param p, EFIDIRImage inputImage, EFIDIRImage resImage);
 * \brief function which computes the output image as : outputImage = gain*inputImage + offset  
 * 
 * 
 *
 * \param p  structure which contains operator parameters  
 * \param inputImage : input image
 * \param resImage : output image
 *
 */
void local_stat(pt_local_stat_param p, EFIDIRImage inputImage, EFIDIRImage resImage);


#endif
