/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * \ingroup common 
 * \defgroup interp Program to calculate interpolated values of an atmosphere file. It can interpolate columns, rows or levels. Effect a 2D interpolation by interpolating first in one direction, followed by a second interpolation. It writes output to an EFIDIR Atmosphere file type .atm
 * \file com_interp.h
 * \author Felicity Lodge
 * 
 * \brief EFIDIR atmosphere manager calculate interp values of EFIDIR atmospheres
 */

#ifndef __COM_INTERP_H__
#define __COM_INTERP_H__

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>

#include "efidir_param.h"
#include "efidir_atmosphere.h"

/**
 * \ingroup interp
 * \struct interp_param
 * \brief structure that contains the parameters for the interp functions
 */

typedef struct interp_param {

  int interp_axis;   /*axis to interpolate  1=row 2=column 3=level*/
  int n_interp;     /*number of interpolation points in altitude*/
  double min;      /*minimum altitude for interpolation*/
  double max;      /*maximum altitude for interpolation*/
  int axis_a;     /*axis a for interpolation*/
  int axis_b;     /*axis b for interpolation*/
  int axis_c;     /*axis c for interpolation*/
  float *c_points; /*data points on c axis*/ 

} interp_param,*Interp_param;


/**
 * \ingroup interp
 * \struct interp_atmospheric_data
 * \brief structure that contains the atmospheric data for the interp function. It holds an EFIDIRAtmosphere structure.
 */

typedef struct interp_atmospheric_data {
  EFIDIRAtmosphere atm; /*!< EFIDIRAtmosphere structure containing Altitude data */
  EFIDIRAtmosphere temp; /*!< EFIDIRAtmosphere structure containing Temperature data */
  EFIDIRAtmosphere rel_hum; /*!< EFIDIRAtmosphere structure containing Relative Humidity data */
} interp_atmospheric_data,*Interp_atmospheric_data;

/**
 * \ingroup interp
 * \struct interp_data
 * \brief structure that contains the calculated interp data. It is held in 2 separate EFIDIRAtmosphere structures of wwpp & wvmr??
 */

typedef struct interp_data {
  EFIDIRAtmosphere wet_m; /*!< EFIDIRAtmosphere structure containing ???? data 	SAME AS ABOVE?? water vapour partial pressure*/
  EFIDIRAtmosphere dry_m; /*!< EFIDIRAtmosphere structure containing ???? data */
  EFIDIRAtmosphere wet_rad; /*!< EFIDIRAtmosphere structure containing ???? data 	SAME AS ABOVE?? water vapour partial pressure*/
  EFIDIRAtmosphere dry_rad; /*!< EFIDIRAtmosphere structure containing ???? data */
} interp_data,*Interp_data;

/**
 * \ingroup interp
 * \fn void define_interp_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_interp_param(char *extra_description);
/**
 * \ingroup interp
 * \fn void define_interp_atmospheric_data(char *extra_description);
 * \brief Default efidir function that defines the atmospheric input data
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_interp_atmospheric_data(char *extra_description);
/**
 * \ingroup interp
 * \fn void define_interp_data(char *extra_description);
 * \brief Default efidir function that defines the output data
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_interp_data(char *extra_description);
/**
 * \ingroup interp
 * \fn Interp_param get_interp_param();
 * \brief Default efidir function that gets parameters
 *  
 * \return An Interp_param that contains the parameters
 *
 */
Interp_param get_interp_param();
/**
 * \ingroup interp
 * \fn Interp_atmospheric_data get_interp_atmospheric_data();
 * \brief Default efidir function that gets the input atmospheric data
 *  
 * \return A Interp_atmospheric_data structure that contains the atmospheric data
 *
 */
Interp_atmospheric_data get_interp_atmospheric_data();
/**
 * \ingroup interp
 * \fn Interp_data get_interp_data();
 * \brief Default efidir function that sets up structure for output data
 *  
 * \return A Interp_data structure that contains the meta-data for the output
 *
 */
Interp_data get_interp_data();
/**
 * \ingroup interp
 * \fn Interp_param new_interp_param();
 * \brief Create a new Interp_param reference
 * \return A new reference on allocated interp_param structure
 *
 */
Interp_param new_interp_param();
/**
 * \ingroup interp
 * \fn Interp_atmospheric_data new_interp_atmospheric_data();
 * \brief Create a new Interp_atmospheric_data reference
 * \return A new reference on allocated interp_atmospheric_data structure
 *
 */
Interp_atmospheric_data new_interp_atmospheric_data();

/**
 * \ingroup interp
 * \fn Interp_data new_interp_data();
 * \brief Create a new Interp_data reference
 * \return A new reference on allocated interp_data structure
 *
 */
Interp_data new_interp_data();

/**
 * \ingroup interp
 * \fn void free_interp_param(Interp_param p);
 * \brief Free an allocated Interp_param reference
 * \param p A reference on a interp_param structure
 *
 */
void free_interp_param(Interp_param p);
/**
 * \ingroup interp
 * \fn void free_interp_atmospheric_data(Interp_atmospheric_data p);
 * \brief Free an allocated Interp_atmospheric_data reference
 * \param p A reference on a interp_atmospheric_data structure
 *
 */
void free_interp_atmospheric_data(Interp_atmospheric_data p);
/**
 * \ingroup interp
 * \fn void free_interp_data(Interp_data p);
 * \brief Free an allocated Interp_data reference
 * \param p A reference on a interp_data structure
 *
 */
void free_interp_data(Interp_data p);

/**
 * \ingroup interp
 * \fn void interp(Interp_param param, Interp_atmospheric_data input_data, Interp_data  output_data);
 * \brief The interp operator calculates the interp due to atmospheric conditions.
 *
 * \param input_data is a structure containing the atmospheric data required to calculate the interpolation. It holds an EFIDIRAtmosphere structure
 * \param output_data is a structure containing the resulting calculated interpolated function. It holds an EFIDIRAtmosphere structure
 * \param param is the structure containing parameters needed to calculate the interpolation 
 *
 */
void interp(Interp_param param, Interp_atmospheric_data input_data, Interp_data  output_data);
/**
 * \ingroup interp
 * \fn double interp_calc(Interp_param param, double *pres_p, double *alt_p, double *temp_p, double *rh_p);
 * \brief Calculate the interp at a single point (longitude and latitude).
 *
 *are arrays containing pressure, altitude, temperature and relative humidity
 *
 * \param param is a structure containing the required constants for the interp calculation
 * \param pres_p array ...
 * \param alt_p array ...
 * \param temp_p array ...
 * \param rh_p array ...
 *
 * \return void 
 *
 */
double interp_calc(Interp_param param, double *pres_p, double *alt_p, double *temp_p, double *rh_p);

/**
 * \ingroup interp
 * \fn void set_interp_axes(Interp_param param, Interp_atmospheric_data input_data);
 * \brief set up the data for the interpolation on the correct axis.
 *
 * \param param is a structure containing the parameters for the interpolation
 * \param input_data is a structure containing the data values to be interpolated
 *
 * \return void 
 *
 */
void set_interp_axes(Interp_param param, Interp_atmospheric_data input_data);

#endif //__COM_INTERP_H__
