/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup display
 * \defgroup imagescale Generate a 8bit colorful image and its scale image from a float image
 */
/**
 * \ingroup imagescale
 * \file com_imagescale.h
 * \author Yajing Yan (LISTIC)
 * 
 * \brief operator of generation of scale image
 * 
 * file includes (stucture, prototype de fonction) of the operator imagescale
 * 
 * 
 */

#ifndef COM_IMAGESCALE_H
#define COM_IMAGESCALE_H

#include <stdlib.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_string.h"


/**
 * \ingroup imagescale
 * \typedef Imagescale_param
 *
 * \brief reference on imagescale_param structure
 *
 */
/**
 * \ingroup imagescale
 * \struct imagescale_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_imagescale_param{
  int index; /*!< index of processing, 1 - magnitude of displacement, 2 - orientation of displacement */
  int r; /*!< number of lines of scale image */
  int c; /*!< number of columns of scale image, 256 is recommended*/
  int mask_index; /*!< index of mask processing */
  float max_val; /*!< maximum value of interval */
  float min_val; /*!< minimum value of interval */
  float max_scale; /*!< maximum value of scale */
  float min_scale; /*!< minimum value of scale */
}imagescale_param, *Imagescale_param;

/**
 * \ingroup imagescale
 * \fn new_imagescale_param()
 *
 * \return A new reference on allocated imagescale_param structure
 *
 * \brief Create a new Imagescale_param reference 
 *
 */
static Imagescale_param new_imagescale_param();
/**
 * \ingroup imagescale
 * \fn free_imagescale_param(Imagescale_param p)
 *
 * \param p A reference on a imagescale_param structure
 *
 * \brief Free an allocated Imagescale_param reference 
 *
 */
void free_imagescale_param(Imagescale_param p);


/**
 * \ingroup imagescale
 * \fn void define_imagescale_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_imagescale_param(char *extra_description);

/**
 * \ingroup imagescale
 * \fn Imagescale_param get_imagescale_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Imagescale_param that contains the parameters.
 */
Imagescale_param get_imagescale_param();

/**
 * \ingroup imagescale
 * \fn void imagescale(Imagescale_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputConf, char *colortable, EFIDIRImage outputImage, EFIDIRImage scaleImage);
 * \brief function of generation of a colorful image and its scale image
 * Function allowing to generate a colorful image and its scale image from a float image of displacement
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the float image to input
 * \param inputMask is the mask of the image to input
 * \param inputConf is the confidence image
 * \param colortable is the color look up table 
 * \param outputImage is the colorful image corresponding to the float image
 * \param scaleImage is the scale image of the colorful image
 *
 */
void imagescale(Imagescale_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputConf, char *colortable, EFIDIRImage outputImage, EFIDIRImage scaleImage);

/**
 * \ingroup imagescale
 * \fn void orientation(Imagescale_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputConf, char *colortable, EFIDIRImage outputImage, EFIDIRImage scaleImage);
 * \brief function of generation of a colorful image and its scale image
 * Function allowing to generate a colorful image and its scale image from a float image of orientation
 *
 * \param p is the structure of the parameters for the operator 
 * \param inputImage is the float image to input
 * \param inputMask is the mask of the image to input
 * \param inputConf is the input confidence image
 * \param colortable is the color look up table
 * \param outputImage is the colorful image corresponding to the float image
 * \param scaleImage is the scale image of the colorful image
 *
 */
void orientation(Imagescale_param p, EFIDIRImage inputImage, EFIDIRImage inputMask, EFIDIRImage inputConf, char *colortable, EFIDIRImage outputImage, EFIDIRImage scaleImage);


/**
 * \ingroup imagescale
 * \fn void colScale(Imagescale_param p, char *colortable, EFIDIRImage scaleImage);
 * \brief function of generation of a scale image from a color table
 * Function allowing to generate a scale image from a defined color table
 *
 * \param p is the structure of the parameters for the operator 
 * \param colortable is color table file
 * \param scaleImage is the scale image of the colorful image
 *
 */
 void colScale(Imagescale_param p, char *colortable, EFIDIRImage scaleImage);

#endif
