/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * \ingroup common_operators
 * \defgroup gain_offset Apply a gain and an offset on an image
 */
/**
 * \ingroup gain_offset
 * \file com_gain_offset.h 
 * \author Emmanuel Trouve (LISTIC)
 * 
 * \brief programing example : multiplication of an image by a constant (gain) followed by addition of a constant (offset) 
 * 
 * Gain and an offset operation followed by modulo: 
 * outputImage  = [ gain*inputImage + offset ] mod modulo if modulo > 0. 
 * OutputImage type can be different from inputImage type. 
 * Values are rounded and troncated to max and min values for integer types.
 * Works on complexe images (same operations on real and imaginary parts).
 * Does not work on multi-band images
 */

#ifndef COM_GAIN_OFFSET_H
#define COM_GAIN_OFFSET_H

#include <stdlib.h>
#include <string.h>
#include <limits.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup gain_offset
 * \typedef Gain_offset_param
 *
 * \brief reference on gain_offset_param structure
 *
 */
/**
 * \ingroup gain_offset
 * \struct gain_offset_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_gain_offset_param{
  double gain; /*!< gain value */
  double offset; /*!< offset value */
  double modulo; /*!< modulo value (>0, 0:ignored)*/
  int output_image_type;  /*!< type of output image (ENVI data type, 0:same as input)*/
  int band; /*!<band for operation, null for all bands*/
}gain_offset_param, *Gain_offset_param;

/**
 * \ingroup gain_offset
 * \fn new_gain_offset_param()
 *
 * \return A new reference on allocated gain_offset structure
 *
 * \brief Create a new Gain_offset reference 
 *
 */
Gain_offset_param new_gain_offset_param();
/**
 * \ingroup gain_offset
 * \fn free_gain_offset_param(Gain_offset_param p)
 *
 * \param p A reference on a gain_offset structure
 *
 * \brief Free an allocated Gain_offset reference 
 *
 */
 void free_gain_offset_param(Gain_offset_param p);


/**
 * \ingroup gain_offset
 * \fn void define_gain_offset_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_gain_offset_param(char *extra_description);

/**
 * \ingroup gain_offset
 * \fn Gain_offset_param get_gain_offset_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Gain_offset_param structure that contains the parameters.
 */
Gain_offset_param get_gain_offset_param();

/**
 * \ingroup gain_offset
 * \fn void gain_offset(Gain_offset_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief function which computes the output image as : outputImage = gain*inputImage + offset  
 * 
 * 
 *
 * \param p  structure which contains operator parameters  
 * \param inputImage : input image
 * \param outputImage : output image
 *
 */
void gain_offset(Gain_offset_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);


#endif
