/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 
 /**
 * \ingroup common 
 * \defgroup fft_manager FFTW manager
 * \file com_fftw.h
 * \author Stephane Guillaso / ENS-GeoLab
 * 
 * \brief EFIDIR fft manager
 */
#ifndef __COM_FFTW_H__
#define __COM_FFTW_H__

#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_matrix.h"
#include <stdlib.h>

/**
 * \ingroup fft_manager
 * \def FFTW_LINEWISE
 * \brief used to calculate 1D FFT transformation over 2D array over each line
 */
#define FFTW_LINEWISE 1

/**
 * \ingroup fft_manager
 * \def FFTW_COLUMNWISE
 * \brief used to calculate 1D FFT transformation over 2D array over each column
 */
#define FFTW_COLUMNWISE 2

/**
 * \ingroup fft_manager
 * \fn void fftw1d(Complex8 *in, Complex8 *out, int ncol, int direction);
 * \brief 1D DFT transformation over 1D array
 *
 * \param in one dimensional input array
 * \param out one dimensional output array containing the fft (forward or backward) of the input array
 * \param ncol indicates the size of the input/output array.
 * \param direction indicates the direction to make the FFT : FFTW_FORWARD (for forward), FFTW_BACKWARD (for backward)
 *
 * \b fftw1d can be used to generate a DFT (or FFT) of an #Complex8 input array. It will put
 * the result into a second #Complex8 array having the same size than the input one. The size is
 * given by the third parameter of the function. Please note that due to improvement of FFTW, it 
 * is not necessary, to be fast, that the input/output arrays have a 2^N size. FFTW itself takes
 * into account this particularity.
 *
 * Below is an example "how to" use \b fftw1d.
 *
 */
/*
 * \include ex_fftw1d.c
 */
void fftw1d(Complex8 *in, Complex8 *out, int ncol, int direction);

/**
 * \ingroup fft_manager
 * \fn void fftw1d_arr(Complex8 **in, Complex8 **out, int nlig, int ncol, int dimension, int direction);
 * \brief 1D DFT transformation over 2D array
 *
 * \param in 2D input array
 * \param out 2D output array containing the fft (forward or backward) of the input array
 * \param nlig (int) number of line of the input/output array
 * \param ncol (int) number of column for each line of input/output array
 * \param dimension parameter indicated the dimension across which the FFT is calculated ( #FFTW_LINEWISE or #FFTW_COLUMNWISE).
 * \param direction indicates the direction to make the FFT : FFTW_FORWARD (for forward), FFTW_BACKWARD (for backward)
 *
 * \b fftw1d_arr can be used to generate a DFT line- or columnwise over an 2D array. 
 * This function is faster than \b fftw1d line by line. It is not necessary that input/output arrays have
 * a 2^N 2D size.
 *
 * To perform a DFT linewise use the command FFTW_LINEWISE.
 *
 * To perform a DFT columnwise use the command FFTW_COLUMNWISE.
 *
 * FFTW_FORWARD and FFTW_BACKWARD are used to indicate the direction of the FFT.
 *
 * An example how to use \b fftw1d_arr follows:
 */
/*
 * \include ex_fftw1d_arr.c 
 *
 */
void fftw1d_arr(Complex8 **in, Complex8 **out, int nlig, int ncol, int dimension, int direction);

/**
 * \ingroup fft_manager
 * \fn fftw2d(Complex8 **in, Complex8 **out, int nlig, int ncol, int direction);
 * \brief 2D DFT transformation over 2D array
 *
 * \param in 2D input array
 * \param out 2D output array containing the fft (forward or backward) of the input array
 * \param nlig (int) number of line of the input/output array
 * \param ncol (int) number of column for each line of input/output array
 * \param direction indicates the direction to make the FFT : FFTW_FORWARD (for forward), FFTW_BACKWARD (for backward)
 *
 * \b fftw2d can be used to generate a 2D of an 2D array. 
 * FFTW_FORWARD and FFTW_BACKWARD are used to indicate the direction of the FFT.
 *
 * An example how to use \b fftw2d follows:
 */
/*
 * \include ex_fftw2d.c 
 *
 */
void fftw2d(Complex8 **in, Complex8 **out, int nlig, int ncol, int direction);

#endif
