/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup common_operators
 * \defgroup difference Calculate the difference between two images
 */
/**
 * \ingroup difference
 * \file com_difference.h
 * \author Yajing Yan and Flavien Vernier (LISTIC)
 * 
 * \brief operator of calculation of difference between two images
 * 
 * file includes (stucture, prototype de fonction) of the operator diff
 * 
 * 
 */

#ifndef COM_DIFFERENCE_H
#define COM_DIFFERENCE_H

#include <stdio.h>
#include <string.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup diff
 * \typedef Diff_param
 *
 * \brief reference on diff_param structure
 *
 */
/**
 * \ingroup diff
 * \struct diff_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct struct_diff_param{
  int index; /*!< intern parameter */
  int shiftX; /*!< shift in X of slave with respect to master*/
  int shiftY; /*!< shift in Y of slave with respect to master*/
  int startX; /*!< first column of master to process */
  int startY; /*!< first line of master to process */
  int nbX; /*!< number of columns of master to process */
  int nbY; /*!< number of lines of master to process */
  float lambda;  /*!< wavelength of signal */
}diff_param, *Diff_param;

/**
 * \ingroup diff
 * \fn Diff_param new_diff_param()
 *
 * \return A new reference on allocated diff_param structure
 *
 * \brief Create a new Diff_param reference 
 *
 */
static Diff_param new_diff_param();
/**
 * \ingroup diff
 * \fn void free_diff_param(Diff_param p)
 *
 * \param p A reference on a diff_param structure
 *
 * \brief Free an allocated Diff_param reference 
 *
 */
void free_diff_param(Diff_param p);


/**
 * \ingroup diff
 * \fn void define_diff_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_diff_param(char *extra_description);

/**
 * \ingroup diff
 * \fn Diff_param get_diff_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Diff_param that contains the parameters.
 */
Diff_param get_diff_param();

/**
 * \ingroup diff
 * \fn void diff1(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage SlaveMask, EFIDIRImage DiffImage, EFIDIRImage DiffMask);
 * \brief DEPRECATED function of generation of difference of two images with mask for both images.
 *
 * Function allowing to calculate the difference between master image and slave image.
 * The masks are defined as follow: 0 for masked (not computed) values and 1 for computed values.
 *
 * \warning Compute Master - Slave except if lambda is defined. RAD2MET(Slave, lambda) - Master is computed if lambda is defined.
 *
 * \param p is the structure of the parameters for the operator 
 * \param MasterImage is the master image
 * \param MasterMask is the master mask
 * \param SlaveImage is the slave image
 * \param SlaveMask is the slave mask
 * \param DiffImage is the difference image
 * \param DiffMask is the difference mask
 *
 */
void diff1(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage SlaveMask, EFIDIRImage DiffImage, EFIDIRImage DiffMask) __attribute__ ((deprecated));


/**
 * \ingroup diff
 * \fn void diff2(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage DiffImage, EFIDIRImage DiffMask);
 * \brief DEPRECATED function of generation of difference of two images with mask for master image
 *
 * Function allowing to calculate the difference between master image and slave image
 * The masks are defined as follow: 0 for masked (not computed) values and 1 for computed values.
 *
 * \warning Compute Master - Slave except if lambda is defined. RAD2MET(Slave, lambda) - Master is computed if lambda is defined.
 *
 * \param p is the structure of the parameters for the operator 
 * \param MasterImage is the master image
 * \param MasterMask is the master mask
 * \param SlaveImage is the slave image
 * \param DiffImage is the difference image
 * \param DiffMask is the difference mask
 *
 */
void diff2(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage DiffImage, EFIDIRImage DiffMask) __attribute__ ((deprecated));


/**
 * \ingroup diff
 * \fn void diff3(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage SlaveImage, EFIDIRImage DiffImage);
 * \brief DEPRECATED function of generation of difference of two images (same dimension)
 * Function allowing to calculate the difference between master image and slave image
 *
 * \warning Compute Master - Slave except if lambda is defined. RAD2MET(Slave, lambda) - Master is computed if lambda is defined.
 *
 * \param p is the structure of the parameters for the operator 
 * \param MasterImage is the master image
 * \param SlaveImage is the slave image
 * \param DiffImage is the difference image
 *
 */
void diff3(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage SlaveImage, EFIDIRImage DiffImage) __attribute__ ((deprecated));

/**
 * \ingroup diff
 * \fn void difference(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage SlaveMask, EFIDIRImage DiffImage, EFIDIRImage DiffMask, EFIDIRImage shiftImage);
 * \brief function of generation of difference of two images
 *
 * Function allowing to calculate the difference between master image and slave image  with or without mask for master or both images.
 * A shift image can be set to compute the difference: master(x,y) - slave(x+dx,y+dy).
 * The masks are defined as follow: 0 for masked (not computed) values and 1 for computed values.
 *
 * \warning Compute Master - Slave except if lambda is defined. RAD2MET(Slave, lambda) - Master is computed if lambda is defined.
 *
 * \param p is the structure of the parameters for the operator 
 * \param MasterImage is the master image
 * \param MasterMask is the master mask
 * \param SlaveImage is the slave image
 * \param SlaveMask is the slave mask
 * \param DiffImage is the difference image
 * \param DiffMask is the difference mask
 * \param shiftImage is the shift image
 *
 */
void difference(Diff_param p, EFIDIRImage MasterImage, EFIDIRImage MasterMask, EFIDIRImage SlaveImage, EFIDIRImage SlaveMask, EFIDIRImage DiffImage, EFIDIRImage DiffMask, EFIDIRImage shiftImage);
#endif
