/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */
/**
 * \ingroup atmosphere_operators
 * \defgroup delay Calculate Delay
 * \file com_delay.h
 * \author Felicity Lodge
 * 
 * \brief EFIDIR atmosphere manager calculate delay due to atmospheric conditions
 */

#ifndef __COM_DELAY_H__
#define __COM_DELAY_H__

#include <string.h>
#include <stdlib.h>
#include <math.h>
#include <gsl/gsl_errno.h>
#include <gsl/gsl_spline.h>

#include "efidir_param.h"
#include "efidir_atmosphere.h"

/**
 * \ingroup delay
 * \struct delay_param
 * \brief structure that contains the parameters for the delay functions
 */

typedef struct delay_param {

//  char *type;             /*!< Name of data type */
//  int nc;                 /*!< Number of loaded atmosphere data columns */
//  int nr;                 /*!< Number of loaded atmosphere data rows */
//  int nl;                 /*!< Number of loaded atmosphere data levels */
//  int nt;                 /*!< Number of loaded atmosphere data times */
//  int nD;                 /*!< 2D or 3D*/
//  int columns;            /*!< Number of samples in atmosphere file */
//  int lines;              /*!< Number of lines in atmosphere file */
//  int levels;             /*!< Number of levels in atmosphere file */
//  int times;              /*!< Number of date in atmosphere file */
//  int nc0;
//  int nr0;
//  int nl0;
//  int nt0;

  double hmin;             /*!< Min altitude for calculation m */
  double hmax;          /*!< Max altitude for calculation m */
  double angle;             /*!< Angle of incidence rad */
  double k1;          /*!< Delay Formula Parameter k1 K.Pa^-1*/
  double k2;            /*!< Delay Formula Parameter k2 K.Pa^-1 */
  double k3;         /*!< Delay Formula Parameter k3 K^2.Pa^-1*/
  double g;           /*!< Acceleration of Gravity m s^-2 */
  double rd;        /*!< Dry air specifi gas constant j/kg/K */
  double rv;             /*!< Water vapour specific gas constant j/kg/K */
  double wl;           /*!< Wavelength m */
  int n_interp;     /*number of interpolation points in altitude*/
  double alt_min;      /*minimum altitude for interpolation*/
  double alt_max;      /*maximum altitude for interpolation*/
} delay_param,*Delay_param;


/**
 * \ingroup delay
 * \struct delay_atmospheric_data
 * \brief structure that contains the atmospheric data for the delay function. It holds 3 separate EFIDIRAtmosphere structures of temperature, relative humidity, pressure??
 */

typedef struct delay_atmospheric_data {
  EFIDIRAtmosphere alt; /*!< EFIDIRAtmosphere structure containing Altitude data */
  EFIDIRAtmosphere temp; /*!< EFIDIRAtmosphere structure containing Temperature data */
  EFIDIRAtmosphere rel_hum; /*!< EFIDIRAtmosphere structure containing Relative Humidity data */
} delay_atmospheric_data,*Delay_atmospheric_data;

/**
 * \ingroup delay
 * \struct delay_data
 * \brief structure that contains the calculated delay data. It is held in 2 separate EFIDIRAtmosphere structures of wwpp & wvmr??
 */

typedef struct delay_data {
  EFIDIRAtmosphere wet_m; /*!< EFIDIRAtmosphere structure containing ???? data 	SAME AS ABOVE?? water vapour partial pressure*/
  EFIDIRAtmosphere dry_m; /*!< EFIDIRAtmosphere structure containing ???? data */
  EFIDIRAtmosphere wet_rad; /*!< EFIDIRAtmosphere structure containing ???? data 	SAME AS ABOVE?? water vapour partial pressure*/
  EFIDIRAtmosphere dry_rad; /*!< EFIDIRAtmosphere structure containing ???? data */
} delay_data,*Delay_data;

/**
 * \ingroup delay
 * \fn void define_delay_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_delay_param(char *extra_description);
/**
 * \ingroup delay
 * \fn void define_delay_atmospheric_data(char *extra_description);
 * \brief Default efidir function that defines the atmospheric input data
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_delay_atmospheric_data(char *extra_description);
/**
 * \ingroup delay
 * \fn void define_delay_data(char *extra_description);
 * \brief Default efidir function that defines the output data
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_delay_data(char *extra_description);
/**
 * \ingroup delay
 * \fn Delay_param get_delay_param();
 * \brief Default efidir function that gets parameters
 *  
 * \return A Delay_param that contains the parameters
 *
 */
Delay_param get_delay_param();
/**
 * \ingroup delay
 * \fn Delay_atmospheric_data get_delay_atmospheric_data();
 * \brief Default efidir function that gets the input atmospheric data
 *  
 * \return A Delay_atmospheric_data structure that contains the atmospheric data
 *
 */
Delay_atmospheric_data get_delay_atmospheric_data();
/**
 * \ingroup delay
 * \fn Delay_data get_delay_data();
 * \brief Default efidir function that sets up structure for output data
 *  
 * \return A Delay_data structure that contains the meta-data for the output
 *
 */
Delay_data get_delay_data();
/**
 * \ingroup delay
 * \fn Delay_param new_delay_param();
 * \brief Create a new Delay_param reference
 * \return A new reference on allocated delay_param structure
 *
 */
Delay_param new_delay_param();
/**
 * \ingroup delay
 * \fn Delay_atmospheric_data new_delay_atmospheric_data();
 * \brief Create a new Delay_atmospheric_data reference
 * \return A new reference on allocated delay_atmospheric_data structure
 *
 */
Delay_atmospheric_data new_delay_atmospheric_data();

/**
 * \ingroup delay
 * \fn Delay_data new_delay_data();
 * \brief Create a new Delay_data reference
 * \return A new reference on allocated delay_data structure
 *
 */
Delay_data new_delay_data();

/**
 * \ingroup delay
 * \fn void free_delay_param(Delay_param p);
 * \brief Free an allocated Delay_param reference
 * \param p A reference on a delay_param structure
 *
 */
void free_delay_param(Delay_param p);
/**
 * \ingroup delay
 * \fn void free_delay_atmospheric_data(Delay_atmospheric_data p);
 * \brief Free an allocated Delay_atmospheric_data reference
 * \param p A reference on a delay_atmospheric_data structure
 *
 */
void free_delay_atmospheric_data(Delay_atmospheric_data p);
/**
 * \ingroup delay
 * \fn void free_delay_data(Delay_data p);
 * \brief Free an allocated Delay_data reference
 * \param p A reference on a delay_data structure
 *
 */
void free_delay_data(Delay_data p);

/**
 * \ingroup delay
 * \fn void delay(Delay_param param, Delay_atmospheric_data input_data, Delay_data  output_data);
 * \brief The delay operator calculates the delay due to atmospheric conditions.
 *
 * \param param is the structure containing parameters needed to calculate the delay 
 * \param input_data is a structure containing the atmospheric data required to calculate the delay. It holds 3 separate EFIDIRAtmosphere structures: temperature, pressure and relative humidity??
 * \param output_data is a structure containing the resulting calculated delay. It holds 2 EFIDIRAtmosphere structures : wvpp, wvmr???
 *
 */
void delay(Delay_param param, Delay_atmospheric_data input_data, Delay_data  output_data);

/**
 * \ingroup delay
 * \fn double delay_calc(Delay_param param, double *pres_p, double *alt_p, double *temp_p, double *rh_p);
 * \brief Calculate the delay at a single point (longitude and latitude).
 *
 *are arrays containing pressure, altitude, temperature and relative humidity
 *
 * \param param is a structure containing the required constants for the delay calculation
 * \param pres_p array ...
 * \param alt_p array ...
 * \param temp_p array ...
 * \param rh_p array ...
 *
 * \return the value of the delay 
 *
 */
double delay_calc(Delay_param param, double *pres_p, double *alt_p, double *temp_p, double *rh_p);

/**
 * \ingroup delay
 * \fn double riemann_mid(int limit_a,int limit_b,double step, double *val);
 * \brief Calculate the integral using the Riemann mid-point method.
 *
 * \param limit_a is the lower limit
 * \param limit_b is ...
 * \param step is ...
 * \param val is ...
 *
 * \return The integrated function 
 *
 */
double riemann_mid(int limit_a, int limit_b, double step, double *val);

#endif //__COM_DELAY_H__
