/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup common_operators
 * \defgroup coherence_phase Calculate the coherence and the phase between two SLC SAR images
 */

/**
 * \ingroup coherence_phase
 * \file com_coherence_phase_rescaled_filtered.h
 * \author Diana Rosu
 *
 * \brief operator of coherence_phase
 *
 * file includes structures, prototypes of function of the operator coherence_phase
 */

#include "com_coherence_phase.h"

#ifndef COM_COHERENCE_PHASE_RESCALED_FILTERED_H
#define COM_COHERENCE_PHASE_RESCALED_FILTERED_H



/**
 * \typedef typedef coherence_phase_param
 * \struct coherence_phase_param
 *
 * \brief structure that contains the parameters of the operator s_coherence_phase_rescaled_fitered_param
 *
 */
typedef struct s_coherence_phase_rescaled_fitered_param{
	int choose_operation; /* Specifies if the current operation will be a rescale, a filtering or both of them */
	int rescale_window_nof_lines; /* Number of lines for the rescale window */
	int rescale_window_nof_columns; /* Number of columns for the rescale window */
	int window_filter_size; /* Number of lines/columns of filter window, there is a rectangular filter window */
	int choose_when_extract_fringes; /* Specifies if the fringes extraction is performed before or after the filtering */
}coherence_phase_rescaled_fitered_param, *Coherence_phase_rescaled_fitered_param;

/**
 * \ingroup coherence_phase
 * \fn Coherence_phase_rescaled_fitered_param new_coherence_phase_rescaled_fitered_param();
 * \brief Function that allocates memory for a new structure s_coherence_phase_rescaled_fitered_param
*/
Coherence_phase_rescaled_fitered_param new_coherence_phase_rescaled_fitered_param();

/**
 * \ingroup coherence_phase
 * \fn void free_coherence_phase_rescaled_fitered_param(Coherence_phase_rescaled_fitered_param cprf);
 * \brief Function that deallocates the memory occupied by a structure s_coherence_phase_rescaled_fitered_param
*/
void free_coherence_phase_rescaled_fitered_param(Coherence_phase_rescaled_fitered_param cprf);

/**
 * \ingroup coherence_phase
 * \fn void define_coherence_phase_rescaled_fitered_param(char *extra_description);
 * \brief Function used in order to define a new parameter Coherence_phase_rescaled_fitered_param
 *
 * \param extra_description Extra description for multi definition of
 * the same parameter (NULL is accepted)
*/
void define_coherence_phase_rescaled_fitered_param(char *extra_description);

/**
 * \ingroup coherence_phase
 * \fn Coherence_phase_rescaled_fitered_param get_coherence_phase_rescaled_fitered_param();
 * \brief Function used in order to retrieve the values of the parameter provided by the user
 * \return A Coherence_phase_rescaled_fitered_param that contains the parameters.
 */
Coherence_phase_rescaled_fitered_param get_coherence_phase_rescaled_fitered_param();

/**
 * \ingroup coherence_phase
 * \fn void gen_interferogram_amplitude_complex_rescaled_imgs(Coherence_phase_rescaled_fitered_param coherence_phase_rescaled_filtered_p, EFIDIRImage master_slc, EFIDIRImage slave_slc, EFIDIRImage phase_model, EFIDIRImage amplitude_complex, EFIDIRImage interferogram_complex);
 * \brief Function which generates a complex interfergram image and a complex amplitude image, both of them rescaled with a user defined dimension window. In fact, the complex amplitude image consists both master and slave amplitudes in one image, this is the real part of the image represents the master amplitude image while the imaginary part represents the slave image. It can consider or not the model for the orbital and topographical fringes.
 * The master and slave images should be complex images (cx4, cx8 or cx16) while the model should be float.
 *
 * \param coherence_phase_rescaled_filtered_p contains the values of the parameters for rescaling process.
 * \param master_slc is the master SAR image used to compute the interferogram and amplitude images.
 * \param slave_slc is the slave SAR image used to compute the interferogram and amplitude images.
 * \param phase_model is the image containing the orbital and topographic fringes that should be extracted.
 * \param amplitude_complex is the output image that will contain both master and slave amplitude images. The real part will represent the master amplitude while the imaginary part will represent the slave amplitude.
 * \param interferogram_complex is the output interferogram which may or may not contain the orbital and topographical fringes.
 */
void gen_interferogram_amplitude_complex_rescaled_imgs(Coherence_phase_rescaled_fitered_param coherence_phase_rescaled_filtered_p, EFIDIRImage master_slc, EFIDIRImage slave_slc, EFIDIRImage phase_model, EFIDIRImage amplitude_complex, EFIDIRImage interferogram_complex);

/**
 * \ingroup coherence_phase
 * \fn void gen_coherence_phase_rescaled_filtered_imgs(Coherence_phase_rescaled_fitered_param coherence_phase_rescaled_filtered_p, EFIDIRImage amplitude_complex, EFIDIRImage interferogram_complex, EFIDIRImage phase_model, int x_start, int y_start, EFIDIRImage coherence_image, EFIDIRImage phase_image);
 * \brief Function which generates the coherence and phase images in a rescaled and/or filtered manner, depending on the input amplitude and interferogram images. The phase image may or may not have the orbital and topographical fringes extracted, depending on the existance of phase model either in this processing step or in the previous processing step.
 *
 * \param coherence_phase_rescaled_filtered_p contains the values of the parameters for filtering process.
 * \param amplitude_complex is the image containing both master and slave amplitude images. The real part will represent the master amplitude while the imaginary part will represent the slave amplitude. It may be obtained using gen_interferogram_amplitude_complex_rescaled_imgs function.
 * \param interferogram_complex is the complex interferogram obtained using gen_interferogram_amplitude_complex_rescaled_imgs function.
 * \param phase_model is the image containing the orbital and topographic fringes that should be extracted.
 * \param x_start is the line index in amplitude and interferogram images where it starts the coverage by phase model.
 * \param y_start is the column index in amplitude and interferogram images where it starts the coverage by phase model.
 * \param coherence_image is the image of coherence which may or may not consider the phase model
 * \param phase_image is the image of phase which may or may not consider the phase model.
 */
void gen_coherence_phase_rescaled_filtered_imgs(Coherence_phase_rescaled_fitered_param coherence_phase_rescaled_filtered_p, EFIDIRImage amplitude_complex, EFIDIRImage interferogram_complex, EFIDIRImage phase_model, int x_start, int y_start, EFIDIRImage coherence_image, EFIDIRImage phase_image);

#endif /* COM_COHERENCE_PHASE_RESCALED_FILTERED_H */
