/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

/**
 * \ingroup common_operators
 * \defgroup coherence_phase Calculate the coherence and the phase between two SLC SAR images
 */

/**
 * \ingroup coherence_phase
 * \file com_coherence_phase.h
 * \author Renaud Fallourd
 *
 * \brief operator of coherence_phase
 *
 * file includs structures, prototypes of function of the operator coherence_phase
 */

#ifndef COHERENCE_PHASE_H
#define COHERENCE_PHASE_H

#include <stdlib.h>
#include<stdio.h>
#include<math.h>
#include "efidir_image.h"
#include "efidir_param.h"
#include "efidir_assert.h"
#include "efidir_math.h"


/**
 * \typedef typedef coherence_phase_param
 * \struct coherence_phase_param
 *
 * \brief structure that contains the parameters of the operator coherence_phase
 *
 */
typedef struct struct_coherence_phase_param{
  int window_nb_l; /*!< number of rows (lines) of the coherence/phase window */
  int window_nb_c; /*!< number of columns of the coherence/phase window */
}coherence_phase_param, *Coherence_phase_param;

Coherence_phase_param new_coherence_phase_param();
void free_coherence_phase_param(Coherence_phase_param p);

/**
 * \ingroup coherence_phase
 * \fn void define_coherence_phase_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of 
 * the same parameter (NULL is accepted)
*/
void define_coherence_phase_param(char *extra_description);
/**
 * \ingroup coherence_phase
 * \fn Coherence_phase_param get_coherence_phase_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Coherence_phase_param that contains the parameters.
 */
Coherence_phase_param get_coherence_phase_param();

/**
 * \ingroup coherence_phase
 * \fn void coherence_phase(Coherence_phase_param p, EFIDIRImage inputImage_1, EFIDIRImage inputImage_2, EFIDIRImage inputImage_3, EFIDIRImage outputCoherence, EFIDIRImage outputPhase, EFIDIRImage outputAverage_1, EFIDIRImage outputAverage_2, int amp_choice);
 * \brief Calculate the coherence and the phase between two SAR images
 * The coherence_phase operator calaculate coherence and phase between two SAR images. Obviously two images must be SLC type. This operator supports cx4, cx8 et cx16 image type.
 *
 * \param p is the parameters structure of operator.
 * \param inputImage_1 Master image that will be used for calculate coherence and phase
 * \param inputImage_2 Slave image that will be used for calculate coherence and phase
 * \param inputImage_3 Phase maodel input image (can be NULL)
 * \param outputCoherence is the coherence image output
 * \param outputPhase is the phase image output
 * \param outputAverage_1 is the master average image output
 * \param outputAverage_2 is the slave average image output
 * \param amp_choice is the amplitude calculation index
 *
 */
void coherence_phase(Coherence_phase_param p, EFIDIRImage inputImage_1, EFIDIRImage inputImage_2, EFIDIRImage inputImage_3, EFIDIRImage outputCoherence, EFIDIRImage outputPhase, EFIDIRImage outputAverage_1, EFIDIRImage outputAverage_2, int amp_choice);

#endif /* COHERENCE_PHASE_H */
