/* 
 *    Copyright (c) 2008. The EFIDIR team. All right reserved. 
 * 
 *    This file is part of EFIDIR tools. 
 * 
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify 
 *    it under the terms of the GNU General Public License as published by 
 *    the Free Software Foundation, either version 3 of the License, or 
 *    (at your option) any later version. 
 * 
 *    EFIDIR tool(s) is distributed in the hope that it will be useful, 
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of 
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 *    GNU General Public License for more details. 
 * 
 *    You should have received a copy of the GNU General Public licence 
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>. 
 */ 

#ifndef COM_CENTRE_H
#define COM_CENTRE_H

#include <stdlib.h>
#include "efidir_param.h"
#include "efidir_image.h"
#include <math.h> //??
#include "com_statistics.h"
#include "com_gain_offset.h"

/**
 * \ingroup common_operators
 * \defgroup centre Centre an image
 *
 */

/**
 * \file com_centre.h
 * \author Felicity Lodge
 *   
 * \brief Centre an EFIDIR image.
 * 
  \verbatim
 	Parameters:
 		--inputImage string	: Input image file name 
 		--outputImage string	: Output image file name 
 		[--desc string]		: Output imagefile description ??
  \endverbatim
 *
 * \return Centred image.
 *
 */

/**
 * \ingroup centre
 * \struct centre_param
 * \brief structure that contains the parameters for the centre functions
 *
 */

typedef struct centre_param
{
EFIDIRStat centre_stats;
Statistics_param centre_param_stat;
Gain_offset_param centre_gain_offset;
}centre_param,*Centre_param;

/**
 * \ingroup centre
 * \fn void define_centre_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *  
 * \param extra_description for multi-definition of the same parameter (NULL is accepted)
 *
 */
void define_centre_param(char *extra_description);

/**
 * \ingroup centre
 * \fn Centre_param get_centre_param();
 * \brief Default efidir function that gets parameters
 *  
 * \return A Centre_param that contains the parameters
 *
 */
Centre_param get_centre_param();

/**
 * \ingroup centre
 * \fn Centre_param new_centre_param();
 * \brief Create a new Centre_param reference
 * \return A new reference on allocated centre_param structure
 *
 */
Centre_param new_centre_param();

/**
 * \ingroup centre
 * \fn void free_centre_param(Centre_param p);
 * \brief Free an allocated Centre_param reference
 * \param p A reference on a centre_param structure
 *
 */
void free_centre_param(Centre_param p);

/**
 * \ingroup centre
 * \fn void centre(Centre_param param, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief Centre an image through a given angle
 * 
 * The centre operator rotates the image anticlockwise about the origin then translates the enlarged box containing the rotated image back onto positive axes. In practice, inverse mapping is used to reduce image degradation.
 *
 * \param param is the structure of center parameters
 * \param inputImage is the image to be centred
 * \param outputImage is the resulting centred image
 *
 */
void centre(Centre_param param, EFIDIRImage inputImage, EFIDIRImage outputImage);

#endif /* COM_CENTRE_H */


