/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup common_operators
 * \defgroup cal_slc SLC image operators
 */
/**
 * \ingroup cal_slc
 * \file com_cal_slc.h
 * \author Yajing Yan
 * 
 * \brief operator of calculating the parameters like the magnitude, the phase, the intensity of a SLC image
 * 
 * file includes (stucture, prototype of function) of operator cal_slc
 * 
 * 
 */

#ifndef COM_CAL_SLC_H
#define COM_CAL_SLC_H

#include <stdlib.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"


/**
 * \ingroup cal_slc
 * \enum complex_conversion
 *
 * \brief definition of complex conversion
 */
typedef enum { AMPLITUDE = 1,  /**< (1) complex to amplitude*/
		INTENSITY = 2, /**< (2) complex to intensity*/
		PHASE = 3,     /**< (3) complex to phase*/
		REAL = 4,      /**< (4) real part*/
		IMAGINARY = 5  /**< (5) imagynary part*/
} complex_conversion;

/**
 * \ingroup cal_slc
 * \struct Cal_slc_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct Cal_slc_param{
  int processing; /*!< the parameter to calculate */
}cal_slc_param, *Cal_slc_param;

/**
 * \ingroup cal_slc
 * \fn new_cal_slc_param()
 *
 * \return A new reference on allocated cal_slc_param structure
 *
 * \brief Create a new Cal_slc_param reference 
 *
 */
Cal_slc_param new_cal_slc_param();
/**
 * \ingroup cal_slc
 * \fn free_cal_slc_param(Cal_slc_param p)
 *
 * \param p A reference on a cal_slc_param structure
 *
 * \brief Free an allocated Cal_slc_param reference 
 *
 */
void free_cal_slc_param(Cal_slc_param p);


/**
 * \ingroup cal_slc
 * \fn void define_cal_slc_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_cal_slc_param(char *extra_description);

/**
 * \ingroup cal_slc
 * \fn Cal_slc_param get_cal_slc_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Cal_slc_param that contains the parameters.
 */
Cal_slc_param get_cal_slc_param();

/**
 * \ingroup cal_slc
 * \fn void cal_slc(Cal_slc_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);
 * \brief function of calculating some parameters of an image SLC
 * Function calculating some parameters such as the magnitude, the phase, the intensity of an image SLC.
 * The lecture of image is realised line par line.
 *
 * \param p is the structure parameter for the operator
 * \param inputImage is the complex float SLC
 * \param outputImage is the image of result
 *
 */
void cal_slc(Cal_slc_param p, EFIDIRImage inputImage, EFIDIRImage outputImage);


#endif
