/*
 *    Copyright (c) 2008. The EFIDIR team. All right reserved.
 *
 *    This file is part of EFIDIR tools.
 *
 *    EFIDIR tool(s) is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    EFIDIR tool(s) is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public licence
 *    along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \ingroup common_operators
 * \defgroup cal_image Image base operators (sum, subtraction, multiplication, division ...)
 */
/**
 * \ingroup cal_image
 * \file com_cal_image.h
 * \author Yajing Yan
 * 
 * \brief operator of computing the sum, the subtraction, the multiplication, the division ... of 2 images
 * 
 * file includes (stucture, prototype of function) of operator cal_image
 * 
 * 
 */

#ifndef COM_CAL_IMAGE_H
#define COM_CAL_IMAGE_H

#include <stdlib.h>
#include <math.h>
#include "efidir_image.h"
#include "efidir_param.h"

/**
 * \ingroup cal_image
 * \enum computation_index
 *
 * \brief definition of computation index
 */
typedef enum { SUM = 1,  /**< (1) sum*/
		SUBTRACTION = 2, /**< (2) subtraction*/
		MULTIPLICATION = 3,     /**< (3) multiplication*/
		DIVISION = 4,      /**< (4) division*/

} computation_index;

/**
 * \ingroup cal_image
 * \struct Cal_imagege_param
 *
 * \brief structure that contains the parameters of the operator
 *
 */
typedef struct Cal_image_param{
  int processing; /*!< processing index*/
  int dx; /*!< offset of columns */
  int dy; /*!< offset of lines */
}cal_image_param, *Cal_image_param;

/**
 * \ingroup cal_image
 * \fn new_cal_image_param()
 *
 * \return A new reference on allocated cal_image_param structure
 *
 * \brief Create a new Cal_image_param reference 
 *
 */
Cal_image_param new_cal_image_param();
/**
 * \ingroup cal_image
 * \fn free_cal_image_param(Cal_image_param p)
 *
 * \param p A reference on a cal_image_param structure
 *
 * \brief Free an allocated Cal_image_param reference 
 *
 */
void free_cal_image_param(Cal_image_param p);


/**
 * \ingroup cal_image
 * \fn void define_cal_image_param(char *extra_description);
 * \brief Default efidir function that defines parameters
 *
 * \param extra_description Extra description for multi definition of the same parameter (NULL is accepted)
 */
void define_cal_image_param(char *extra_description);

/**
 * \ingroup cal_image
 * \fn Cal_image_param get_cal_image_param();
 * \brief Default efidir function that gets parameters
 *
 * \return A Cal_imagege_param that contains the parameters.
 */
Cal_image_param get_cal_image_param();

/**
 * \ingroup cal_image
 * \fn void cal_image(Cal_image_param p, EFIDIRImage inputImage1, EFIDIRImage inputMask1, EFIDIRImage inputImage2, EFIDIRImage inputMask2,EFIDIRImage outputImage, EFIDIRImage outputMask);
 * \brief function of calculating the sum, the subtraction, the multiplication, the division ... of 2 images
 * Function calculating the sum, the subtraction, the multiplication, the division ... of 2 images
 *
 * \param p is the structure parameter for the operator
 * \param inputImage1 is the first image
 * \param inputMask1 is the mask of the first image
 * \param inputImage2 is the second image
 * \param inputMask2 is the mask of the second image
 * \param outputImage is the image of result
 * \param outputMask is the mask of image to output
 *
 */
void cal_image(Cal_image_param p, EFIDIRImage inputImage1, EFIDIRImage inputMask1, EFIDIRImage inputImage2, EFIDIRImage inputMask2,EFIDIRImage outputImage, EFIDIRImage outputMask);


#endif
