#!/usr/bin/env python

# Copyright (c) 2013. The EFIDIR team. All right reserved.
#
# This file is part of EFIDIR tools.
#
# EFIDIR tool(s) is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# EFIDIR tool(s) is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public licence
# along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
#
# author: Matthieu Volat / ISTerre
#

import os, sys, datetime

from efidir.config import *
from efidir.goodies import *
from efidir.image import *
from efidir.param import *
from efidir.sws import *
from efidir.utils import tsx

## 
# @page sar_amplitude_2d_displacement_ground sar_amplitude_2d_displacement_ground
# @author Matthieu Volat (ISTerre)
#
# Process a serie of SAR images to ouput the 2D ground displacement maps
# using amplitude correlation.
# 
# @verbatim
#    Parameters:
#        --input-images-file     : Input file with list of images (one per line)
#        --input-imagepairs-file : Input file with list of comma-separated images pairs (one per line)
#        --input-dem-image       : DEM image
#        --input-aerial-image    : Aerial photography
#        [--correlation-master-window-size] : Size of the correlation window in the master image
#        [--correlation-search-window-size] : Size of the correlation search window
#        [--output-dir]          : Where to put output files
#        [--njobs]               : Start up to n concurrent jobs
#        [--verbose]             : Display as much info as possible
#@endverbatim

# Parameters
define_param("input-images-file",
             STRING,
             "Input file with list of images (one per line)",
             None)
define_param("input-imagepairs-file",
             STRING,
             "Input file with list of comma-separated images pairs (one per line)",
             None)
define_param("input-dem-image",
             STRING,
             "DEM image",
             None)
define_param("input-aerial-image",
             STRING,
             "Aerial photography",
             None)
define_optional_param("correlation-master-window-size",
                      INT,
                      "Size of the correlation window in the master image",
                      None)
define_optional_param("correlation-search-window-size",
                      INT,
                      "Size of the correlation search window",
                      None)
"""
define_optional_param("ground-control-area",
                      STRING,
                      "coordinates of the ground control area",
                      None)
define_optional_param("projection-vector",
                      STRING,
                      "",
                      None)
"""
define_optional_param("output-dir",
                      STRING,
                      "Where to put output files",
                      None)
define_optional_param("njobs",
                      INT,
                      "Start up to n concurrent jobs",
                      None)
define_optional_param("verbose",
                      INT,
                      "Display as much info as possible",
                      None)

init_param(sys.argv)

input_images_file = get_string_value_param("input-images-file")
input_imagepairs_file = get_string_value_param("input-imagepairs-file")
input_dem_image = get_string_value_param("input-dem-image")
input_aerial_image = get_string_value_param("input-aerial-image")
corr_mwinsize = get_int_value_param("correlation-master-window-size") or 65
corr_swinsize = get_int_value_param("correlation-search-window-size") or 105
outputdir = get_string_value_param("output-dir") or "output"
verbose = get_int_value_param("verbose")

set_njobs(get_int_value_param("njobs"))

# Target list to be built
targets = []

# Read input images list
input_images = [l.rstrip() for l in open(input_images_file)]

# Read input pairs list
input_imagepairs = [tuple(l.rstrip().split(",")) for l in open(input_imagepairs_file)]

# From the input image list, create a date list to refer to images, with
# attributes like xml and cos files, sat direction (asc/des) and such
# NOTE: For now, this is very TSX specific
dates = []
xml = {}; cos = {}; images = {}
for i in input_images:
    infos = tsx.readinfos(i)
    infos2 = read_image_open(i)
    d = parse_iso8601(infos2["acquisition time"]).strftime("%Y%m%d")
    dates.append(d)
    images[d] = i
    xml[d] = infos["xml"]
    cos[d] = infos["cos"]
# Oh, btw, create a datepairs list, more handy than input_imagepairs
datepairs = [(dates[input_images.index(t[0])], dates[input_images.index(t[1])]) for t in input_imagepairs]


# 1.1 Create azimuth and range images from sat input and DEM
for d in dates:
    outputs = [
        os.path.join(outputdir, "ground", "gen_azimuth_range_imgs_user_precision_"+d+".save"),
        os.path.join(outputdir, "ground", d+"_azimuth"),
        os.path.join(outputdir, "ground", d+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", d+"_range"),
        os.path.join(outputdir, "ground", d+"_range.hdr") ]
    inputs = [
        xml[d], input_dem_image, input_dem_image+".hdr" ]
    commands = [
        EFIDIR_BINDIR+"/gen_azimuth_range_imgs_user_precision"
            +"  --save "+outputs[0]
            +"  --file_type 0"
            +"  --dist_between_points 1"
            +"  --pol_deg 7"
            +"  --ignored_val 0"
            +"  --input_metadata_file "+inputs[0]
            +"  --dem_name "+inputs[1]
            +"  --azimuth_img "+outputs[1]
            +"  --range_img "+outputs[3] ]
    targets.append(target_create(commands, inputs, outputs))

# 1.2 Crop the SLC images to the zone covered by the DEM
for d in dates:
    outputs = [
        os.path.join(outputdir, "sar", "crop_SAR_image4specific_DEM_"+d+".save"),
        os.path.join(outputdir, "sar", d+"_slc"),
        os.path.join(outputdir, "sar", d+"_slc.hdr") ]
    inputs = [
        images[d],
        os.path.join(outputdir, "ground", d+"_azimuth"),
        os.path.join(outputdir, "ground", d+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", d+"_range"),
        os.path.join(outputdir, "ground", d+"_range.hdr") ]
    commands = [
        EFIDIR_BINDIR+"/crop_SAR_image4specific_DEM"
            +"  --save "+outputs[0]
            +"  --ignored_val 0"
            +"  --input_SAR "+inputs[0]
            +"  --input_azimuth "+inputs[1]
            +"  --input_range "+inputs[3]
            +"  --additional_nof_lines 0"
            +"  --additional_nof_columns 0"
            +"  --output_cropped_SAR "+outputs[1] ]
    targets.append(target_create(commands, inputs, outputs))

# 1.3 SLC -> amplitude conversion
for d in dates:
    outputs = [
        os.path.join(outputdir, "sar", "cal_slc_"+d+".save"),
        os.path.join(outputdir, "sar", d+"_amp0"),
        os.path.join(outputdir, "sar", d+"_amp0.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", d+"_slc"),
        os.path.join(outputdir, "sar", d+"_slc.hdr") ]
    commands = [
        EFIDIR_BINDIR+"/cal_slc"
            +"  --save "+outputs[0]
            +"  --input "+inputs[0]
            +"  --output "+outputs[1]
            +"  --processing 1"
            +"  --description 0" ]
    targets.append(target_create(commands, inputs, outputs))

# 1.4.a Determine common size between image series
outputs = [ os.path.join(outputdir, "sar", "common_pad_size.txt") ]
inputs = [ os.path.join(outputdir, "sar", d+"_amp0.hdr") for d in dates ]
commands = [
    "awk 'BEGIN { FS=\" *= *\"; maxlines=0; maxsamples=0; } "
        +" { if ($$1~/lines/ && $$2>maxlines) { maxlines=$$2 }; "
        +"   if ($$1~/samples/ && $$2>maxsamples) { maxsamples=$$2 } } "
        +"END { print \"--nrows \"maxlines\"  --ncols \"maxsamples } "
        +" ' "+" ".join(inputs)+" > "+outputs[0] ]
targets.append(target_create(commands, inputs, outputs))
# 1.4.b Then resize
for d in dates:
    outputs = [
        os.path.join(outputdir, "sar", "zeropad_amp0_"+d+".save"),
        os.path.join(outputdir, "sar", d+"_amp"),
        os.path.join(outputdir, "sar", d+"_amp.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", d+"_amp0"),
        os.path.join(outputdir, "sar", d+"_amp0.hdr"),
        os.path.join(outputdir, "sar", "common_pad_size.txt") ]
    commands = [
        EFIDIR_BINDIR+"/zeropad"
            +"  --save "+outputs[0]
            +"  --input "+inputs[0]
            +"  --output "+outputs[1]
            +"  --desc \"0\""
            +"  `cat "+inputs[2]+"`" ]
    targets.append(target_create(commands, inputs, outputs))

# 2.1 Project SAR image in orthogonal geometry
for d in dates:
    outputs = [
        os.path.join(outputdir, "ground", "interpol_azim_range_radar2ground_"+d+".save"),
        os.path.join(outputdir, "ground", d+"_ortho"),
        os.path.join(outputdir, "ground", d+"_ortho.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", d+"_amp"),
        os.path.join(outputdir, "sar", d+"_amp.hdr"),
        os.path.join(outputdir, "ground", d+"_azimuth"),
        os.path.join(outputdir, "ground", d+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", d+"_range"),
        os.path.join(outputdir, "ground", d+"_range.hdr") ]
    commands = [
        EFIDIR_BINDIR+"/interpol_azim_range_radar2ground"
            +"  --save "+outputs[0]
            +"  --amplitude_img_name "+inputs[0]
            +"  --azimuth_img_name "+inputs[2]
            +"  --range_img_name "+inputs[4]
            +"  --interpolation_type 1"
            +"  --input_ignored_value 0"
            +"  --output_ignored_value 0"
            +"  --amplitude_2_ground_img_name "+outputs[1] ]
    targets.append(target_create(commands, inputs, outputs))

# 2.2 Project latitude, longitude and elevation in SAR geometry
for d in dates:
    outputs = [
        os.path.join(outputdir, "sar","gen_LUTs_on_master_grid_"+d+".save"),
        os.path.join(outputdir, "sar", d+"_lat"),
        os.path.join(outputdir, "sar", d+"_lat.hdr"),
        os.path.join(outputdir, "sar", d+"_lon"),
        os.path.join(outputdir, "sar", d+"_lon.hdr"),
        os.path.join(outputdir, "sar", d+"_up"),
        os.path.join(outputdir, "sar", d+"_up.hdr"),
        os.path.join(outputdir, "sar", d+"_foldovermask") ]
    inputs = [
        input_dem_image,
        input_dem_image+".hdr",
        os.path.join(outputdir, "sar", d+"_amp"),
        os.path.join(outputdir, "sar", d+"_amp.hdr"),
        os.path.join(outputdir, "ground", d+"_azimuth"),
        os.path.join(outputdir, "ground", d+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", d+"_range"),
        os.path.join(outputdir, "ground", d+"_range.hdr") ]
    commands = [
        EFIDIR_BINDIR+"/gen_LUTs_on_master_grid"
            +"  --save "+outputs[0]
            +"  --ignored_val 0"
            +"  --dem_name "+inputs[0]
            +"  --input_SAR_master "+inputs[2]
            +"  --input_range_master "+inputs[6]
            +"  --input_azimuth_master "+inputs[4]
            +"  --output_lat "+outputs[1]
            +"  --output_long "+outputs[3]
            +"  --output_alt "+outputs[5]
            +"  --output_mask "+outputs[7] ]
    targets.append(target_create(commands, inputs, outputs))

# 2.3 Projection of the geocoded image in SAR geometry
outputs = [
    os.path.join(outputdir, "sar", "interpol_lut_ground2radar_multiband_"+d+".save"),
    os.path.join(outputdir, "sar", d+"_aerial"),
    os.path.join(outputdir, "sar", d+"_aerial.hdr") ]
inputs = [
    input_aerial_image,
    input_aerial_image+".hdr",
    os.path.join(outputdir, "sar", d+"_lat"),
    os.path.join(outputdir, "sar", d+"_lat.hdr"),
    os.path.join(outputdir, "sar", d+"_lon"),
    os.path.join(outputdir, "sar", d+"_lon.hdr") ]
commands = [
    EFIDIR_BINDIR+"/interpol_lut_ground2radar_multiband"
        +"  --save "+outputs[0]
        +"  --input_image "+inputs[0]
        +"  --nof_bands 1"
        +"  --input_lat "+inputs[2]
        +"  --input_long  "+inputs[4]
        +"  --interpolation_type 1"
        +"  --input_ignored_values 0"
        +"  --output_ignored_values 0"
        +"  --output_image "+outputs[1] ]
targets.append(target_create(commands, inputs, outputs))

# 3.1.a Compute range/azimuth offset between master/slaves
for md, sd in datepairs:
    outputs = [
        os.path.join(outputdir, "sar", "gen_delta_azimuth_range_master_slave_"+md+"-"+sd+".save"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_azimuth0"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_azimuth0.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_range0"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_range0.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", md+"_amp"),
        os.path.join(outputdir, "sar", md+"_amp.hdr"),
        os.path.join(outputdir, "ground", md+"_azimuth"),
        os.path.join(outputdir, "ground", md+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", md+"_range"),
        os.path.join(outputdir, "ground", md+"_range.hdr"),
        os.path.join(outputdir, "ground", sd+"_azimuth"),
        os.path.join(outputdir, "ground", sd+"_azimuth.hdr"),
        os.path.join(outputdir, "ground", sd+"_range"),
        os.path.join(outputdir, "ground", sd+"_range.hdr") ]
    commands = [
        EFIDIR_BINDIR+"/gen_delta_azimuth_range_master_slave"
            +"  --save "+outputs[0]
            +"  --ignored_val 0"
            +"  --input_SAR_master "+inputs[0]
            +"  --input_range_master "+inputs[4]
            +"  --input_range_slave "+inputs[8]
            +"  --input_azimuth_master "+inputs[2]
            +"  --input_azimuth_slave "+inputs[6]
            +"  --output_delta_azimuth "+outputs[1]
            +"  --output_delta_range "+outputs[3] ]
    targets.append(target_create(commands, inputs, outputs))
# 3.1.b Some delta images may be cropped, pad them (as amplitude before)
for md, sd in datepairs:
    for direction in ["azimuth", "range"]:
        outputs = [
            os.path.join(outputdir, "sar", "zeropad_delta_"+direction+"_"+md+"-"+sd+".save"),
            os.path.join(outputdir, "sar", md+"-"+sd+"_delta_"+direction),
            os.path.join(outputdir, "sar", md+"-"+sd+"_delta_"+direction+".hdr") ]
        inputs = [
            os.path.join(outputdir, "sar", md+"-"+sd+"_delta_"+direction+"0"),
            os.path.join(outputdir, "sar", md+"-"+sd+"_delta_"+direction+"0.hdr"),
            os.path.join(outputdir, "sar", "common_pad_size.txt") ]
        commands = [
            EFIDIR_BINDIR+"/zeropad"
                +"  --save "+outputs[0]
                +"  --input "+inputs[0]
                +"  --output "+outputs[1]
                +"  --desc \"0\""
                +"  `cat "+inputs[2]+"`" ]
        targets.append(target_create(commands, inputs, outputs))

# 3.2 Normalized crosscorrelation between masters/slaves
for md, sd in datepairs:
    outputs = [
        os.path.join(outputdir, "sar", "dist_corr_"+md+"_"+sd+".save"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_vec"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_vec.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_peak"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_peak.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_fwhm"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_fwhm.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", md+"_amp"),
        os.path.join(outputdir, "sar", md+"_amp.hdr"),
        os.path.join(outputdir, "sar", sd+"_amp"),
        os.path.join(outputdir, "sar", sd+"_amp.hdr") ]
    outputs2 = [os.path.basename(f) for f in outputs]
    inputs2 = [os.path.basename(f) for f in inputs]
    commands = [
        "#cd "+outputdir+"/sar; "+EFIDIR_PREFIX+"/bin/dist_corr_splitter"
            +" --save "+outputs2[0]
            +" --masterFileName "+inputs2[0]
            +" --slaveFileName "+inputs2[2]
            +" --first_line 0"
            +" --displacementFileName "+outputs2[1]
            +" --correlationPeakFileName "+outputs2[3]
            +" --correlationFWHMFileName "+outputs2[5]
            +" --window_nb_rows %d" % corr_mwinsize
            +" --window_search_nb_rows %d" % corr_swinsize
            +" --optimization 3"
            +" --sub_pixel 1"
            +" --requiredCore %d" % get_njobs() if get_njobs() >= 2 else 2 ]
    targets.append(target_create(commands, inputs, outputs))

# 3.3.a Compute azimuth and range translation params for each master/slave 
#       pair
for md, sd in datepairs:
    # TODO: split into range/azimuth translation files
    # TODO: for now use only range
    outputs = [ 
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_translation_range.txt"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_translation_azimuth.txt") ]
    inputs = [
        os.path.join(outputdir, "sar", md+"_amp.hdr"),
        os.path.join(outputdir, "sar", sd+"_amp.hdr") ]
    commands = [
        "awk 'BEGIN { FS=\" *= *\"; mxstart=0; mystart=0; xcount=0; sxstart=0; systart=0; } "
            +"{ if ($$1~/x start/) { "
            +"    if (xcount==0) {mxstart=$$2; xcount=1} else {sxstart=$$2} "
            +"  }; "
            +"  if ($$1~/y start/) { "
            +"    if (ycount==0) {mystart=$$2; ycount=1} else {systart=$$2} "
            +"  }; "
            +"} END { "
            +"  rtrans=sxstart-mxstart; atrans=systart-mystart;" 
            +"  print rtrans > \""+outputs[0]+"\";"
            +"  print atrans > \""+outputs[1]+"\";"
            +"} ' "+" ".join(inputs) ]
    targets.append(target_create(commands, inputs, outputs))
# 3.3.b Compute azimuth_offset_res for each pair
for md, sd in datepairs:
    # TODO: for now, just put 0
    outputs = [
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_azimuth_offset_res.txt") ]
    inputs = [ ]
    commands = [ "echo 0 > "+outputs[0] ]
    targets.append(target_create(commands, inputs, outputs))
# 3.3.c Conversion of pixel offsets into 2D displacement
for md, sd in datepairs:
    tdelta = (datetime.datetime.strptime(sd, "%Y%m%d") - datetime.datetime.strptime(md, "%Y%m%d")).days
    outputs = [ 
        os.path.join(outputdir, "sar", "cal_dpi_ori_"+md+"-"+sd+".save"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_dpl"), #displacement (m/day) = sqrt(sqr(dplx)+sqr(dply))
        os.path.join(outputdir, "sar", md+"-"+sd+"_dpl.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_ori"), #orientation
        os.path.join(outputdir, "sar", md+"-"+sd+"_ori.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_dplx"), #range (column) displacement per day in m
        os.path.join(outputdir, "sar", md+"-"+sd+"_dplx.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_dply"), #azimuth (line) displacement per day in m
        os.path.join(outputdir, "sar", md+"-"+sd+"_dply.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_sigmax"), #column fwhm (m/day???)
        os.path.join(outputdir, "sar", md+"-"+sd+"_sigmax.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_sigmay"), #line fwhm (m/day???)
        os.path.join(outputdir, "sar", md+"-"+sd+"_sigmay.hdr") ]
    inputs = [
        os.path.join(outputdir, "sar", md+"-"+sd+"_vec"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_vec.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_range"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_range.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_azimuth"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_delta_azimuth.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_fwhm"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_fwhm.hdr"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_translation_range.txt"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_translation_azimuth.txt"),
        os.path.join(outputdir, "sar", md+"-"+sd+"_param_azimuth_offset_res.txt") ]
    commands = [
        EFIDIR_PREFIX+"/bin/cal_dpl_ori"
            +"  --save "+outputs[0]
            +"  --input_1 "+inputs[0]
            +"  --input_2 "+inputs[2]
            # TODO: --input_3 (need a corrected deltaY)
            +"  --input_5 "+inputs[6]
            +"  --output_1 "+outputs[1]
            +"  --output_2 "+outputs[3]
            +"  --output_3 "+outputs[5]
            +"  --output_4 "+outputs[7]
            +"  --output_5 "+outputs[9]
            +"  --output_6 "+outputs[11]
            +"  --range_translation $$(cat "+inputs[8]+")"
            #+"  --azimuth_translation $$(cat "+inputs[9]+")" # TODO: if corrected deltaY
            +"  --azimuth_offset_res $$(cat "+inputs[10]+")"
            +"  --range_px 1.36410540008545" # TODO: hardcoded is bad
            +"  --azimuth_px 1.98064713660364311" # TODO: hardcoded is bad
            +"  --temporal_distance "+str(tdelta)
            +"  --ignored_val_choice 2"
            +"  --verbose "+str(verbose) ]
    targets.append(target_create(commands, inputs, outputs))

# 4.1 Project 2D displacement measures in ground geometry
for md, sd in datepairs:
    for i in ["dplx", "dply", "peak", "sigmax", "sigmay"]:
        outputs = [
            os.path.join(outputdir, "ground", "interpol_azim_range_radar2ground_"+i+"_"+md+"-"+sd+".save"),
            os.path.join(outputdir, "ground", md+"-"+sd+"_"+i),
            os.path.join(outputdir, "ground", md+"-"+sd+"_"+i+".hdr") ]
        inputs = [
            os.path.join(outputdir, "sar", md+"-"+sd+"_"+i),
            os.path.join(outputdir, "sar", md+"-"+sd+"_"+i+".hdr"),
            os.path.join(outputdir, "ground", md+"_range"),
            os.path.join(outputdir, "ground", md+"_range.hdr"),
            os.path.join(outputdir, "ground", md+"_azimuth"),
            os.path.join(outputdir, "ground", md+"_azimuth.hdr") ]
        commands = [
            EFIDIR_PREFIX+"/bin/interpol_azim_range_radar2ground"
                +"  --save "+outputs[0]
                +"  --amplitude_img_name "+inputs[0]
                +"  --range_img_name "+inputs[2]
                +"  --azimuth_img_name "+inputs[4]
                +"  --interpolation_type 1"
                +"  --input_ignored_value 0"
                +"  --output_ignored_value 0"
                +"  --amplitude_2_ground_img_name "+outputs[1] ]
        targets.append(target_create(commands, inputs, outputs))

# 4.2 Include projection vectors in images
# TODO

wf = workflow_create("sar_amplitude_2d_displacement_ground.mk", targets)
workflow_run(wf)
