#!/usr/bin/env python
# -*- coding: utf-8 -*-

# Copyright (c) 2016. The EFIDIR team. All right reserved.
#
# This file is part of EFIDIR tools.
#
# EFIDIR tool(s) is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# EFIDIR tool(s) is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public licence
# along with EFIDIR tools.  If not, see <http://www.gnu.org/licenses/>.
#
# author: Matthieu Volat / ISTerre
#

## 
# @page compress_s1_safe compress_s1_safe
# @author Matthieu Volat (ISTerre)
#
# Compress a serie of Sentinel-1 SAFE directories images using the deflate
# algorithm.
# 
# @verbatim
#    Parameters:
#        --input-images-file     : Input file with list of images (one per line)
#        [--only-check]          : Only check images, useful with --verbose
#        [--verbose]             : Display as much info as possible
#@endverbatim

import glob, os, os.path as path, sys
from efidir.param import *
from efidir.sws import *

# Parameters
define_param("input-images-file",
             STRING,
             "Input file with list of images (one per line)",
             None)
define_optional_param("only-check",
                      INT,
                      "Only check images, useful with --verbose",
                      None)
define_optional_param("verbose",
                      INT,
                      "Display as much info as possible",
                      None)

init_param(sys.argv)
input_images_file = get_string_value_param("input-images-file")
only_check = get_int_value_param("only-check")
verbose = get_int_value_param("verbose")

# Read input images list
input_images = [l.rstrip() for l in open(input_images_file)]


# Here's the deal: input and output files are the same, so we have to use
# cookie files. But I don't want to leave that in any directory, so to 
# workaround that, I'll run first a first workflow that create cookies
# for images already compressed, then run the compression workflow, then 
# remove any cookie.


# Create cookies
if verbose == 1:
    print "find images that are already compressed"

tasks = []
for i in input_images:
    tiffs = glob.glob(path.join(i, "measurement", "s1a-*.tiff"))
    for t in tiffs:
        outputs = [path.join(path.dirname(t),
                             path.basename(t)+".deflated")]
        inputs = [t]
        commands = ["gdalinfo -nogcp -norat -noct %s|grep COMPRESSION=DEFLATE && touch %s || true" % (inputs[0], outputs[0])]
        tasks.append(target_create(commands, inputs, outputs))
workflow_find_already_compressed = workflow_create("compress_s1_safe_find_already_compressed", tasks)
workflowrun(workflow_find_already_compressed)

if verbose == 1:
    tcount = 0
    dcount = 0
    for i in input_images:
        tiffs = glob.glob(path.join(i, "measurement", "s1a-*.tiff"))
        deflated = glob.glob(path.join(i, "measurement", "s1a-*.tiff.deflated"))
        tcount = tcount + len(tiffs)
        dcount = dcount + len(deflated)
    print "search done,",tcount, "tiffs,",dcount,"compressed image(s) found"
if only_check == 1:
    # Remove cookies
    for i in input_images:
        deflated = glob.glob(path.join(i, "measurement", "s1a-*.tiff.deflated"))
        for f in deflated:
            os.unlink(f)
    exit(0)


# Now compress
if verbose == 1:
    print "start compression"

tasks = []
for i in input_images:
    tiffs = glob.glob(path.join(i, "measurement", "s1a-*.tiff"))
    for t in tiffs:
        outputs = [path.join(path.dirname(t),
                             path.basename(t)+".deflated")]
        inputs = [t]
        commands = ["gdal_translate -q -co COMPRESS=DEFLATE "+inputs[0]+" "+inputs[0]+".tmp",
                    "mv "+inputs[0]+".tmp "+inputs[0],
                    "touch "+outputs[0]]
        tasks.append(target_create(commands, inputs, outputs))
workflow_compress = workflow_create("compress_s1_safe", tasks)
workflowrun(workflow_compress)

if verbose == 1:
    print "images compressed, cleaning up SAFE directories"

# Remove cookies
for i in input_images:
    deflated = glob.glob(path.join(i, "measurement", "s1a-*.tiff.deflated"))
    for f in deflated:
        os.unlink(f)
if verbose == 1:
    print "cleaning done"
