package gnu.trove;

// made "main method runnable" instead of JUnit runnable to avoid Junit dependency
public class MapTest {
    public static void main(String[] args) {
        testTIntObjectHashMap();
        testTHashMap();
    }

    private static void testTIntObjectHashMap() {
        TIntObjectHashMap<String> map = new TIntObjectHashMap<String>();
        // init correctly
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // first put placed at one
        int key = 1;
        assertTrue(map._hashingStrategy.computeHashCode(key) == 1);
        String value = "1";
        String prev = map.put(key, value);
        assertTrue(prev == null);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        int index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        // remove leaves cell "removed"
        String remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // reinsert after remove must reuse the cell
        prev = map.put(key, value);
        assertTrue(prev == null);
        index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // second put double hashes
        int key2 = 8;
        assertTrue(map._hashingStrategy.computeHashCode(key2) == 8); // conflict
        prev = map.put(key2, value);
        assertTrue(prev == null);
        index = map.index(key2);
        assertTrue(index == 4);
        assertTrue(map.size() == 2);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 2);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // remove of the first key leaves gap
        index = map.index(key);
        remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // search for the second key must probe through the gap
        String v = map.get(key2);
        assertTrue(v == value); // traverse through REMOVED

        // second remove leaves two removed cells
        index = map.index(key2);
        assertTrue(index == 4);
        remove = map.remove(key2);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._values)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // clear resets everything
        map.clear();
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // prepare for rehash
        map.put(1, value);
        map.put(2, value);
        map.put(3, value);
        map.put(4, value);
        map.put(5, value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // nothing changes
        map.put(5, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // rehash
        map.put(6, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 17);
        assertTrue(map._free == map.capacity()-6);
        assertTrue(map._size == 6);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));
    }

    private static void testTHashMap() {
        THashMap<Integer,String> map = new THashMap<Integer, String>();
        // init correctly
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity());
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // first put placed at one
        int key = 1;
        assertTrue(map._hashingStrategy.computeHashCode(key) == 1);
        String value = "1";
        String prev = map.put(key, value);
        assertTrue(prev == null);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        int index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        // remove leaves cell "removed"
        String remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // reinsert after remove must reuse the cell
        prev = map.put(key, value);
        assertTrue(prev == null);
        index = map.index(key);
        assertTrue(index == 1);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-1);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // second put double hashes
        int key2 = 8;
        assertTrue(map._hashingStrategy.computeHashCode(key2) == 8); // conflict
        prev = map.put(key2, value);
        assertTrue(prev == null);
        index = map.index(key2);
        assertTrue(index == 4);
        assertTrue(map.size() == 2);
        assertTrue(((Object[])map._values)[index] == value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 2);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));


        // remove of the first key leaves gap
        index = map.index(key);
        remove = map.remove(key);
        assertTrue(remove == value);
        assertTrue(map.size() == 1);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 1);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 1);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // search for the second key must probe through the gap
        String v = map.get(key2);
        assertTrue(v == value); // traverse through REMOVED

        // second remove leaves two removed cells
        index = map.index(key2);
        assertTrue(index == 4);
        remove = map.remove(key2);
        assertTrue(remove == value);
        assertTrue(map.size() == 0);
        assertTrue(map.capacity() == 7);
        assertTrue(((Object[])map._set)[index] == TObjectHash.REMOVED);

        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // clear resets everything
        map.clear();
        // _deadkeys was not reset due to optimisation "do nothing on empty map .clear()"
        assertTrue(map._deadkeys == 2);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-2);
        assertTrue(map._size == 0);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // prepare for rehash
        map.put(1, value);
        map.put(2, value);
        map.put(3, value);
        map.put(4, value);
        map.put(5, value);

        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // nothing changes
        map.put(5, "xxx");
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 7);
        assertTrue(map._free == map.capacity()-5);
        assertTrue(map._size == 5);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));

        // rehash
        map.put(6, value);
        assertTrue(map._deadkeys == 0);
        assertTrue(map._hashingStrategy == map);
        assertTrue(map.capacity() == 17);
        assertTrue(map._free == map.capacity()-6);
        assertTrue(map._size == 6);
        assertTrue(map._loadFactor == THash.DEFAULT_LOAD_FACTOR);
        assertTrue(map._maxSize == (int)(map.capacity() * map._loadFactor));
    }

    private static void assertTrue(boolean b) {
        if (!b) throw new AssertionError();
    }
}