<?php
/**
 * Handle get or set value of an object
 * with caching feature (only cache information of variable, but no variable value)
 *
 * === Variable information ===
 * array
 * {
 *    Variable Name = array
 *        {
 *            NAME = Name,
 *            GETTER = Getter,
 *            SETTER = Setter
 *        }, ...
 * }
 *
 * === Notice ===
 * 1. Return value/parameter should be pointer
 * 2. Variable and its getter/setter must be follow the rule
 * 2.1. First character of variable name must be lower
 * 2.2. Getter must be start with 'get' and continue with First character upper of variable name
 * 2.3. Setter must be start with 'set' and continue with First character upper of variable name
 * 2.4. Getter/Setter's return value/parameter should be pointer
 */
class epObjectUtil {
	private static $_instance = null;

	public static function instance() {
		if (is_null(self::$_instance)) {
			self::$_instance = new epObjectUtil();
		}
		return self::$_instance;
	}

	public static function &_TRUE() {
		$lVariable = true;
		return $lVariable;
	}

	public static function &_FALSE() {
		$lVariable = false;
		return $lVariable;
	}

	public static function &_NULL() {
		$lVariable = null;
		return $lVariable;
	}

	private $_cachedVariables = array();

	protected function setCachedVariables(&$aObject, &$aVariables) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to cache variables for non-object');
		}
		if (is_null($aVariables)) {
			unset($_cachedVariables[get_class($aObject)]);
		} else {
			$_cachedVariables[get_class($aObject)] = $aVariables;
		}
	}

	protected function &getCachedVariables(&$aObject, &$aAutoCreate = false) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to get cached variables of non-object');
		}
		if (isset($_cachedVariables[get_class($aObject)])) {
			$lVariables = $_cachedVariables[$aClass];
		} else {
			$lVariables = null;
		}
		if ($aAutoCreate && is_null($lVariables)) {
			// Don't using caching, or inf-loop
			$lVariables = $this->getVariables($aObject, epObjectUtil::_FALSE());
		}
		return $lVariables;
	}

	public function clearCachedVariables() {
		$_cachedVariables = array();
	}

	public function &getVariables(&$aObject, &$aUsingCaching = false) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to get variables from non-object');
		}
		if ($aUsingCaching) {
			// Don't using auto create, or inf-loop
			$lVariables = $this->getCachedVariables($aObject, epObjectUtil::_FALSE());
			if (!is_null($lVariables)) {
				return $lVariables;
			}
		}
		$lVariables = array();
		// Searching getter and setter
		$lMethods = get_class_methods(get_class($aObject));
		$lGetters = array();
		$lSetters = array();
		$lNames = array();
		foreach ($lMethods as $lMethod) {
			$lLowerMethodName = strtolower($lMethod);
			$lIndex = strpos($lLowerMethodName, 'is');
			if (!($lIndex === false) && $lIndex == 0) {
				$lVarName = substr($lLowerMethodName, 2);
				$lNames[$lVarName] = substr($lMethod, 2);
				$lNames[$lVarName] = strtolower(substr($lNames[$lVarName], 0, 1)).substr($lNames[$lVarName], 1);
				$lGetters[$lVarName] = $lMethod;
			} else {
				$lIndex = strpos($lLowerMethodName, 'get');
				if (!($lIndex === false) && $lIndex == 0) {
					$lVarName = substr($lLowerMethodName, 3);
					$lNames[$lVarName] = substr($lMethod, 3);
					$lNames[$lVarName] = strtolower(substr($lNames[$lVarName], 0, 1)).substr($lNames[$lVarName], 1);
					$lGetters[$lVarName] = $lMethod;
				} else {
					$lIndex = strpos($lLowerMethodName, 'set');
					if (!($lIndex === false) && $lIndex == 0) {
						$lVarName = substr($lLowerMethodName, 3);
						$lNames[$lVarName] = substr($lMethod, 3);
						$lNames[$lVarName] = strtolower(substr($lNames[$lVarName], 0, 1)).substr($lNames[$lVarName], 1);
						$lSetters[$lVarName] = $lMethod;
					}
				}
			}
		}
		// Identify variable by contains both getter and setter
		foreach ($lGetters as $lVarName => $lMethod) {
			if (isset($lSetters[$lVarName])) {
				$lDetails = array('NAME' => $lNames[$lVarName], 'GETTER' => $lGetters[$lVarName], 'SETTER' => $lSetters[$lVarName]);
				$lVariables[$lVarName] = $lDetails;
			}
		}
		// Add public variables which doesn't identidy by getter and setter
		$lVars = get_object_vars($aObject);
		foreach ($lVars as $lVarName => $lVar) {
			$lLowerVarName = strtolower($lVarName);
			if (!isset($lVariables[$lLowerVarName])) {
				$lDetails = array('NAME' => $lVarName, 'GETTER' => null, 'SETTER' => null);
			} else {
				$lDetails = $lVariables[$lLowerVarName];
				$lDetails['NAME'] = $lVarName;
			}
			$lVariables[$lLowerVarName] = $lDetails;
		}
		// Add OID
		if (!isset($lVariables['oid'])) {
			$lDetails = array('NAME' => 'oid', 'GETTER' => null, 'SETTER' => null);
			$lVariables['oid'] = $lDetails;
		}
		// Using real name rather then lower case name
		$lTempVariables = array();
		foreach ($lVariables as $lVarName => $lDetails) {
			$lVarName = $lDetails['NAME'];
			$lTempVariables[$lVarName] = $lDetails;
		}
		$lVariables = $lTempVariables;
		$this->setCachedVariables($aObject, $lVariables);
		// Debug
		// $this->debugVariable($aObject, $lVariables);
		return $lVariables;
	}

	public function setBySetter(&$aObject, &$aVariables, &$aVarName, &$aVarValue, &$aAutoCreate = false) {
		$lDetails = $this->getVariableDetails($aObject, $aVariables, $aVarName, $aAutoCreate);
		$lSetter = $lDetails['SETTER'];
		if (!is_null($lSetter)) {
			$aObject->$lSetter($aVarValue);
		} else {
			$aObject->$aVarName = $aVarValue;
		}
	}

	public function &getByGetter(&$aObject, &$aVariables, &$aVarName, &$aAutoCreate = false) {
		$lDetails = $this->getVariableDetails($aObject, $aVariables, $aVarName, $aAutoCreate);
		$lGetter = $lDetails['GETTER'];
		if (!is_null($lGetter)) {
			$lVarValue =& $aObject->$lGetter();
		} else {
			$lVarValue =& $aObject->$aVarName;
		}
		return $lVarValue;
	}

	public function isVariable(&$aObject, &$aVariables, &$aVarName, &$aAutoCreate = false) {
		$lDetails = $this->getVariableDetails($aObject, $aVariables, $aVarName, $aAutoCreate);
		return (!is_null($lDetails));
	}

	public function &getVariableValues(&$aObject, &$aVariables, &$aAutoCreate = false) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to get variable values on non-object');
		}
		if (is_null($aVariables)) {
			if ($aAutoCreate) {
				$aVariables = $this->getVariables($aObject, $aAutoCreate);
			} else {
				throw new Exception('Unable to get variable values without variables');
			}
		}
		$lVariableValues = array();
		foreach ($aVariables as $lVarName => $lDetails) {
			$lVariable = $this->getVariableName($aObject, $aVariables, $lVarName, $aAutoCreate);
			$lValue = $this->getByGetter($aObject, $aVariables, $lVarName, $aAutoCreate);
			$lVariableValues[$lVariable] = $lValue;
		}
		return $lVariableValues;
	}

	protected function &getVariableDetails(&$aObject, &$aVariables, &$aVarName, &$aAutoCreate = false) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to get variable details of '.$aVarName.' on non-object');
		}
		if (is_null($aVariables)) {
			if ($aAutoCreate) {
				$aVariables = $this->getVariables($aObject, $aAutoCreate);
			} else {
				throw new Exception('Unable to get variable details of '.$aVarName.' without variables');
			}
		}
		$lVarNameSearch = strtolower($aVarName);
		foreach ($aVariables as $lVarName => $lDetails) {
			$lVarName = strtolower($lVarName);
			if ($lVarNameSearch == $lVarName) {
				return $lDetails;
			}
		}
		throw new Exception('Object '.get_class($aObject).' does not contains variable '.$aVarName);
	}

	protected function &getVariableName(&$aObject, &$aVariables, &$aVarName, &$aAutoCreate = false) {
		$lDetails = $this->getVariableDetails($aObject, $aVariables, $aVarName, $aAutoCreate);
		$lVarName = $lDetails['NAME'];
		if (is_null($lVarName)) {
			return $aVarName;
		}
		return $lVarName;
	}

	protected function debugVariable(&$aObject, &$aVariables) {
		if (is_object($aObject)) {
			$lMessage = ' of '.get_class($aObject);
		} else {
			$lMessage = '';
		}
		echo '<table border="1">';
		echo '<tr><th colspan="2">Debug Variables'.$lMessage.'</th></tr>';
		foreach ($aVariables as $lVarName => $lDetails) {
			echo '<tr><td colspan="2"><b>'.$lVarName.'</b></td></tr>';
			echo '<tr><td>Name</td><td>'.$lDetails['NAME'].'</td></tr>';
			echo '<tr><td>Setter</td><td>'.$lDetails['SETTER'].'</td></tr>';
			echo '<tr><td>Getter</td><td>'.$lDetails['GETTER'].'</td></tr>';
		}
		echo '</table>';
	}

	public function &computeModifiedObjectId(&$aObject, &$aDatabase = null) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to compute object id on non-object');
		}
		if (!($aObject instanceof epObject)) {
			throw new Exception('Unable to compute object id on object which is not epObject');
		}
		$lOriginalObjectId = $this->getDecodeObjectId($aObject->epGetObjectId());
		$lObjectId = $this->getDecodeObjectId($this->computeObjectId($aObject, $aDatabase));
		$lModifiedVars = $aObject->epGetModifiedVars();
		$lObjectIdArray = array();
		foreach ($lModifiedVars as $lModifiedVar => $lModifiedValue) {
			$lObjectIdString = '';
			foreach ($lOriginalObjectId as $lOriginalIndex => $lOriginalParts) {
				$lOriginalName = $lOriginalParts['NAME'];
				$lOriginalValue = $lOriginalParts['VALUE'];
				if ($lOriginalName == $lModifiedVar) {
					foreach ($lObjectId as $lIndex => $lParts) {
						$lName = $lParts['NAME'];
						$lValue = $lParts['VALUE'];
						if ($lName == $lModifiedVar) {
							$lOriginalValue = $lValue;
							break;
						}
					}
				}
				$lObjectIdString .= $lOriginalName.':'.$lOriginalValue.';';
			}
			$lObjectIdArray[$lModifiedVar] = $lObjectIdString;
		}
		return $lObjectIdArray;
	}

	public function &computeObjectId(&$aObject, &$aDatabase = null) {
		if (!is_object($aObject)) {
			throw new Exception('Unable to compute object id on non-object');
		}
		if (!($aObject instanceof epObject)) {
			throw new Exception('Unable to compute object id on object which is not epObject');
		}
		$lOriginalObjectId = $this->getDecodeObjectId($aObject->epGetObjectId());
		$lClassname = $aObject->epGetClass();
		$lClassMap = epClassMapFactory::instance()->make($lClassname);
		$lFieldMaps = $lClassMap->getAllFields();
		$lIdFields = array();
		foreach($lFieldMaps as $lFieldMap) {
			if ($lFieldMap->isID()) {
				$lIdFields[] = $lFieldMap;
			}
		}
		$lObjectId = array();
		if (count($lIdFields) > 0) {
			foreach($lIdFields as $lIdField) {
				$lName = $lIdField->getName();
				$lValue = $aObject->epGet($lName);
				if (((is_bool($lValue) && !$lValue) || is_null($lValue)) && $lIdField->getIDType() == 'autogenerate') {
					if (!is_null($aDatabase)) {
						$lValue = $aDatabase->lastInsertId($lIdField->getColumnName());
					}
				} else if ($lValue instanceof epObject) {
					$lValueClassname = $lValue->epGetClass();
					$lValueClassMap = epClassMapFactory::instance()->make($lValueClassname);
					$lValuePKFields = $lValueClassMap->getPKField();
					$lValueName = $lValuePKFields[0]->getName();
					$lValue = $lValue->epGet($lValueName);
				}
				$lObjectId[] = array('NAME' => $lName, 'VALUE' => $lValue);
			}
		} else {
			$lName = $lClassMap->getOidColumn();
			$lValue = null;
			if (!is_null($aDatabase)) {
				$lValue = $aDatabase->lastInsertId($lName);
			}
			$lObjectId[] = array('NAME' => $lName, 'VALUE' => $lValue);
		}
		$lObjectIdString = '';
		foreach ($lObjectId as $lIndex => $lParts) {
			$lName = $lParts['NAME'];
			$lValue = $lParts['VALUE'];
			if ((is_bool($lValue) && !$lValue) || is_null($lValue)) {
				foreach ($lOriginalObjectId as $lOriginalIndex => $lOriginalParts) {
					$lOriginalName = $lOriginalParts['NAME'];
					$lOriginalValue = $lOriginalParts['VALUE'];
					if ($lOriginalName == $lName) {
						$lValue = $lOriginalValue;
						break;
					}
				}
			}
			if ((is_bool($lValue) && !$lValue) || is_null($lValue)) {
				throw new Exception('Fail to retireve value of "'.$lName.'" during compute object id.');
			}
			$lObjectIdString .= $lName.':'.$lValue.';';
		}
		return $lObjectIdString;
	}

	protected function &getDecodeObjectId(&$aObjectId) {
		$lIds = array();
		if (is_null($aObjectId)) {
			return $lIds;
		}
		$lParts = preg_split('/[;]+/', $aObjectId , -1, PREG_SPLIT_NO_EMPTY);
		foreach ($lParts as $lPart) {
			$lUnits = preg_split('/[:]+/', $lPart , -1, PREG_SPLIT_NO_EMPTY);
			$lIds[] = array('NAME' => $lUnits[0], 'VALUE' => $lUnits[1]);
		}
		return $lIds;
	}
}
?>