<?php


/**
 * $Id: epDbObject.php 1030 2007-01-19 10:38:55Z nauhygon $
 * 
 * Copyright(c) 2005 by Oak Nauhygon. All rights reserved.
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @author Trevan Richins <developer@ckiweb.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.db
 */

/**
 * need epClassMap class
 */
include_once (EP_SRC_ORM.'/epClassMap.php');

/**
 * Class of SQL statement generator
 * 
 * This class is responsible for converting class map info 
 * ({@link epClassMap}) into SQL statements
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.db
 */
class epObj2Sql {

    /**#@+
     * Used for return value to avoid reference notice in 5.0.x and up
     * @var bool
     */
    static public $false = false;
    static public $true = true;
    static public $null = null;
    /**#@-*/

	/**
	 * The cached db portability factory
	 * @var epDbPortableFactory
	 */
	static public $dbpf = false;

	/**
	 * Get the portable object
	 * @param string $dbtype
	 * @return false|epDbPortable
	 */
    static public function &getPortable($dbtype) {

		// check if we have portability factory cached already
		if (!epObj2Sql :: $dbpf) {
			include_once (EP_SRC_DB."/epDbPortable.php");
			epObj2Sql :: $dbpf = epDbPortFactory :: instance();
		}

		// get the portability object for the db
		if (!($dbp = & epObj2Sql :: $dbpf->make($dbtype))) {
            return self::$false;
		}

		return $dbp;
	}

	/**
     * Makes a SQL create index and unique statement for a class map
     * @param epDbObject $db the db connection 
     * @param epClassMap the class map for the object
     * @return false|string|array
     */
    static public function sqlCreateIndex($db, $cm, $curIndexes) {
        
        // get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
            return false;
        }

        $sqls = array();

        // build the CREATE INDEX queries as well
        $indexes = $dbp->createIndex($cm, $db, $curIndexes[1]);

        // build the CREATE UNIQUE INDEX queries as well
        $uniques = $dbp->createUnique($cm, $db, $curIndexes[0]);

        $sqls = array_merge(
            $sqls, 
            $indexes['drop'], $uniques['drop'], 
            $indexes['create'], $uniques['create']
            );

        return $sqls;
    }

    /**
	 * Makes a SQL create table statement for a class map
	 * @param epDbObject $db the db connection 
	 * @param epClassMap the class map for the object
	 * @return false|string|array
	 */
	static public function sqlCreate($db, $cm, $indent = '  ') {

			// get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
			return false;
		}

        // array to hold sql stmts
        $sqls = array();

		// call portability object to produce 
        $sqls[] = $dbp->createTable($cm, $db);

        // build the CREATE INDEX queries as well
        $indexes = $dbp->createIndex($cm, $db);
        
        // build the CREATE UNIQUE INDEX queries as well
        $uniques = $dbp->createUnique($cm, $db);

        // merge all sql statements
        $sqls = array_merge(
            $sqls,
            $indexes['drop'], $uniques['drop'],
            $indexes['create'], $uniques['create']
            );

        return $sqls;
	}

	/**
	 * Makes a SQL drop table if exists statement for a class map
	 * @param epDbObject $db the db connection  
	 * @param epClassMap the class map for the object
	 * @return string
	 */
	static public function sqlDrop($db, $cm) {

		// get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
			return false;
		}

		// call portability object to produce 
		return $dbp->dropTable($cm->getTable(), $db);
	}

	/**
	 * Makes a SQL truncate table if exists statement for a class map
	 * @param epDbObject $db the db connection 
	 * @param epClassMap the class map for the object
	 * @return string
	 */
	static public function sqlTruncate($db, $cm) {

		// get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
			return false;
		}

		// call portability object to produce 
		return $dbp->truncateTable($cm->getTable(), $db);
	}

	/**
	 * Makes a SQL count statement to get the total number rows in table 
	 * @param epDbObject $db the db connection  
	 * @param epClassMap the class map for the object
	 * @return string
	 */
	static public function sqlCount($db, $cm) {
        return 'SELECT COUNT(' . $db->quoteId($cm->getOidColumn()) . 
            ') FROM ' . $db->quoteId($cm->getTable());
	}

	/**
	 * Makes a SQL select statement from object variables
	 * If the object is null, select all from table
	 * @param epDbObject $db the db connection  
	 * @param epObject the object for query
	 * @param epClassMap the class map for the object
     * @param array (of integers) $oids_ex object ids to be excluded
     * @param array (of integers) $oids_in object ids to be included
	 * @return false|string
	 * @author Oak Nauhygon <ezpdo4php@gmail.com>
	 * @author Trevan Richins <developer@ckiweb.com>
	 */
    static public function sqlSelect($db, $cm, $o = null, $oids_ex = null, $oids_in = null) {

	 // Build a SQL statement with :
	 // - Discriminator value
	 // - Support customized eoid ( @see epObj2Sql::sqlSelectChildren )
        
        // !!!important!!! with a large db, the list of oid to be excluded
        // $oids_ex can grown really large and can significantly slow down 
        // queries. so it is suppressed and moved to epDbObject::_rs2obj() 
        // to process.
        // Remove $oids_ex from $oids_in
        if (is_array($oids_in) && is_array($oids_ex)) {
            foreach ($oids_ex as $lOidExKey => $lOidEx) {
                $lOidInKey = array_search($lOidEx, $oids_in);
                if (isset($oids_in[$lOidInKey])) {
                    unset($oids_in[$lOidInKey]);
                    unset($oids_ex[$lOidExKey]);
                }
            }
            if (count($oids_in) == 0) {
                // return array();
            }
        } else {
            $oids_ex = null;
        }
        // $oids_ex = null;

        // arrays to collect 'from' and 'where' parts
        $from = array();
        $where = array();

        
        // add table for the object in 'from'
        $from[] = $db->quoteId($cm->getTable());
        
        // if object is specified, recursively collect 'from' and 'where' 
        // for the select statement 
        if ($o) {
            $from_where = epObj2Sql::sqlSelectChildren($db, $o, 1, $cm->getTable());
            $from = array_merge($from, $from_where['from']);
            $where = array_merge($where, $from_where['where']);
        }
        
        // any oids to exclude?
        if ($oids_ex) {
            // add oids to be excluded (shouldn't get here. see comments above.)
            foreach($oids_ex as $oid) {
                // Convert to actual column name
                $oids = preg_split('/[;]+/', $oid, -1, PREG_SPLIT_NO_EMPTY);
                foreach ($oids as $oid) {
                    list($column, $value) = split(':', $oid);
                    $lField = $cm->getField($column);
                    if (!is_null($lField) && is_object($lField) && ($lField->isPrimitive() || ($lField instanceof epFieldMapRelationship))) {
                        // Handle primary key column with foreign Key constraint
                        $lFKs = $lField->getFK();
                        if (is_array($lFKs) && count($lFKs) > 0) {
                            $where[] = $db->quoteId($lFKs[0]) . ' <> ' . $db->quoteByTypeValue($value, $lField->getType());
                        } else {
                            $where[] = $db->quoteId($lField->getColumnName()) . ' <> ' . $db->quoteByTypeValue($value, $lField->getType());
                        }
                        // $where[] = $db->quoteId($lField->getColumnName()) . ' <> ' . $db->quoteByTypeValue($value, $lField->getType());
                    } else {
                        $where[] = $db->quoteId($column) . ' <> ' . $db->quote($value);
                    }
                }
                // $where[] = $db->quoteId($cm->getOidColumn()) . ' <> ' . $db->quote($oid);
            }
        }
        
        // add oids to be included
        if ($oids_in) {
            $_oids_in = array();

            // Convert to actual column name
            foreach($oids_in as $oid) {
                $oids = preg_split('/[;]+/', $oid, -1, PREG_SPLIT_NO_EMPTY);
                foreach ($oids as $oid) {
                    list($column, $value) = split(':', $oid);
                    $lField = $cm->getField($column);
                    if (!is_null($lField) && is_object($lField) && ($lField->isPrimitive() || ($lField instanceof epFieldMapRelationship))) {
                    // if (!is_null($lField) && is_object($lField) && $lField->isPrimitive()) {
                        // Handle primary key column with foreign Key constraint
                        $lFKs = $lField->getFK();
                        if (is_array($lFKs) && count($lFKs) > 0) {
                            $_oids_in[] = $db->quoteId($lFKs[0]) . ' = ' . $db->quoteByTypeValue($value, $lField->getType());
                        } else {
                            $_oids_in[] = $db->quoteId($lField->getColumnName()) . ' = ' . $db->quoteByTypeValue($value, $lField->getType());
                        }
                        // $_oids_in[] = $db->quoteId($lField->getColumnName()) . ' = ' . $db->quoteByTypeValue($value, $lField->getType());
                    } else {
                        $_oids_in[] = $db->quoteId($column) . ' = ' . $db->quote($value);
                    }
                }
                // $_oids_in[] = $db->quoteId($cm->getOidColumn()) . ' = ' . $db->quote($oid);
            }
            $where[] = '('.join(' OR ', $_oids_in).')';
        }

        // Add discriminator value into where cause [Added 7 lines below]
        if ($cm->getDiscriminatorValue() && epClassMap::getTopParent($cm) != $cm) {
            $dis_value = $cm->getDiscriminatorValue();
            $dis_col = $cm->getBaseSuperClassDiscriminatorColumn();
            $dis_col_name = array_keys($dis_col);
            $dis_col_type = array_values($dis_col);
            $where[] = $db->quoteId($dis_col_name[0]) . ' = ' . $db->quoteByTypeValue($dis_value, $dis_col_type[0]);
        }
        
        // columns to be selected (*: all of them)
        $columns = $db->quoteId($cm->getTable()) . '.*';
        // $columns = $db->quoteId($cm->getTable() . '.*');
        
        // assemble the select statement
        return epObj2Sql::_sqlSelectAssemble($columns, $from, $where);
	}
	
	/**
	 * Get Certain array field map record
	 *
	 */
	static public function sqlGetArrayTableRecord($db, $o, $cm, $array_fm){
		$pk_var_vals = $o->getDecodeObjectId();
		$pk_vars = array_keys($pk_var_vals);
		$pk_vals = array_values($pk_var_vals);
		$keyColumns = $array_fm->getArrayKeyColumns();
		
		$sql = 'Select '. $db->quoteId($array_fm->getArrayIndexColumn()) . ', ' . $db->quoteId($array_fm->getColumnName());
		$sql .= ' FROM ' . $db->quoteId($array_fm->getArrayTable());
		$sql .= ' WHERE ';
		
		for($i = 0; $i < count($pk_vals); $i++){
			$fm = $cm->getField($pk_vars[$i]);
			$sql .= $db->quoteId($keyColumns[$i]) . ' = ' . $db->quote($pk_vals[$i], $fm);
			if($i != count($pk_vals)-1){
				$sql .= ' AND ';
			}
		}
		
//		echo 'get array record sql : ' . $sql .'<bR>';
		return $sql;
	}

    /**
     * Assemble a select statement from parts. 
     * Note identifiers and values in all parts should have been properly quoted.
     * @param string|array $columns 
     * @param array $from from expressions
     * @param array $where where expressions ('1=1' if empty) 
     * @return string 
     */
    static protected function _sqlSelectAssemble($columns, $from, $where = array()) {
        
        // the columns clause
        $columns = is_array($columns) ? implode(' ', $columns) : $columns;
        
        // the from caluse
        $from = implode(', ', $from);
        
        // the where clause
        $where = $where ? implode(' AND ', $where) : '1=1';
        
        // put them together
        return 'SELECT '.$columns.' FROM '.$from.' WHERE '.$where;
    }

    /**
     * Make where part of a SQL select to get children values
     * @param epDbObject $db the db connection  
     * @param epObject $o the child object for query
     * @param int $depth how many children down we are
     * @param string $parent the parent of this child
     * @return array('from', 'where')
     * @author Oak Nauhygon <ezpdo4php@gmail.com>
     * @author Trevan Richins <developer@ckiweb.com>
     */
    static public function sqlSelectChildren($db, $o, $depth, $parent) {
        
     // Update the where cause to support customized eoid
     // - Since eoid is first customized

        // array to keep new tables in 'from'
        $from = array();

        // array to keep new expression in 'where'
        $where = array();
        
        // get the class map for the child object
        $cm = $o->epGetClassMap();
        
        // get all vars in the object
        $vars = $o->epGetVars();
        
        // if object has oid, select use oid
        if ($oid = $o->epGetObjectId()) {
            // Modify where cause
            $pieces = $o->getDecodeObjectId();
            foreach ($pieces as $name => $val) {
                $fm = $cm->getField($name);
                if ($fm != false) {
                    // Handle primary key column with foreign Key constraint
                    $lFKs = $fm->getFK();
                    if (is_array($lFKs) && count($lFKs) > 0) {
                        $where[] = $db->quoteId($lFKs[0]) . ' = ' . $db->quoteByTypeValue($val, $fm->getType());
                    } else {
                        $where[] = $db->quoteId($fm->getColumnName()) . ' = ' . $db->quoteByTypeValue($val, $fm->getType());
                    }
                } else {
                    $where[] = $db->quoteId($name) . ' = ' . $val;
                }
            }
            // $where[] = $db->quoteId($cm->getOidColumn()) . ' = ' . $oid; 
            return array('from'=>$from, 'where'=>$where);
        }
        
        // mark child object under search (to avoid loops)
        $o->epSetSearching(true);

        // total number of vars (primitive or non-primitive) collected
        $n = 0;

        // new depth
        $depth ++;

        // number of non-primitive (relationship) fields collected 
        $nprim_id = 0;

        // loop through vars
        while (list($var, $val) = each($vars)) { 
            
            // get field map
            if (!($fm = & $cm->getField($var))) {
                // should not happen
                continue;
            }
            
            // exclude null values (including empty strings)
            if (is_null($val) || (!$val && $fm->getType() == epFieldMap::DT_CHAR)) {
                continue;
            }
            
            // is it a primitive var?
            if ($fm->isPrimitive()) {
                $where[] = $db->quoteId($parent) . '.' . $db->quoteId($fm->getColumnName()) . ' = ' . $db->quote($val, $fm); 
                // done for this var
                $n ++;
                continue;
            }

            // okay we are dealing with a non-primitive (relationship) var
            if ($val instanceof epArray) {

                foreach ($val as $obj) {

                    // skip object that is under searching
                    if (!$obj || $obj->epIsSearching()) {
                        continue;
                    }
                    
                    // get 'where' and 'from' from relationship 
                    $from_where = epObj2Sql::sqlSelectRelations(
                        $db, $fm, $cm, $obj->epGetClassMap()->getTable(), 
                        $depth.$nprim_id, $parent
                        );
                    $where = array_merge($where, $from_where['where']);
                    $from = array_merge($from, $from_where['from']);

                    // get 'where' and 'from' from relationship 
                    $from_where = epObj2Sql::sqlSelectChildren($db, $obj, $depth, '_'.$depth.$nprim_id);
                    $where = array_merge($where, $from_where['where']);
                    $from = array_merge($from, $from_where['from']);
                    
                    $nprim_id++;
                }

            } else if ($val instanceof epObject && !$val->epIsSearching()) {
                
                // get 'where' and 'from' from relationship 
                $from_where = epObj2Sql::sqlSelectRelations(
                    $db, $fm, $cm, $val->epGetClassMap()->getTable(), 
                    $depth.$nprim_id, $parent
                    );
                $where = array_merge($where, $from_where['where']);
                $from = array_merge($from, $from_where['from']);

                // get 'where' and 'from' from relationship 
                $from_where = epObj2Sql::sqlSelectChildren($db, $val, $depth, '_'.$depth.$nprim_id);
                $where = array_merge($where, $from_where['where']);
                $from = array_merge($from, $from_where['from']);

                $nprim_id++;
            }

            $n ++;
        } 
        
        // reset search flag on child object
        $o->epSetSearching(false);

        return array('from' => $from, 'where' => $where);
    }

    /**
     * Make where part of a SQL select for relationship fields
     * @param epDbObject $db the db connection  
     * @param epFieldMap $fm the field map
     * @param epClassMap $cm the child object for query
     * @param string $alias the alias of this table in the previous part
     * @return array('from', 'where')
     * @author Oak Nauhygon <ezpdo4php@gmail.com>
     * @author Trevan Richins <developer@ckiweb.com>
     */
    static public function sqlSelectRelations($db, $fm, $cm, $table, $alias, $parentTable) {

    // - Do not use relation table

        $base_a = $fm->getBase_a();
        $class_a = $cm->getName();
        $var_a = $fm->getName();
        $base_b = $fm->getBase_b();

        // call manager to get relation table for base class a and b
        // Do not use relation table [Commentted 1 line below]
        // $rt = epManager::instance()->getRelationTable($base_a, $base_b);
        
        // the alias of the table we are dealing with right now
        $tbAlias = '_'.$alias;
        // Do not use relation table [Commentted 1 line below]
        // $rtAlias = 'rt'.$alias;
        
        // quoted aliases (avoid repeating)
        $tbAlias_q = $db->quoteId($tbAlias);
        // Do not use relation table [Commentted 1 line below]
        // $rtAlias_q = $db->quoteId($rtAlias);
        
        // compute 'from' parts: tables with aliases
        $from = array();
        $from[] = $db->quoteId($table) . ' AS '.$tbAlias_q;
        // Do not use relation table [Commentted 1 line below]
        // $from[] = $db->quoteId($rt) . ' AS '.$rtAlias_q;

        // compute expressions 'where'
        $where = array();
        
        // rt.class_a = 
        // Do not use relation table
        $lParentTablePKs = $cm->getPKField();
        $lChildTableCm = epClassMapFactory::instance()->make($cm->getTable());
        $lChildRelationFields = epClassMapFactory::instance()->getRelationFields($cm->getTable());

        $index = 0;
        foreach ($lChildRelationFields as $lChildRelationField) {
        	if ($index < count($lParentTablePKs)) {
        		$lChildFKs = $lChildRelationField->getFK();
        		$lParentRelationField = $lChildTableCm->getInverseField($lChildFKs);
        		$where[] = $parentTable.'.'.$lParentTablePKs[$index]->getColumnName().' = '.$tbAlias_q.'.'.$lChildFKs[0];
        	}
        	$index++;
        }
        // $where[] = $rtAlias_q.'.'.$db->quoteId('class_a').' = '.$db->quote($class_a);
        
        // rt.var_a = 
        // Do not use relation table [Commentted 1 line below]
        // $where[] = $rtAlias_q.'.'.$db->quoteId('var_a').' = '.$db->quote($var_a);
        
        // rt.base_b =
        // Do not use relation table [Commentted 1 line below]
        // $where[] = $rtAlias_q.'.'.$db->quoteId('base_b').' = '.$db->quote($base_b);
        
        // rt.class_b =  TODO: doesn't look like it is used
        //$where .= 'rt.'.$db->quoteId('class_b') . ' = ' . $db->quote($val->getClass());
        
        // A.oid = rt.oid_a
        // Do not use relation table [Commentted 1 line below]
        // $where[] = $db->quoteId($parentTable).'.'.$db->quoteId($cm->getOidColumn()).' = ' . $rtAlias_q.'.'.$db->quoteId('oid_a');
        
        // Child.oid = rt.oid_b
        // Do not use relation table [Commentted 1 line below]
        // $where[] = $tbAlias_q.'.'.$db->quoteId($fm->getClassMap()->getOidColumn()).' = ' . $rtAlias_q.'.'.$db->quoteId('oid_b');
        
        return array('from' => $from, 'where' => $where);
    }
	
	/**
	 * generate get partial table values sql
	 */
	 static public function sqlGetPartialTableRecord($db, $o, $cm, $table, $fks){
	 	$p_fms = $cm->getPartialTableField($table);
	 	$pk_var_vals = $o->getDecodeObjectId();
	 	$pk_vars = array_keys($pk_var_vals);
	 	$pk_vals = array_values($pk_var_vals);
	 	
	 	$sql = 'SELECT ';
	 	for($i = 0; $i < count($p_fms); $i++){
	 		$sql .= $db->quoteId($p_fms[$i]->getColumnName());
	 		if($i != count($p_fms) - 1){
	 			$sql .= ', ';	
	 		}
	 	}
	 	$sql .= ' FROM ' . $db->quoteId($table);
	 	$sql .= ' WHERE ';
	 	for($i = 0 ; $i < count($fks) ; $i++){
	 		$sql .= $db->quoteId($fks[$i]) . '=' . $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
	 		if($i != count($fks) - 1){
	 			$sql .= ' AND ';	
	 		}	
	 	}
	 	
//	 	echo '<strong> select partial table sql : </strong>' . $sql . '<br>';
	 	return $sql;
	 }

	/**
	 * Make a SQL insert statement from object variables
	 * @param epDbObject $db the db connection 
	 * @param epObject the object for query
	 * @param epClassMap the class map for the object
	 * @return false|string
	 */
	static public function sqlInsert($db, $cm, $o) {
		//get manager
		$m = $o->epGetManager();
		// get all vars
		if (!($vars = $o->epGetVars())) {
			return false;
		}
		
		// make select statement
		$sql = 'INSERT INTO '.$db->quoteId($cm->getTable()).' (';

		// Get object variables
		$lVars = $o->epGetVars();

		// get column names
		$i = 0;
		while (list ($var, $val) = each($vars)) {

            // exclude 'oid'
            if ($var == 'oid') {
                continue;
            }

			// shouldn't happen
			if (!($fm = $cm->getField($var))) {
				continue;
			}
			
			if($fm->isPrimitiveArray()){
				continue;	
			}
			
			if($fm->getPartialTable()){
				continue;	
			}
			// exclude non-primitive fields			
			//             if (!$fm->isPrimitive()) {
			//				continue;
			//            }

			if (is_null($val)) {
				continue;
			}

			//check is it autogenerate
			if ($fm->getIDType() == 'autogenerate') {
				// Allow auto generate id when value is epObject [Added 7 lines below]
				if ($val instanceof epObject && $val->epGetObjectId()) {
					$lFKs = $fm->getFK();
					foreach ($lFKs as $lFK) {
						$sql .= $db->quoteId($lFK). ', ';
						$i++;
					}
				}
				continue;
			}

			if ($fm->isPrimitive()) {
				$sql .= $db->quoteId($fm->getColumnName()).', ';
				$i++;
			} else {
				// Add relationship [Added 7 lines below]
				if ($val instanceof epObject && $val->epGetObjectId()) {
					$lFKs = $fm->getFK();
					foreach ($lFKs as $lFK) {
						$sql .= $db->quoteId($lFK). ', ';
						$i++;
					}
				} else
				if (get_class($val) == 'epObjectWrapper' and $val->epGetObjectId() and $foreignkeyCols = $fm->getFK()) {
					foreach ($foreignkeyCols as $fkCol) {
						$sql .= $db->quoteId($fkCol).', ';
						$i ++;
					}
					
					//further process for order collection
					//such as get the index column
					//for list
					$cm_inverse_a = $m->getClassMap($fm->getClass());
					$inverse_fm_a = $cm_inverse_a->getInverseFieldByInverse($fm->getFK());
					if(!is_null($inverse_fm_a) and $coll = $inverse_fm_a->getCollection()){
						$sql .= $db->quoteId($coll).', ';
						$i ++;
					}
				}
			} //end else
		}

		// no need to insert if we don't have any var to insert
//		if ($i == 0) {
//			$sql .= $db->quoteId($cm->getOidColumn()).') VALUES ('.$db->quote('', $fm).');';
//			return $sql;
//		}
		
		//add the discriminate value for inheritance class
		if($discriminator_column = $cm->getBaseSuperClassDiscriminatorColumn()){
			foreach($discriminator_column as $columnName => $columnType){
				$sql .= $db->quoteId($columnName). ', ';
				$i++;
			}
		}

		// Add collection index [Added 16 lines below]
		$lRelationFms = epClassMapFactory::instance()->getRelationFields($cm->getName());
		foreach ($lRelationFms as $lRelationFm) {
			$lCollection = $lRelationFm->getCollection();
			if (is_string($lCollection)) {
				$lInverse = $lRelationFm->getInverse();
				if (isset($lVars[$lInverse])) {
					$lInverseValue = $lVars[$lInverse];
					$lRelationName = $lRelationFm->getName();
					$lCollectionValue = $lInverseValue->$lRelationName;
					if ($lCollectionValue instanceof epArray) {
						$sql .= $db->quoteId($lCollection). ', ';
						$i++;
					}
				}
			}
		}

		// remove the last ', '
		if ($i > 0) {
			$sql = substr($sql, 0, strlen($sql) - 2);
		}

		$sql .= ') VALUES (';

		// get values
		$i = 0;
		reset($vars);

		while (list ($var, $val) = each($vars)) {

            // exclude 'oid'
            if ($var == 'oid') {
                continue;
            }

			if (!($fm = & $cm->getField($var))) {
				continue;
			}
			
			if($fm->isPrimitiveArray()){
				continue;	
			}
			
			if($fm->getPartialTable()){
				continue;	
			}
			
			// exclude non-primitive fields
			//            if (!$fm->isPrimitive()) {
			//                continue;
			//            }

			if (is_null($val)) {
				continue;
			}

			if ($fm->getIDType() == 'autogenerate') {
				// Allow auto generate id when value is epObject [Added 9 lines below]
				if ($val instanceof epObject && $val->epGetObjectId()) {
					$lParentCm = epClassMapFactory::instance()->make($fm->getClass());
					$lParentPKs = $lParentCm->getPKField();
					foreach ($lParentPKs as $lParentPK) {
						$lParentVar = $lParentPK->getName();
						$sql .= $db->quoteByTypeValue($val->$lParentVar, $lParentPK->getType()) . ', ';
						$i++;
					}
				}
				continue;
			}

			// get quoted field value
			if ($fm->isPrimitive()) {
				$sql .= $db->quote($val, $fm).', ';
				$i++;
			} else {
				// Add relationship [Added 20 lines below]
				if ($val instanceof epObject && $val->epGetObjectId()) {
					$lParentCm = epClassMapFactory::instance()->make($fm->getClass());
					$lParentPKs = $lParentCm->getPKField();
					foreach ($lParentPKs as $lParentPK) {
						$lParentVar = $lParentPK->getName();
						$lParentValue = $val->$lParentVar;
						if ($lParentValue instanceof epObject && $lParentValue->epGetObjectId()) {
							$lParentValueCm = epClassMapFactory::instance()->make($lParentValue->epGetClass());
							$lParentValuePKs = $lParentValueCm->getPKField();
							foreach ($lParentValuePKs as $lParentValuePK) {
								$lParentValueVar = $lParentValuePK->getName();
								$sql .= $db->quoteByTypeValue($lParentValue->$lParentValueVar, $lParentValuePK->getType()) . ', ';
								$i++;
							}
						} else {
							$sql .= $db->quoteByTypeValue($lParentValue, $lParentPK->getType()) . ', ';
							$i++;
						}
					}
				} else
				
//				if ($val instanceof epObjectWrapper and !$val->epGetObjectId()){
//					$val->commit();	
//				}
				
				if ($val instanceof epObjectWrapper and $val->epGetObjectId() and $fm->getFK()) {
					$primaryKeys = epObj2Sql :: getOidValues($val->epGetObjectId());
					foreach ($primaryKeys as $attr => $primaryKeyVal) {
						$oppFm = $val->epGetClassMap()->getField($attr);
						$sql .= $db->quote($primaryKeyVal, $oppFm).', ';
						++$i;
					}
					
					$cm_inverse_a = $m->getClassMap($fm->getClass());
					$inverse_fm_a = $cm_inverse_a->getInverseFieldByInverse($fm->getFK());
					if(!is_null($inverse_fm_a) and $list = $inverse_fm_a->getCollection()){
						$var = $inverse_fm_a->getName();
						$sql .= $db->quoteByTypeValue($val->$var->getIndex($o), $inverse_fm_a->getCollectionType()) . ', ';
						++$i;
					}
				}
			}
		}
		
		//set the discriminator_value for this class
		if($discriminator_value = $cm->getDiscriminatorValue()){
			$dis_type = array_values($discriminator_column);
			$sql .= $db->quoteByTypeValue($discriminator_value, $dis_type[0]). ', ';
			$i++;
		}

		// Add collection index [Added 16 lines below]
		$lRelationFms = epClassMapFactory::instance()->getRelationFields($cm->getName());
		foreach ($lRelationFms as $lRelationFm) {
			$lCollection = $lRelationFm->getCollection();
			if (is_string($lCollection)) {
				$lInverse = $lRelationFm->getInverse();
				if (isset($lVars[$lInverse])) {
					$lInverseValue = $lVars[$lInverse];
					$lRelationName = $lRelationFm->getName();
					$lCollectionValue = $lInverseValue->$lRelationName;
					if ($lCollectionValue instanceof epArray) {
						$sql .= $db->quoteByTypeValue($lCollectionValue->getIndex($o), $lRelationFm->getCollectionType()). ', ';
						$i++;
					}
				}
			}
		}

		// remove the last ', '
		if ($i > 0) {
			$sql = substr($sql, 0, strlen($sql) - 2);
		}

		// end of statement
		$sql .= ')';
		return $sql;
	}
	
	static public function sqlInsertArrayAttribute($db, $cm, $o){
		$sqls = array();
		if (!($vars = $o->epGetVars())) {
			return false;
		}
		
		while (list ($var, $val) = each($vars)) {
			if (!($fm = $cm->getField($var))) {
				continue;
			}
			
			if (is_null($val)) {
				continue;
			}
			
			if($fm->isPrimitiveArray()){
				$pk_fms = $cm->getPKField();
				foreach($val as $index => $v){
					$sql = 'INSERT INTO ' . $db->quoteId($fm->getArrayTable()) . '(';
					//key column
					$keyColumns = $fm->getArrayKeyColumns();
					for($i = 0; $i < count($keyColumns); $i++){
						$sql .= $db->quoteId($keyColumns[$i]);
						if($i != count($keyColumns)-1){
							$sql .= ', ';	
						}
					}
					//index
					$sql .= ', ';
					$sql .= $db->quoteId($fm->getArrayIndexColumn());
					$sql .= ', ';
					$sql .= $db->quoteId($fm->getColumnName());
					$sql .= ')VALUES(';
					
					if(count($pk_fms) > 0){
						for($i = 0; $i < count($pk_fms); $i++){
							$pk_fm = $pk_fms[$i];
							$pk_val = $o->epGet($pk_fm->getName());
							// Handle primary key column with foreign Key constraint
							if ($pk_val instanceof epObject) {
								$lReferenceClassMap = $pk_val->epGetClassMap();
								$lReferencePKs = $lReferenceClassMap->getPKField();
								if (is_array($lReferencePKs) && count($lReferencePKs) > 0) {
									$lReferencePKName = $lReferencePKs[0]->getName();
									$sql .= $db->quote($pk_val->$lReferencePKName, $pk_fm);
								} else {
									$sql .= '1 = 1';
								}
							} else {
								$sql .= $db->quote($pk_val, $pk_fm);
							}
							if($i != count($pk_fms) - 1){
								$sql .= ', ';	
							}	
						}
					}else{
						$ids = $o->getDecodeObjectId();
						$pk_column = $cm->getOidColumn();
						$sql .= $ids[$pk_column];	
					}
					$sql .= ', ';
//					if($fm->getArrayIndexType()){
//						$sql .= $db->quote($index);
//					}else{
//						$sql .= $index;	
//					}
					$sql .= $db->quoteByTypeValue($index, $fm->getArrayIndexType());
					$sql .= ', ';
					$sql .= $db->quote($v, $fm);
					$sql .= ')';
					$sqls[] = $sql;
//					echo 'insert array table sql : ' . $sql . '<br>';
				}	
				
			}
		}
		return $sqls;
	}
	
	static public function sqlInsertPartialTables($db, $cm, $o){
		$sqls = false;
		$partialTables = $cm->getPartialTables();
		$pk_var_vals = $o->getDecodeObjectId();
		$pk_vars = array_keys($pk_var_vals);
		$pk_vals = array_values($pk_var_vals);
		
		foreach($partialTables as $table => $fks_name){
			$count = 0;
			$sql = 'INSERT INTO ' . $db->quoteId($table) . '(';
			for($i = 0; $i < count($fks_name); $i++){
				$sql .= $db->quoteId($fks_name[$i]);
				if($i != count($fks_name) - 1){
					$sql .= ', ';	
				}
				$count++;	
			}
			
			$fms = $cm->getPartialTableField($table);
//			echo count($fms) . '<br>';
			if (!($vars = $o->epGetVars())) {
				return false;
			}
			
			$sql .= ', ';
			
			foreach($fms as $fm){
				if(is_null($vars[$fm->getName()])){
					continue;	
				}else{
					$count++;	
				}
				$sql .= $db->quoteId($fm->getColumnName()) . ', ';
			}
			if($count > 0){
				$sql = substr($sql, 0, strlen($sql) - 2);
			}else{
				return;	
			}
			
			$count = 0;
			$sql .= ')VALUES(';
			for($i = 0; $i < count($pk_vals); $i++){
				$sql .= $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
				if($i != count($pk_vals) - 1){
					$sql .= ', ';
				}
				$count++; 			
			}
			$sql .= ', ';
			foreach($fms as $fm){
				if(is_null($vars[$fm->getName()])){
					continue;	
				}else{
					$count++;	
				}
				$sql .= $db->quote($vars[$fm->getName()], $fm) . ', ';
			}
			
			if($count > 0){
				$sql = substr($sql, 0, strlen($sql) - 2);
				$sql .= ')';
			}else{
				return;	
			}
			$sqls[] = $sql;
//			echo '<strong>insert partial table sql : </strong>' . $sql .' <br>';
		}
		return $sqls;
	}

	/**
	 * Make a SQL delete statement from object variables
	 * @param epDbObject $db the db connection 
	 * @param epObject the object for query
	 * @param epClassMap the class map for the object
	 * @return false|string
	 */
	static public function sqlDelete($db, $cm, $o) {

		// get all vars
		$vars = $o->epGetVars();
		if (!$vars) {
			return false;
		}
		// delete row with the object id
		// $sql = 'DELETE FROM ' . $db->quoteId($cm->getTable()) . ' WHERE ' . $db->quoteId($cm->getOidColumn()) . ' = ' . $o->epGetObjectId();
		$sql = 'DELETE FROM '.$db->quoteId($cm->getTable()).' WHERE '.epObj2Sql :: getSqlOidColumnNameAndValues($o->epGetObjectId(), $cm, $db);
//		echo 'delete sql : ' . $sql . '<br>';
		return $sql;
	}
	
	/**
	 * generate sql to delete the primitive array
	 */
	static public function sqlDeletePrimitiveArray($db, $cm, $o){
		$sqls = false;
		$array_fms = $cm->getPrimitiveArray();
		$pk_var_vals = $o->getDecodeObjectId();
		$pk_vars = array_keys($pk_var_vals);
		$pk_values = array_values($pk_var_vals);
		
		foreach($array_fms as $array_fm){
			$keyCols = $array_fm->getArrayKeyColumns();
			$sql = 'DELETE FROM ' . $db->quoteId($array_fm->getArrayTable());
			$sql .= ' WHERE ';
			for($i = 0; $i < count($keyCols); $i++){
				$sql .= $db->quoteId($keyCols[$i]) . ' = ' . $db->quote($pk_values[$i], $cm->getField($pk_vars[$i]));
				if($i != count($keyCols) - 1){
					$sql .= ' AND ';
				}	
			}
			$sqls[] = $sql;
		}
//		print_r($sqls);
		return $sqls;
	}
	
	/**
	 * generate sql to delete the partial table
	 */
	 static public function sqlDeletePartialTables($db, $cm, $o, $partial_tables){
	 	$sqls = false;
	 	$pk_var_vals = $o->getDecodeObjectId();
		$pk_vars = array_keys($pk_var_vals);
		$pk_values = array_values($pk_var_vals);
		
		foreach($partial_tables as $table => $fks){
			$sql = 'DELETE FROM ';
			$sql .= $db->quoteId($table);
			$sql .= ' WHERE ';
			for($i = 0; $i < count($fks); $i++){
				$sql .= $db->quoteId($fks[$i]) . ' = ' . $db->quote($pk_values[$i], $cm->getField($pk_vars[$i]));
				if($i != count($fks) - 1){
					$sql .= ' AND ';	
				}		
			}
//			echo '<strong> delete partial table record : </strong>' . $sql . '<br>';
			$sqls[] = $sql;
		}
		return $sqls;
	 }

	/**
	 * Generate sql query to remove the relationship to many side 
	 * set the Foreign Key values to be NULL
	 */
	static public function sqlRemoveRelationship($db, $delObj, $non_p_fm, $inverse_fm, $cm_b) {
		$sqls = array ();
		if ($fk_b = $inverse_fm->getFK()) {
			$sql = 'Update '.$cm_b->getTable().' SET ';
			for ($i = 0; $i < count($fk_b); $i ++) {
				$sql .= $db->quoteId($fk_b[$i]).' = Null';
				if ($i != count($fk_b) - 1) {
					$sql .= ', ';
				}
			}
			$sql .= ' WHERE ';
			$obj_b = $delObj->epGet($non_p_fm->getName());

			if (!is_null($obj_b)) {
				if ($obj_b instanceof epArray && $obj_b->count() > 0) {
					$i = 0;
					foreach ($obj_b as $b) {
						$oid_b = $b->epGetObjectId();
						$sql .= epObj2Sql :: getSqlOidColumnNameAndValues($oid_b, $cm_b, $db);
						if ($i != $obj_b->count() - 1) {
							$sql .= ' OR ';
						}
						$i++;
					}
				} else if ($obj_b instanceof epObject) {
				// } else if($obj_b instanceof epObjectWrapper){
					$oid_b = $obj_b->epGetObjectId();
					$sql .= epObj2Sql :: getSqlOidColumnNameAndValues($oid_b, $cm_b, $db);
				}else{
					$sql = null;
				}
			} else {
				$sql = null;
			}
		}
//		echo 'delete many side sql :'.$sql.'<br>';
		return $sql;
	}

	/**
	 * 
	 */
	static public function sqlRemoveJointableRelationship($db, $jointable, $delObj, $cm, $fm) {
		$oid_attrs_vals = epObj2Sql :: getOidValues($delObj->epGetObjectId());
		$oid_vals = array_values($oid_attrs_vals);
		$oid_attrs = array_keys($oid_attrs_vals);
		$fk = $fm->getFK();
		$sql = 'DELETE FROM '.$jointable;
		$sql .= ' WHERE ';
		for ($i = 0; $i < count($fk); $i ++) {
			$fm = $cm->getField($oid_attrs[$i]);
			$sql .= $db->quoteId($fk[$i]).'='.$db->quote($oid_vals[$i], $fm);
			if ($i != count($fk) - 1) {
				$sql .= ' AND ';
			}
		}
//		$sql .= ';';
//		echo '<strong>delete jointable sql : '.$sql.'</strong><br>';
		return $sql;
	}

	/**
	 * Make a SQL update statement from object variables
	 * @param epObject the object for query
	 * @param epClassMap the class map for the object
	 * @return false|string
	 */
	static public function sqlUpdate($db, $cm, $o) {
		
		// get the modified vars
        $vars = $o->epGetModifiedVars(epObject::VAR_PRIMITIVE);
		if (!$vars) {
			return false;
		}

		$sql = 'UPDATE '.$db->quoteId($cm->getTable()).' SET ';
		$i = 0;
		while (list ($var, $val) = each($vars)) {
			// get field map
			if (!($fm = & $cm->getField($var))) {
				// should not happen
				continue;
			}

            // exclude 'oid'
            if ($fm->getName() == 'oid') {
                continue;
            }
            
			// exclude non-primitive fields
			if (!$fm->isPrimitive()) {
				continue;
			}
			
			if($fm->isPrimitiveArray()){
				continue;	
			}
			
			if($fm->getPartialTable()){
				continue;	
			}

			// get column name
			$sql .= $db->quoteId($fm->getColumnName()).'='.$db->quote($val, $fm).', ';

			$i ++;
		}

		if ($i == 0) {
			return false;
		}

		// remove the last ', '
		if ($i > 0) {
			$sql = substr($sql, 0, strlen($sql) - 2);
		}

		//        $sql .= ' WHERE ' . $db->quoteId($cm->getOidColumn()) . ' = ' . $o->epGetObjectId(); 
		$sql .= ' WHERE '.epObj2Sql :: getSqlOidColumnNameAndValues($o->epGetObjectId(), $cm, $db);
//		echo '<strong>update object sql : </strong>' . $sql .'<br>';
		return $sql;
	}
	
	static public function sqlUpdatePrimitiveArray($db, $cm, $o){
		//sqls statement include update, add, delete for the array table
		$sqls = false;
		$vars = $o->epGetModifiedVars();
		if (!$vars) {
			return false;
		}
		
		while (list ($var, $val) = each($vars)) {
			// get field map
			if (!($fm = & $cm->getField($var))) {
				// should not happen
				continue;
			}
			
			// exclude non-primitive fields
			if (!$fm->isPrimitive()) {
				continue;
			}
			
			if($fm->isPrimitiveArray()){
				$pk_var_vals = $o->getDecodeObjectId();
				$pk_vars = array_keys($pk_var_vals);
				$pk_vals = array_values($pk_var_vals);
				
				$keyColumns = $fm->getArrayKeyColumns();
				
				$sql_select_db = epObj2Sql::sqlGetArrayTableRecord($db, $o, $cm, $fm);
				$db->_execute($sql_select_db);
				$db_array = epObj2Sql::getPrimitiveArrayDBData($db, $fm);
				
				//start to compare object var's array withe $db array
				$obj_array = $o->epGet($fm->getName());
				
				$update_index = false;
				$add_index = false;
				$delete_index = false;
				
				foreach($obj_array as $obj_index => $obj_val){
					if(!isset($db_array[$obj_index])){
						$add_index[$obj_index] = $obj_val;
					}else if($obj_val != $db_array[$obj_index]){
						$update_index[$obj_index] = $obj_val;
					}
				}//end foreach
				
				foreach($db_array as $db_index => $db_val){
					if(!isset($obj_array[$db_index])){
						$delete_index[] = $db_index;	
					}	
				}
				
				//generate sql for update index
				if($update_index){
					foreach($update_index as $index => $value){
						$sql = 'UPDATE ' . $db->quoteId($fm->getArrayTable()) . ' SET ';
						
						$sql .= $db->quoteId($fm->getColumnName()) . ' = ';
						$sql .= $db->quote($value, $fm);
						$sql .= ' WHERE ' . $db->quoteId($fm->getArrayIndexColumn()) . ' = ';
						$sql .= $db->quoteByTypeValue($index, $fm->getArrayIndexType());
						$sql .= ' AND ';
						
						for($i= 0; $i < count($keyColumns); $i++){
							$sql .= $db->quoteId($keyColumns[$i]) . ' = ' . $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
							if($i != count($keyColumns) - 1){
								$sql .= ' AND ';	
							}
						}
						$sqls[] = $sql;	
					} 
				}//end generate update sql
				
				if($add_index){
					foreach($add_index as $index => $value){
						$sql = 'INSERT INTO ' . $db->quoteId($fm->getArrayTable()) . '(';
						for($i= 0; $i < count($keyColumns); $i++){
							$sql .= $db->quoteId($keyColumns[$i]);
							if($i != count($keyColumns) - 1){
								$sql .= ', ';	
							}
						}
						$sql .= ', ';
						$sql .= $db->quoteId($fm->getArrayIndexColumn());
						$sql .= ', ';
						$sql .= $db->quoteId($fm->getColumnName());
						
						$sql .= ')VALUES(';
						for($i = 0; $i < count($pk_vals); $i++){
							$sql .= $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
							if($i != count($pk_vals) - 1){
								$sql .= ', ';	
							}	
						} 
						$sql .= ', ';
						
						if($fm->getArrayIndexType() == 'string'){
							$sql .= $db->quote($index);	
						}else{
							$sql .= $index;	
						}
						
						$sql .= ', ';
						$sql .= $db->quote($value, $fm);
						$sql .= ')';
						$sqls[] = $sql;
					}	
				}//end to add new data to array table
				
				if($delete_index){
					foreach($delete_index as $value){
						$sql = 'DELETE FROM ' . $db->quoteId($fm->getArrayTable());
						$sql .= ' WHERE ';
						for($i= 0; $i < count($keyColumns); $i++){
							$sql .= $db->quoteId($keyColumns[$i]) . ' = ' . $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
							if($i != count($keyColumns) - 1){
								$sql .= ' AND ';	
							}
						}
						$sql .= ' AND ';
						$sql .= $db->quoteId($fm->getArrayIndexColumn()) . ' = ';
//						if($fm->getArrayIndexType() == 'string'){
//							$sql .= $db->quote($value);	
//						}else{
//							$sql .= $value;	
//						}
						$sql .= $db->quoteByTypeValue($value, $fm->getArrayIndexType());
						$sqls[] = $sql;
					}
				}//end delete data in array table
			}//end is primitive array
			
//			print_r($sqls);
		}
		return $sqls;
	}
	
	/**
	 * update value on partial table
	 */
	 static public function sqlUpdatePartialTable($db, $cm, $o){
	 	$sqls = false;
		
		$pk_var_vals = $o->getDecodeObjectId();
		$pk_vars = array_keys($pk_var_vals);
		$pk_vals = array_values($pk_var_vals);
		
		$vars = $o->epGetModifiedVars();
		if (!$vars) {
			return false;
		}
		$partialTables = $cm->getPartialTables();
		foreach($partialTables as $table => $fks){
			$sql = 'UPDATE ' . $db->quoteId($table) . ' SET ';
			$fms = $cm->getPartialTableField($table);
			$count = 0;
			foreach($fms as $fm){
				if(is_null($vars[$fm->getName()])){
					continue;
				}
				$count++;
				$sql .= $db->quoteId($fm->getColumnName()) . ' = ' . $db->quote($vars[$fm->getName()], $fm);
				$sql .= ", ";
			}
			
			if($count > 0){
				$sql = substr($sql, 0, strlen($sql) - 2);	
			}
			
			$sql .= ' WHERE ';
			for($i = 0; $i < count($fks); $i++){
				$sql .= $db->quoteId($fks[$i]) . '=' . $db->quote($pk_vals[$i], $cm->getField($pk_vars[$i]));
				if($i != count($fks) - 1){
					$sql .= ' AND ';	
				}
			}
//			echo '<strong>update partial table : </strong>' . $sql .'<br>';
			if($count > 0){
				$sqls[] = $sql;
			}
		}
		return $sqls;
	 }
	
	/**
	 * method to return the db array...
	 */
	static public function getPrimitiveArrayDBData($db, $fm){
		$array = array();
		$okay = $db->connection()->rsRestart();
		while($okay){
			$index = $db->connection()->rsGetCol($fm->getArrayIndexColumn());
			$value = $db->connection()->rsGetCol($fm->getColumnName());
			$array[$index] = $value;
			$okay = $db->connection()->rsNext();	
		}
		return $array;
	}
	
	/**
	 * Get all the B(Many) From A(One)
	 * A-->B
	 */
	static public function sqlGetAllB($db, $cm, $o, $fm) {
		if ($inverseAttr_b = $fm->getInverseAttribute()) {
			//1. select all bs from db 
			//2. compare to find delete and update record
			//3. generate sql
			$m = $o->epGetManager();
			$b_name = $fm->getClass(); //
			$cm_b = $m->getClassMap($b_name);
			$pk_fm_b = $cm_b->getPKField();

			$pk_a = $o->getDecodeObjectId();
			$pk_attr_a = array_keys($pk_a);
			$pk_val_a = array_values($pk_a);

//			$fk_b = $cm_b->getField($inverseAttr_b)->getFK();
			$fk_b = $inverseAttr_b;
			$select_all_sql_b = 'SELECT ';
			//b primary key value
			if (count($pk_fm_b) > 0) {
				// Record of column [Added 1 line below]
				$lColumnArray = array();
				for ($i = 0; $i < count($pk_fm_b); $i ++) {
					// Handle primary key column with foreign Key constraint
					$lFKs = $pk_fm_b[$i]->getFK();
					if (is_array($lFKs) && count($lFKs) > 0) {
						if (!isset($lColumnArray[$lFKs[0]])) {
							if (count($lColumnArray) > 0) {
								$select_all_sql_b .= ', ';
							}
							$select_all_sql_b .= $db->quoteId($lFKs[0]);
							$lColumnArray[$lFKs[0]] = true;
						}
					} else {
						if (!isset($lColumnArray[$pk_fm_b[$i]->getColumnName()])) {
							if (count($lColumnArray) > 0) {
								$select_all_sql_b .= ', ';
							}
							$select_all_sql_b .= $db->quoteId($pk_fm_b[$i]->getColumnName());
							$lColumnArray[$pk_fm_b[$i]->getColumnName()] = true;
						}
					}
					// $select_all_sql_b .= $db->quoteId($pk_fm_b[$i]->getColumnName());
					// if ($i != count($fk_b) - 1) {
					// 	$select_all_sql_b .= ', ';
					// }
				}
			} else {
				$select_all_sql_b .= $db->quoteId($cm_b->getOidColumn());
			}
			$select_all_sql_b .= ' FROM '.$db->quoteId($cm_b->getTable()).' WHERE ';
			for ($i = 0; $i < count($pk_a); $i ++) {
				$select_all_sql_b .= $db->quoteId($fk_b[$i]).' = '.$db->quote($pk_val_a[$i], $cm->getField($pk_attr_a[$i]));
				if ($i != count($pk_a) - 1) {
					$select_all_sql_b .= ' AND ';
				}
			}
//			$select_all_sql_b .= ';';
			return $select_all_sql_b;
		}
	}

	/**
	 * Select * from jointable where foreign key = ? and fk = ?
	 */
	static public function sqlGetAllRecordFromJoinTable($commitObj, $cm, $fm, $db) {
		$sql = 'SELECT * FROM ';
		$sql .= $db->quoteId($fm->getJointable());
		$sql .= ' WHERE ';
		$fk = $fm->getFK();

		$oid = $commitObj->epGetObjectId();
		$oid_col_vals = epObj2Sql :: getOidValues($oid);
		$oid_vals = array_values($oid_col_vals);
		$pk_fms = $cm->getPKField();

		$count = count($fk);
		for ($i = 0; $i < $count; $i ++) {
			$sql .= $db->quoteId($fk[$i]).' = '.$db->quote($oid_vals[$i], $pk_fms[$i]);
			if ($i != $count -1) {
				$sql .= ' AND ';
			}
		}
//		$sql .= ';';
		return $sql;
	}

    /**
     * Update relationships for a relationship var
     * 
     * @param string $rtable the name of the relationship table
     * @param string $base_a name of base class a
     * @param string $class_a name of class a
     * @param integer $oid_a object id of the class a object
     * @param integer $var_a the relational field of object a
     * @param string $base_b name of base b
     * @param array $oids_b oids of the class b object related to the class a object
     * 
     * @return false|array
     */
    static public function sqlUpdateRelationship($db, $rtable, $class_a, $var_a, $oid_a, $base_b, $oids_b) {
        
        // make sure we have params (except $oids_b) not empty
        if (!$rtable || !$class_a || !$var_a|| !$oid_a || !$base_b) {
            return false;
        }

        // delete all existing relationships
        $sql_del = 'DELETE FROM ' . $db->quoteId($rtable) . ' WHERE ';
        $sql_del .= $db->quoteId('class_a') . '=' . $db->quote($class_a) . ' AND ';
        $sql_del .= $db->quoteId('var_a') . '=' . $db->quote($var_a) . ' AND ';
        $sql_del .= $db->quoteId('oid_a') . '=' . $db->quote($oid_a) . ' AND ';
        $sql_del .= $db->quoteId('base_b') . '=' . $db->quote($base_b);

        // done if we don't have oids_b to insert
        if (!$oids_b) {
            return $sql_del;
        }

        // get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
            return false;
        }

        // columns for a relationship
        $cols = array('class_a', 'var_a', 'oid_a', 'base_b', 'class_b', 'oid_b');

        // the common values for a row
        $common = array($class_a, $var_a, $oid_a, $base_b);

        // rows to be inserted
        $rows = array();
        foreach($oids_b as $oid_b) {
            $row = $common;
            $row[] = $oid_b['class'];
            $row[] = $oid_b['oid'];
            $rows[] = $row;
        }
        
        // call portability object to create sql insert stmt
        $sql_ins = $dbp->insertValues($rtable, $db, $cols, $rows);
        if (is_array($sql_ins)) {
            // prepend the delete stmt
            array_unshift($sql_ins, $sql_del);
            return $sql_ins;
        }
        
        return array($sql_del, $sql_ins);
    }
    
    /**
     * Update relationships for a relationship var
     * @param string $rtable the name of the relationship table
     * @param string $class name of class a
     * @param string $oid name of oid
     * @return false|array
     */
    static public function sqlDeleteRelationship($db, $rtable, $class, $oid) {
        
        // make sure we have params (except $oids_b) not empty
        if (!$rtable || !$class) {
            return false;
        }

        // quoted ids
        $class_a_q = $db->quoteId('class_a');
        $oid_a_q = $db->quoteId('oid_a');
        $class_b_q = $db->quoteId('class_b');
        $oid_b_q = $db->quoteId('oid_b');
        
        // delete all existing relationships
        $sql  = 'DELETE FROM ' . $db->quoteId($rtable) . ' WHERE (';
        $sql .= $class_a_q . '=' . $db->quote($class);
        if ($oid) {
            $sql .= ' AND ' . $oid_a_q . '=' . $db->quote($oid);
        }
        $sql .= ') OR (';
        $sql .= $class_b_q . '=' . $db->quote($class);
        if ($oid) {
            $sql .= ' AND ' . $oid_b_q . '=' . $db->quote($oid);
        }
        $sql .= ')';
        
        return $sql;
    }

	/**
	 * Generate relationship sql from one-to-many
	 * when many side B oid is not null (mean uncommit)
	 */
	static public function sqlInsertRelationship($db, $obj_a, $obj_b, $cm, $fm) {
		
		if(!$obj_b->epGetObjectId() and count($fm->getInverseAttribute()) > 0){
			$obj_b->commit();
		}
		
		if ($obj_b->epGetObjectId() && !$fm->isMany()) {
			$objOidVals = $obj_b->getDecodeObjectId();
			$pk_attrs_b = array_keys($objOidVals);
			$pk_vals_b = array_values($objOidVals);

			$obj_a_oidVals = $obj_a->getDecodeObjectId();
			$pk_attrs_a = array_keys($obj_a_oidVals);
			$pk_vals_a = array_values($obj_a_oidVals);

			$cm_b = $obj_b->epGetClassMap();
			$pk_fms_b = $cm_b->getPKField();
//			$inverse_fm_b = $cm_b->getField($fm->getInverseAttribute());
			$inverse_fm_b = $cm_b->getInverseFieldByFK($fm->getInverseAttribute());
			if ($foreignKeys = $fm->getInverseAttribute()) {
				$sql = 'UPDATE '.$db->quoteId($cm_b->getTable()).' SET ';
				for ($i = 0; $i < count($foreignKeys); $i ++) {
//					$pk_fm_a = $cm->getField($oid_attrs_a[$i]);
					$pk_fm_a = $cm->getField($pk_attrs_a[$i]);
					$sql .= $db->quoteId($foreignKeys[$i]).' = '.$db->quote($pk_vals_a[$i], $pk_fm_a);
					if ($i != count($foreignKeys) - 1) {
						$sql .= ', ';
					}
				}
				$sql .= ' WHERE ';
				if (count($pk_fms_b)) {
					for ($i = 0; $i < count($pk_fms_b); $i ++) {
						$sql .= $db->quoteId($pk_fms_b[$i]->getColumnName()).' = '.$db->quote($pk_vals_b[$i], $pk_fms_b[$i]);
						if ($i != count($pk_fms_b) - 1) {
							$sql .= ' AND ';
						}
					}
				} else {
					$sql .= $db->quoteId($cm_b->getOidColumn()).'='.$pk_vals_b[0];
				}
//				$sql .= ';';
//				echo 'insertRelationShip SQL : '.$sql.'<br>';
				//prevent the $obj_b to update the relationship again
				//remove the var in Modified Vars array
				if($inverse_fm_b){
					$obj_b->epUnsetModifiedVars($inverse_fm_b->getName());
				}
				$sqls[] = $sql;
			}
			return $sqls;
		}
	}
	
	/**
	 * remove for JoinTable record
	 * Delete from jointable where col1 = val1 and col2 = val2
	 */
	static public function sqlRemoveJoinTableRecord($commitObj, $cm, $fm, $delete_oids_b, $cm_b, $db) {
		$sqls = null;
		$oid_a = $commitObj->epGetObjectId();
		$oid_attr_vals_a = epObj2Sql :: getOidValues($oid_a);
		$oid_vals_a = array_values($oid_attr_vals_a);

		$fk_a = $fm->getFK();
		$jointable = $fm->getJointable();
		$fk_b = $fm->getInverseFK();

		//get A and B primary Key field from class map
		$pk_fms_a = $cm->getPKField();
		$pk_fms_b = $cm_b->getPKField();				
		foreach ($delete_oids_b as $delete_oid_b) {
			$oid_attr_vals_b = epObj2Sql :: getOidValues($delete_oid_b);
			$oid_vals_b = array_values($oid_attr_vals_b);
			$sql = 'DELETE FROM '.$db->quoteId($jointable);
			$sql .= ' WHERE ';
			for ($i = 0; $i < count($fk_a); $i ++) {			
				$sql .= $db->quoteId($fk_a[$i])." = ".$db->quote($oid_vals_a[$i], $pk_fms_a[$i])." AND ";				
			}

			for ($i = 0; $i < count($fk_b); $i ++) {
				$sql .= $db->quoteId($fk_b[$i])." = ".$db->quote($oid_vals_b[$i], $pk_fms_b[$i]);
				if ($i != count($fk_b) - 1) {
					$sql .= ' AND ';
				}
			}
			$sqls[] = $sql;
		}
		return $sqls;
	}
	

	/**
	 * insert for JoinTable record
	 * Insert into jointable(col, col) values (val1, val2);
	 */
	static public function sqlInsertJoinTableRecord($commitObj, $cm, $fm, $insert_oids_b, $cm_b, $db) {
		$sqls = null;

		$oid_a = $commitObj->epGetObjectId();
		$oid_attr_vals_a = epObj2Sql :: getOidValues($oid_a);
		$oid_vals_a = array_values($oid_attr_vals_a);

		$fk_a = $fm->getFK();
		$jointable = $fm->getJointable();
//		$jointable_fm_b = $cm_b->getFieldByJoinTable($jointable);
//		$fk_b = $jointable_fm_b->getFK();
		$fk_b = $fm->getInverseFK();

		//get A and B primary Key field from class map
		$pk_fms_a = $cm->getPKField();
		$pk_fms_b = $cm_b->getPKField();
		
		foreach ($insert_oids_b as $insert_oid_b) {
			$oid_attr_vals_b = epObj2Sql :: getOidValues($insert_oid_b);
			$oid_vals_b = array_values($oid_attr_vals_b);
			$sql = 'INSERT INTO '.$db->quoteId($jointable);
			$sql .= '( ';

			foreach ($fk_a as $fk) {
				$sql .= $db->quoteId($fk).', ';
			}

			for ($i = 0; $i < count($fk_b); $i ++) {
				$sql .= $db->quoteId($fk_b[$i]);
				if ($i != count($fk_b) - 1) {
					$sql .= ', ';
				}
			}
			$sql .= ')';

			$sql .= ' VALUES (';

			if (count($pk_fms_a)) {
				for ($i = 0; $i < count($pk_fms_a); $i ++) {
					$sql .= $db->quote($oid_vals_a[$i], $pk_fms_a[$i]).', ';
				}
			} else {
				$sql .= $oid_vals_a[$i].', ';
			}

			if (count($pk_fms_b)) {
				for ($i = 0; $i < count($pk_fms_b); $i ++) {
					$sql .= $db->quote($oid_vals_b[$i], $pk_fms_b[$i]);
					if ($i != count($oid_vals_b) - 1) {
						$sql .= ', ';
					}
				}
			} else {
				$sql .= $oid_vals_a[$i];
			}
			$sql .= ')';
//			echo '<strong> Insert to join table : '.$sql.'</strong><br>';
			$sqls[] = $sql;
		}
		return $sqls;
	}
	/**
	 * Generate SQL to get the Foreign Key relationship from a --- > b
	 * a : many/one side - hold - the foreign key
	 * b : one side
	 */
	// - Add primary key column
	static public function sqlGetForeignKeyRelationship($db, $cm, $fm, $pk_vals_a, $pk_fms_a, $cm_b) {
		$fk_a = $fm->getFK();
		$sql = 'SELECT ';
		// Add primary key column [Added 22 lines below]
		$lColumnArray = array();
		for ($i = 0; $i < count($pk_fms_a); $i ++) {
			// Handle primary key column with foreign Key constraint
			$lFKs = $pk_fms_a[$i]->getFK();
			if (is_array($lFKs) && count($lFKs) > 0) {
				if (!isset($lColumnArray[$lFKs[0]])) {
					if (count($lColumnArray) > 0) {
						$sql .= ', ';
					}
					$sql .= $db->quoteId($lFKs[0]);
					$lColumnArray[$lFKs[0]] = true;
				}
			} else {
				if (!isset($lColumnArray[$pk_fms_a[$i]->getColumnName()])) {
					if (count($lColumnArray) > 0) {
						$sql .= ', ';
					}
					$sql .= $db->quoteId($pk_fms_a[$i]->getColumnName());
					$lColumnArray[$pk_fms_a[$i]->getColumnName()] = true;
				}
			}
		}

		for ($i = 0; $i < count($fk_a); $i ++) {
			if (!isset($lColumnArray[$fk_a[$i]])) {
				if (count($lColumnArray) > 0) {
					$sql .= ', ';
				}
				$sql .= $db->quoteId($fk_a[$i]);
				$lColumnArray[$fk_a[$i]] = true;
			}
			// $sql .= $db->quoteId($fk_a[$i]);
			// if ($i != count($fk_a) - 1) {
			// 	$sql .= ', ';
			// }
		}
		$sql .= ' FROM '.$db->quoteId($cm->getTable()).' WHERE ';

		if (count($pk_fms_a)) {
			$lColumnArray = array();
			for ($i = 0; $i < count($pk_fms_a); $i ++) {
				// Handle primary key column with foreign Key constraint
				if (!isset($pk_vals_a[$i])) {
					// In some case, primary key doesn't assigned value
					if (count($lColumnArray) > 0) {
						$sql .= ' AND ';
					}
					$sql .= '1 = 1';
					continue;
				}
				$lFKs = $pk_fms_a[$i]->getFK();
				if (is_array($lFKs) && count($lFKs) > 0) {
					if (!isset($lColumnArray[$lFKs[0]])) {
						if (count($lColumnArray) > 0) {
							$sql .= ' AND ';
						}
						$sql .= $db->quoteId($lFKs[0]).' = '.$db->quote($pk_vals_a[$i], $pk_fms_a[$i]);
						$lColumnArray[$lFKs[0]] = true;
					}
				} else {
					if (!isset($lColumnArray[$pk_fms_a[$i]->getColumnName()])) {
						if (count($lColumnArray) > 0) {
							$sql .= ' AND ';
						}
						$sql .= $db->quoteId($pk_fms_a[$i]->getColumnName()).' = '.$db->quote($pk_vals_a[$i], $pk_fms_a[$i]);
						$lColumnArray[$pk_fms_a[$i]->getColumnName()] = true;
					}
				}
				// $sql .= $db->quoteId($pk_fms_a[$i]->getColumnName()).' = '.$db->quote($pk_vals_a[$i], $pk_fms_a[$i]);
				// if ($i != count($pk_fms_a) - 1) {
				// 	$sql .= ' AND ';
				// }
			}
		} else {
			$sql .= $db->quoteId($cm->getOidColumn()).' = '.$pk_vals_a[0];
		}
//		$sql .= ';';
//		echo '<strong>foreign Key sql : '.$sql.'</strong><br>';
		return $sql;
	}

	/**
	 * Generate SQL to get the inverse relationship from a ---> b
	 * a: one side
	 * b: many/one side
	 */
	static public function sqlGetInverseRelatinship($db, $fm_a, $pk_val_a, $pks_fm_a, $cm_b, $fk_b) {
		$pk_fms_b = $cm_b->getPKField();
		$sql = 'SELECT ';
		if (count($pk_fms_b) > 0) {
			// Record of column [Added 1 line below]
			$lColumnArray = array();
			for ($i = 0; $i < count($pk_fms_b); $i ++) {
				// Handle primary key column with foreign Key constraint
				$lFKs = $pk_fms_b[$i]->getFK();
				if (is_array($lFKs) && count($lFKs) > 0) {
					if (!isset($lColumnArray[$lFKs[0]])) {
						if (count($lColumnArray) > 0) {
							$sql .= ', ';
						}
						$sql .= $db->quoteId($lFKs[0]);
						$lColumnArray[$lFKs[0]] = true;
					}
				} else {
					if (!isset($lColumnArray[$pk_fms_b[$i]->getColumnName()])) {
						if (count($lColumnArray) > 0) {
							$sql .= ', ';
						}
						$sql .= $db->quoteId($pk_fms_b[$i]->getColumnName());
						$lColumnArray[$pk_fms_b[$i]->getColumnName()] = true;
					}
				}
				// $sql .= $db->quoteId($pk_fms_b[$i]->getColumnName());
				// if ($i < count($pk_fms_b) - 1) {
				// 	$sql .= ', ';
				// }
			}
		} else {
			$sql = 'SELECT '.$db->quoteId($cm_b->getOidColumn());
		}
		
		//get the collection...
		if($coll = $fm_a->getCollection()){
			$sql .= ', ' . $db->quoteId($coll);	
		}

		$sql .= ' From '.$db->quoteId($cm_b->getTable()).' WHERE ';
		for ($i = 0; $i < count($fk_b); $i ++) {
			$sql .= $db->quoteId($fk_b[$i]) . ' = ' . $db->quote($pk_val_a[$i], $pks_fm_a[$i]);
			if ($i != count($fk_b) - 1) {
				$sql .= ' AND ';
			}
		}
		
//		echo '<strong> inverse rel sql :'.$sql.'</strong><br>';
		return $sql;
	}

	/**
	 * Generate SQL to get the relationship from JoinTable
	 */
	static public function sqlGetJoinTableRelationship($db, $pk_val_a, $pks_fm_a, $fk_a, $jointable, $fk_b) {
		$sql = 'SELECT ';
		for ($i = 0; $i < count($fk_b); $i ++) {
			$sql .= $db->quoteId($fk_b[$i]);
			if ($i != count($fk_b) - 1) {
				$sql .= ', ';
			}
		}

		$sql .= ' From '.$db->quoteId($jointable).' WHERE ';
		if (count($pks_fm_a)) {
			for ($i = 0; $i < count($fk_a); $i ++) {
				$sql .= $db->quoteId($fk_a[$i]).'='.$db->quote($pk_val_a[$i], $pks_fm_a[$i]);
				if ($i != count($fk_a) - 1) {
					$sql .= ' AND ';
				}
			}
		} else {
			$sql .= $db->quoteId($fk_a).'='.$pk_val_a[0];
		}
//		echo 'jointable sql : '.$sql.'<br>';
		return $sql;
	}

	/*
	 * To decode the oid String the format like
	 * authorID:author1;
	 * column Name : val; other col : other val;
	 */
	static public function getSqlOidColumnNameAndValues($oidString, $cm, $db) {
		$idSqlPart = null;
		$pieces = preg_split('/[;]+/', $oidString, -1, PREG_SPLIT_NO_EMPTY);
		$count = 0;
		foreach ($pieces as $unit) {
			list ($name, $val) = preg_split('/[:]+/', $unit, -1, PREG_SPLIT_NO_EMPTY);
			$fm = $cm->getField($name);
			if ($fm != false) { // check the $fm, if oid cannot get Field, it is int
				// Handle primary key column with foreign Key constraint
				$lFKs = $fm->getFK();
				if (is_array($lFKs) && count($lFKs) > 0) {
					$idSqlPart .= $db->quoteId($lFKs[0]).' = '.$db->quote($val, $fm).' AND ';
				} else {
					$idSqlPart .= $db->quoteId($fm->getColumnName()).' = '.$db->quote($val, $fm).' AND ';
				}
				// $idSqlPart .= $db->quoteId($fm->getColumnName()).' = '.$db->quote($val, $fm).' AND ';
				$count ++;
			} else {
				$idSqlPart .= $db->quoteId($name).' = '.$val.' AND ';
				$count ++;
			}
		}

		//remove last AND
		if ($count > 0) {
			$idSqlPart = substr($idSqlPart, 0, strlen($idSqlPart) - 5);
		}
		return $idSqlPart;
	}

	static public function getOidValues($oidString) {
		$values = array ();
		$pieces = preg_split('/[;]+/', $oidString, -1, PREG_SPLIT_NO_EMPTY);
		foreach ($pieces as $unit) {
			list ($name, $val) = preg_split('/[:]+/', $unit, -1, PREG_SPLIT_NO_EMPTY);
			$values[$name] = $val;
		}
		return $values;
	}

	/**
	 * Makes a SQL comment for a table (class map)
	 * @param epClassMap the class map for the object
	 * @return string
	 */
	static public function sqlTableComments($db, $cm) {
		$sql = "\n";
		$sql .= "-- \n";
		$sql .= "-- Table for class ".$cm->getName()."\n";
		$sql .= "-- Source file: ".$cm->getClassFile()."\n";
		$sql .= "-- \n\n";
		return $sql;
	}
    
    /**
     * Returns the random function
     * @param epDbObject $db the db connection 
     * @return false|string
     */
    static public function sqlRandom($db) {
        
        // get the portable
        if (!($dbp = & epObj2Sql::getPortable($db->dbType()))) {
            return false;
        }

        // call portability object to get random function
        return $dbp->randomFunc();
    }
}

/**
 * Exception class for {@link epDbObject}
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.base 
 */
class epExceptionDbObject extends epException {
}

/**
 * Class for object operations with databases
 * 
 * This class provides a layer between the database access (i.e. 
 * {@link epDb}) and the persisent objects. 
 * 
 * It translates persistence-related operations into SQL statements 
 * and executes them by calling {@link epDb::execute()}. 
 * 
 * It implements the two-way conversions, from database rows to 
 * persistent objects, and vice versa. 
 * 
 * It also supports table-level operations for mapped classes - 
 * table creation, table dropping, and emptying. 
 * 
 * Objects of this class is managed by the db factory, {@link 
 * epDbFactory}. 
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.db
 */
class epDbObject {

    /**#@+
     * Used for return value to avoid reference notice in 5.0.x and up
     * @var bool
     */
    static public $false = false;
    static public $true = true;
    static public $null = null;
    /**#@-*/
    
	/**
	 * The reference to the epDb object that connects to the database
	 * @var epDb
	 */
	protected $db;

	/**
	 * Last inserted table
	 * @var string
	 */
	protected $table_last_inserted = false;

	/**
     * Whether to check if table exists before db operation
     * @var boolean
     */
    protected $check_table_exists = true;

    /**
	 * The cached manager
	 * @var epManager
	 */
	protected $ep_m = false;

	/**
     * The 'order by's used for sorting
     * @var array
     */
    protected $orderbys = array();
    
    /**
	 * Constructor
	 * @param epDb
	 */
	public function __construct($db) {
		$this->db = $db;
	}

	/**
	 * Destructor
	 * Close db connection
	 */
	public function __destruct() {
        if ($this->db) {
            $this->db->close();
        }
    }

	/**
     * Returns whether to check a table exists before any db operation
     * @param boolean $v (default to true)
     */
    public function getCheckTableExists() {
        return $this->check_table_exists;
    }

    /**
     * Sets whether to check a table exists before any db operation
     * @param boolean $v (default to true)
     */
    public function setCheckTableExists($v = true) {
        $this->check_table_exists = $v;
    }

    /**
	 * Return the database type defined in {@link epDb}
	 * @return string
	 */
	public function dbType() {
		return $this->db->dbType();
	}

	/**
	 * Return the db connection (epDb)
	 * @return epDb
	 */
	public function & connection() {
		return $this->db;
	}

	/**
	 * Fetchs objects using the variable values specified in epObject
	 * If the object is null, get all objects in table. 
     * @param array $cms an array of epClassMap
     * @param array $sql_stmts an array of SQL statements
     * @param string $aggr_func
     * @param string $orderby
     * @param string $limit
     * @return false|array
     */
    public function &query($cms, $sql_stmts, $orderby = false, $limit = false, $aggr_func = false) {
        if ($aggr_func) {
            $result = $this->_queryAggrFunc($sql_stmts, $aggr_func);
            return $result;
        }
        $os = $this->_queryObjects($sql_stmts, $cms, $orderby, $limit);
        return $os;
    }

    /**
     * Fetchs objects using the variable values specified in epObject
     * If the object is null, get all objects in table. 
	 * @param string $sql
     * @param array $cms an array of epClassMap
     * @param string $orderbys
     * @param string $limit
	 * @return false|array
     * @throws epExceptionDbObject
	 */
    protected function _queryObjects($sql_stmts, $cms, $orderbys, $limit) {

        $result = array();
        foreach ($sql_stmts as $index => $sql_stmt) {
            
            // stmt preproc 
            $sql_stmt = $this->_queryPreproc($sql_stmt);
            
            // execute sql stmt
            if (!$this->_execute($sql_stmt)) {
                return self::$false;
            }

            // result conversion
            if ($r = $this->_rs2obj($cms[$index])) {
                // Handle array or partial table attributes [Added 23 lines below]
                $cm = $cms[$index];
                // handle array attributes
                if ($cm->hasPrimitiveArray()) {
                    $array_fms = $cm->getPrimitiveArray();
                    foreach ($r as $individual_r) {
                        foreach ($array_fms as $array_fm) {
                            $sql = epObj2Sql::sqlGetArrayTableRecord($this, $individual_r, $cm, $array_fm);
                            $this->_execute($sql);
                            $this->addArrayValueToObj($individual_r, $array_fm);
                        }
                    }
                }

                // handle partial table attribute
                if ($partialTables = $cm->getPartialTables()) {
                    foreach($r as $individual_r){
                        foreach($partialTables as $table => $fks){
                            $sql = epObj2Sql::sqlGetPartialTableRecord($this, $individual_r, $cm, $table , $fks);
                            $this->_execute($sql);
                            $this->addPartialTableValueToObj($individual_r, $cm, $table);
                        }
                    }
                }
                $result = array_merge($result, $r);
            }
        }
        
        // sortby
        if ($orderbys) {
            
            // random orderby
            if (count($orderbys) && $orderbys[0]['dir'] == 'random') {
                shuffle($result);
            } 
            // asc|desc orderbys 
            else {
                $this->orderbys = $orderbys;
                usort($result, array($this, '__sort'));
            }
        }
        
        // limit (string)
        if ($limit) {
            $limit = trim(substr(trim($limit), strlen('limit')));
            $parts = explode(' OFFSET ', $limit);
            if (count($parts) == 2) {
                $amount = $parts[0];
                $offset = $parts[1];
            } else {
                $amount = $parts[0];
                $offset = 0;
            }
            $result = array_slice($result, $offset, $amount);
        }

        return $result;
    }

    /**
     * Preprocesses a SQL statement before query
     * @return string 
     */
    private function _queryPreproc($sql_stmt) {
        
        if (false !== strpos($sql_stmt, 'RANDOM()')) {
            
            // replace RANDOM
            $sql_stmt = str_replace('RANDOM()', epObj2Sql::sqlRandom($this).'()', $sql_stmt);
            
            // kludge: SELECT DISTINCT does not work with ORDER BY RANDOM()
            if ($this->db->dbType() == 'Postgres') {
                $sql_stmt = str_replace('SELECT DISTINCT', 'SELECT', $sql_stmt);
            }
        }

        return $sql_stmt;
    }

    /**
     * Sorts two objects 
     * @param epObject $a
     * @param epObject $b
     * @throws epExceptionDbObject
     */
    private function __sort($a, $b) {
        
        // tie if no orderbys
        if (!$this->orderbys) {
            return 0;
        }
        
        // go through each orderby
        foreach($this->orderbys as $orderby) {
            
            // sign by direction
            $sign = $orderby['dir'] == 'desc' ? -1 : + 1;
            
            // get values from a and b
            $path = $orderby['path'];
            $va = epArrayGet($a, $path);
            $vb = epArrayGet($b, $path);
            
            // boolean or numeric
            if (is_bool($va) || is_numeric($va)) {
                // a < b
                if ($va < $vb) {
                    return -1 * $sign;
                }
                // a > b
                else if ($va > $vb) {
                    return +1 * $sign;
                }
                continue;
            } 
            
            // string
            if (is_string($va)) {
                // a < b
                if (($r = strcmp($va, $vb)) < 0) {
                    return -1 * $sign;
                }
                // a > b
                else if ($r > 0) {
                    return +1 * $sign;
                }
                continue;
            }
            
            // invalid orderby value
            throw new epExceptionDbObject('Invalid ORDER BY [' . $path . '] value');
        }
        
        // tie
        return 0;
    }

    /**
     * Execute queries with aggregate functions
     * @param array $cms an array of class maps (epClassMap)
     * @param array $sql_stmts an array of SQL statements
     * @param string $aggr_func
     * @return false|array
     */
    protected function _queryAggrFunc($sql_stmts, $aggr_func)    {
        
        // it is a single sql stmt?
        if (1 == count($sql_stmts)) {
            return $this->_queryAggrFunc1($sql_stmts[0], $aggr_func);
        }
        
        // special treatment for average func
        if (0 === stripos($aggr_func, 'AVG(')) {
            // aggreate function: AVG()
            return $this->_queryAggrFuncAverage($sql_stmts, $aggr_func);
        }
        
        // simple aggregate functions: COUNT, MAX, MIN, SUM
        return $this->_queryAggrFuncSimple($sql_stmts, $aggr_func);
    }

    /**
     * Execute a single SQL stmt with aggregate function 
     * @param array $cms an array of class maps (epClassMap)
     * @param array $sql_stmts an array of SQL statements
     * @param string $aggr_func
     * @return false|array
     */
    protected function _queryAggrFunc1($sql_stmt, $aggr_func) {

		// execute sql
        if (!$this->_execute($sql_stmt)) {
            return self::$false;
		}

        // are we dealing with an aggregation function
        return $this->_rs2aggr($aggr_func);
    }

    /**
     * Execute queries with simple aggregate functions (COUNT, MIN, MAX, SUM)
     * @param array $cms an array of class maps (epClassMap)
     * @param array $sql_stmts an array of SQL statements
     * @param string $aggr_func
     * @return false|array
     */
    protected function _queryAggrFuncSimple($sql_stmts, $aggr_func) {
        
        $result = null;
        foreach ($sql_stmts as $index => $sql_stmt) {
            
            // execute single sql stmt with aggregate func
            try {
                $r = $this->_queryAggrFunc1($sql_stmt, $aggr_func);
	}
            catch(Exception $e) {
                $r = null;
            }
            if (is_null($r)) {
                continue;
            }


            // collect results according to aggregate function
            if (0 === stripos($aggr_func, 'COUNT') || 0 === stripos($aggr_func, 'SUM')) {
                $result = is_null($result) ? $r : ($result + $r);
            }
            else if (0 === stripos($aggr_func, 'MIN')) {
                $result = is_null($result) ? $r : min($result, $r);
            }
            else if (0 === stripos($aggr_func, 'MAX')) {
                $result = is_null($result) ? $r : max($result, $r);
            }
        }
        
        return $result;
    }

	/**
     * Execute queries with aggregate function AVG (special treatment)
     * @param array $sql_stmts an array of SQL statements
     * @param string $aggr_func
     * @return false|array
     */
    protected function _queryAggrFuncAverage($sql_stmts, $aggr_func) {
        
        // sum stmts
        $sum_func = str_ireplace('AVG(', 'SUM(', $aggr_func);
        $sql_stmts_sum = array();
        foreach($sql_stmts as $sql_stmt) {
            $sql_stmts_sum[] = str_replace($aggr_func, $sum_func, $sql_stmt);
        }
        $sum = $this->_queryAggrFuncSimple($sql_stmts_sum, $sum_func);
        
        // count stmts
        $count_func = 'COUNT(*)';
        $sql_stmts_count = array();
        foreach($sql_stmts as $sql_stmt) {
            $sql_stmts_count[] = str_replace($aggr_func, $count_func, $sql_stmt);
        }
        $count = $this->_queryAggrFuncSimple($sql_stmts_count, $count_func);
        
        return $sum / $count;
    }

    /**
	 * Returns the total number of stored object in a class
	 * @param epClassMap
	 * @return false|integer
	 */
	public function count($cm) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return false;
        }
        
		// preapre sql statement
		if (!($sql = epObj2Sql :: sqlCount($this, $cm))) {
			return false;
		}

		// execute sql
		if (($r = $this->_execute($sql)) === false) {
			return false;
		}

		// check query result
		$this->db->rsRestart();
		$count = $this->db->rsGetCol('COUNT('.$this->quoteId($cm->getOidColumn()).')', 'count');
		if (!is_numeric($count)) {
			return false;
		}

		// return the number of rows found in the class table
		return $count;
	}

	/**
	 * Fetchs objects using the variable values specified in epObject
	 * If the object is null, get all objects in table. 
	 * @param epObject $o
	 * @param epClassMap
     * @param array (of integer) $oids_ex object ids to be excluded
     * @param array (of integer) $oids_in object ids to be included 
     * @param bool $objs convert the rows into objects or leave as uoids
	 * @return false|array
	 */
    public function fetch($cm, $o = null, $oids_ex = null, $oids_in = null, $objs = true) {
        
        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return self::$false;
        }
        
        // make sure the table is created
        if (!$this->create($cm, false)) {
            return self::$false;
        }

        // preapre sql statement
        if (!($sql = epObj2Sql::sqlSelect($this, $cm, $o, $oids_ex, $oids_in))) {
            return self::$false;
		}
		
//		echo 'epdbobject : 557 :select sql :'.$sql.'<br>';
		// execute sql
		if (!$this->_execute($sql)) {
            return self::$false;
		}

        if ($objs) {
            // result conversion
            $r = $this->_rs2obj($cm, $oids_ex);
        } else {
            $r = $this->_rs2uoid($cm, $oids_ex);
        }
		
		//handle array attributes
		if($cm->hasPrimitiveArray()){
			$array_fms = $cm->getPrimitiveArray();
			foreach($r as $individual_r){
				foreach($array_fms as $array_fm){
					$sql = epObj2Sql::sqlGetArrayTableRecord($this, $individual_r, $cm, $array_fm);
					$this->_execute($sql);
					$this->addArrayValueToObj($individual_r, $array_fm);
				}
			}
		}
		
		//handle partial table attribute
		if($partialTables = $cm->getPartialTables()){
			foreach($r as $individual_r){
				foreach($partialTables as $table => $fks){
					$sql = epObj2Sql::sqlGetPartialTableRecord($this, $individual_r, $cm, $table , $fks);
					$this->_execute($sql);
					$this->addPartialTableValueToObj($individual_r, $cm, $table);
				}
			}		
		}
		
		// result conversion
		return ($r);
	}

	/**
	 * Fetchs records using the variable values specified in epObject
	 * @param epObject $o
	 * @param epClassMap
	 * @return bool
	 */
	public function insert($cm, $o) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return false;
        }
        
		// make sure the table is created
		if (!$this->create($cm, false)) {
			return false;
		}

		// update table for last insertion
		$this->table_last_inserted = $cm->getTable();

		// preapre sql statement
		if (!($sql = epObj2Sql :: sqlInsert($this, $cm, $o))) {
			return false;
		}
		
//		echo 'insert sql : '.$sql.'<br>';
		// execute sql
		return ($r = $this->_execute($sql));
	}
	
	public function insertArray($cm, $o){
		//handle array attribute
		$sqls = epObj2Sql::sqlInsertArrayAttribute($this, $cm, $o);
		if(count($sqls) > 0){
			return ($r = $this->_execute($sqls));
		}else{
			return true;
		}
	}
	
	public function insertPartialTableData($cm, $o){
		$sqls = epObj2Sql::sqlInsertPartialTables($this, $cm, $o);
		if($sqls){
			$this->_execute($sqls);	
		}
	}

	/**
	 * Update Many side
	 * If many is have id
	 */
	public function insertRelationShip($cm, $o) {
		$sqls = null;
		if (!($vars = $o->epGetVars())) {
			return false;
		}

		foreach ($vars as $var => $val) {
			$vars[$var] = $o->$var;
		}

		while (list ($var, $val) = each($vars)) {
			if (!($fm = $cm->getField($var))) {
				continue;
			}

			// Add relationship [Added 54 lines below]
			if ($val instanceof epObject && $val->epGetObjectId()) {
				$sqls = array();
				$lFKs = $fm->getFK();
				$lSql = 'UPDATE '.$this->quoteId($cm->getTable()).' SET ';
				$lReferenceCm = epClassMapFactory::instance()->make($fm->getClass());
				$lReferencePKs = $lReferenceCm->getPKField();
				$lIndex = 0;
				foreach ($lReferencePKs as $lReferencePK) {
					$lReferenceVar = $lReferencePK->getName();
					if ($lIndex > 0) {
						$lSql .= ', ';
					}
					$lSql .= $this->quoteId($lFKs[$lIndex]).' = '.$this->quoteByTypeValue($val->$lReferenceVar, $lReferencePK->getType());
					$lIndex++;
				}
				$lSql .= ' WHERE ';
				$lPKs = $cm->getPKField();
				$lIndex = 0;
				foreach ($lPKs as $lPK) {
					$lVar = $lPK->getName();
					if ($lIndex > 0) {
						$lSql .= ' AND ';
					}
					// Handle primary key column with foreign Key constraint
					$lFKs = $lPK->getFK();
					if (is_array($lFKs) && count($lFKs) > 0) {
						$lReferenceValue = $o->$lVar;
						if ($lReferenceValue instanceof epObject) {
							$lReferenceClassMap = $lReferenceValue->epGetClassMap();
							$lReferencePKs = $lReferenceClassMap->getPKField();
							if (is_array($lReferencePKs) && count($lReferencePKs) > 0) {
								$lReferencePKName = $lReferencePKs[0]->getName();
								$lSql .= $this->quoteId($lFKs[0]) . ' = ' . $this->quoteByTypeValue($lReferenceValue->$lReferencePKName, $lPK->getType());
							} else {
								$lSql .= '1 = 1';
							}
						} else {
							$lSql .= $this->quoteId($lFKs[0]) . ' = ' . $this->quoteByTypeValue($lReferenceValue, $lPK->getType());
						}
					} else {
						$lSql .= $this->quoteId($lPK->getColumnName()) . ' = ' . $this->quoteByTypeValue($o->$lVar, $lPK->getType());
					}
					$lIndex++;
				}
				if (count($lPKs) == 0) {
					$units = preg_split('/[;]+/', $o->epGetObjectId(), -1, PREG_SPLIT_NO_EMPTY);
					for ($i = 0; $i < count($units); $i++) {
						$parts = preg_split('/[:]+/', $units[$i], -1, PREG_SPLIT_NO_EMPTY);
						$lSql .= $this->quoteId($parts[0]).' = '.$this->quoteId($parts[1]);
					}
				}
//				$lSql .= ';';
				$sqls[] = $lSql;
			} else

			if (!$fm->isPrimitive() && $inverseAttr = $fm->getInverseAttribute()) {
				$oid_attrs_vals_a = $o->getDecodeObjectId();
				$oid_vals_a = array_values($oid_attrs_vals_a);
				$oid_attrs_a = array_keys($oid_attrs_vals_a);
				if (get_class($val) == 'epArray' && $val->count() > 0) { //one-to-many
					foreach ($val as $obj_b) {
						$sqls = epObj2Sql :: sqlInsertRelationship($this, $o, $obj_b, $cm, $fm);
					}
				} else if (($val instanceof epObjectWrapper)){
					$sqls = epObj2Sql :: sqlInsertRelationship($this, $o, $val, $cm, $fm);
				}
			}
		} //while

		if (!is_null($sqls)) {
			foreach ($sqls as $sql) {
				$this->_execute($sql);
			}
		}
	}

	/*
	 * Create the Inverse class Primary Key String
	 */
	public function getInverseEncodePrimaryKey($db, $o, $cm, $fm, $pk_val_a, $cm_b, $fm_b) {
		$oids = array ();
		$oid_attrs_vals_a = epObj2Sql :: getOidValues($o->epGetObjectId());
		$pk_vals_a = array_values($oid_attrs_vals_a);
		$pk_fms_a = $cm->getPKField();
		$fk_b = $fm_b->getFK();
		$pk_fms_b = $cm_b->getPKField();
		$sql = epObj2Sql :: sqlGetInverseRelatinship($this, $fm, $pk_vals_a, $pk_fms_a, $cm_b, $fk_b);

		$this->_execute($sql);
		$okay = $this->db->rsRestart();
		while ($okay) {
			if (count($pk_fms_b)) {
				$oid = '';
				foreach ($pk_fms_b as $pk_fm_b) {
					// Handle primary key column with foreign Key constraint
					$lFKs = $pk_fm_b->getFK();
					if (is_array($lFKs) && count($lFKs) > 0) {
						$val = $this->db->rsGetCol($lFKs[0]);
					} else {
						$val = $this->db->rsGetCol($pk_fm_b->getColumnName());
					}
					// $val = $this->db->rsGetCol($pk_fm_b->getColumnName());
					$oid .= /*$pk_fm_b->getColumnName() . ':' . */
					$val.';';
				}
			} else {
				$val = $this->db->rsGetCol($cm_b->getOidColumn());
				$oid = /*$cm_b->getOidColumn() . ':' .*/
				$val.';';
			}
			
			// If collection is exist get the collection
			$coll_value = null;
			if($coll = $fm->getCollection()){
				$coll_value = $this->db->rsGetCol($coll);
			}
			
			$oid = '('.$cm_b->getName().')'.$oid;
			if(is_null($coll_value)){
				$oids[] = $oid;
			}else{
				$oids[$coll_value] = $oid;	
			}
			$okay = $this->db->rsNext();
		}
		return $oids;
	}

	/*
	 * Generate the Foreign Key value to relation class
	 */
	public function getForeignKeyEncodePrimaryKey($o, $cm, $fm, $cm_b) {
		$oids = array ();

		$oid_attrs_vals_a = epObj2Sql :: getOidValues($o->epGetObjectId());
		$pk_vals_a = array_values($oid_attrs_vals_a);
		$pk_fms_a = $cm->getPKField();
		$pk_fms_b = $cm_b->getPKField();

		$sql = epObj2Sql :: sqlGetForeignKeyRelationship($this, $cm, $fm, $pk_vals_a, $pk_fms_a, $cm_b);
		$this->_execute($sql);
		$okay = $this->db->rsRestart();
		$fks = $fm->getFK();
		while ($okay) {
			$oid = '';
			foreach ($fks as $fk) {
				$val = $this->db->rsGetCol($fk);
				$oid .= /*$pk_fm_b->getColumnName() . ':' . */
				$val.';';
				if (is_null($val)) {
					$oid = null;
				}
			}

			if (!is_null($oid)) {
				$oid = '('.$cm_b->getName().')'.$oid;
				$oids[] = $oid;
			}

			$okay = $this->db->rsNext();
		}
		return $oids;
	}

	/**
	 * Generate the jointable relationship oid of B classes
	 */
	public function getJoinTableEncodePrimaryKey($o, $cm, $fk_a, $jointable, $cm_b, $fm) {
		$oids = array ();

		$oid_attrs_vals_a = epObj2Sql :: getOidValues($o->epGetObjectId());
		$pk_val_a = array_values($oid_attrs_vals_a);
		$pks_fm_a = $cm->getPKField();
//		$fk_b = $fm_b->getFK();
		$fk_b = $fm->getInverseFK();
		
		$sql = epObj2Sql :: sqlGetJoinTableRelationship($this, $pk_val_a, $pks_fm_a, $fk_a, $jointable, $fk_b);
		$this->_execute($sql);
		$okay = $this->db->rsRestart();
		while ($okay) {
			$oid = null;
			foreach ($fk_b as $fk) {
				$val = $this->db->rsGetCol($fk);
				$oid .= $val.';';
			}

			$oid = '('.$cm_b->getName().')'.$oid;
			$oids[] = $oid;
			$okay = $this->db->rsNext();
		}
		return $oids;
	}

	/**
	 * Fetchs records using the variable values specified in epObject
	 * @param epObject $o
	 * @param epClassMap
	 * @return bool
	 */
	public function delete($cm, $o) {
        
        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return false;
        }

		$sql = false;
		$del_array_sql = false;
		$del_partial_table_sql = false;
		// preapre sql statement
		$sql = epObj2Sql :: sqlDelete($this, $cm, $o);
		if($cm->hasPrimitiveArray()){
			$del_array_sql = epObj2Sql::sqlDeletePrimitiveArray($this, $cm, $o);
		}
		
		if($partial_tables = $cm->getPartialTables()){
			$del_partial_table_sql = epObj2Sql::sqlDeletePartialTables($this, $cm, $o, $partial_tables);
		}
		
		//execute delete sql
		if($del_array_sql){
			$this->_execute($del_array_sql);
		}
		
		if($del_partial_table_sql){
			$this->_execute($del_partial_table_sql);	
		}
		
		// execute sql
		if($sql){
			return ($r = $this->_execute($sql));
		}else{
			return false;
		}
	}

	/**
	 * Fetchs records using the variable values specified in epObject
	 * @param epObject $o
	 * @param epClassMap
	 */
	public function update($cm, $o) {
        
        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return false;
        }

		// preapre sql statement
		$r = true;
		$sql = epObj2Sql :: sqlUpdate($this, $cm, $o);
		
		if($sql){
			$r = $this->_execute($sql);
		}
		
		if($cm->hasPrimitiveArray()){
			$this->updatePrimitiveArray($cm, $o);
		}
		
		if($cm->getPartialTables()){
			$this->updatePartialTable($cm, $o);
		}
		return $r;
	}
	
	public function updatePrimitiveArray($cm, $o){
		$sqls = epObj2Sql::sqlUpdatePrimitiveArray($this, $cm, $o);
		if($sqls){
			return ($r = $this->_execute($sqls));
		}
		return true;
	}
	
	public function updatePartialTable($cm, $o){
		$sqls = epObj2Sql::sqlUpdatePartialTable($this, $cm, $o);
		if($sqls){
			return ($r = $this->_execute($sqls));	
		}
		return true;
	}

	/**
	 * Update the modified relatonShip var (non-primitive)
	 */
	public function updateRelationShip1($cm, $o/*, $m*/) {
		$sqls = null;
		$vars = $o->epGetModifiedVars();
		$m = $o->epGetManager();
		
		if (!$vars) {
			return false;
		}

		foreach ($vars as $var => $val) {
			$vars[$var] = $o->$var;
		}

		//get db record to create string array with oids_b[]{(xclass)pk;pk1;
		while (list ($var, $val) = each($vars)) {
			if (is_null($val)) {
				continue;
			}

			// get field map
			if (!($fm = & $cm->getField($var))) {
				// should not happen
				continue;
			}

			if (!$fm->isPrimitive()) {
				$classname_b = $fm->getClass();
				$cm_b = $m->getClassMap($classname_b);
				if ($fm->getInverseAttribute()) {
					$objOids = null;
					//$val
					if (is_array($val) || $val instanceof epArray) {
						foreach ($val as $val_b) {
							if(!$val_b->epGetObjectId()){
								$val_b->commit();	
							}
							$objOids[] = $val_b->epGetObjectId();
						}
					} else {
						if(!$val->epGetObjectId()){
							$val->commit();	
						}
						$objOids[] = $val->epGetObjectId();
					}
					
					$getAllsql = epObj2Sql :: sqlGetAllB($this, $cm, $o,$fm);
					if ($getAllsql) {
						$cm_b = $m->getClassMap($fm->getClass());
						$oids_b = $this->encodeOidValue($getAllsql, $cm_b);
					}

					

					$deleteOids = array ();
					foreach ($oids_b as $oid_b) {
						if(is_null($objOids)){
							$deleteOids[] = $oid_b;
						}
						//not exist, deletet oid 
						else if (!in_array($oid_b, $objOids)) {
							$deleteOids[] = $oid_b;
						}
					}
					$inverseAttr = $fm->getInverseAttribute();
					//tmp overwriteObj
					$fm_b = $cm_b->getInverseFieldByFK($inverseAttr);
					if (is_array($val) || $val instanceof epArray) {
						if($overwriteObjs = $val->getOverWriteObjs()){
							foreach($overwriteObjs as $obj){
								$obj->epUnsetModifiedVars($fm_b->getName());	
							}
						}	
					}
									
					if (is_array($val) || $val instanceof epArray) {
						foreach ($val as $val_b) {
							$fm_b = $cm_b->getInverseFieldByFK($inverseAttr);
							if ($o !== $val_b->epGet($fm_b->getName())) {
								$val_b->epSet($fm_b->getName(), $o);
							}
						}
					} else {
						$fm_b = $cm_b->getInverseFieldByFK($inverseAttr);
						if ($o !== $val->epGet($fm_b->getName())) {
							$val->epSet($fm_b->getName(), $o);
						}
					}

					//create delete and update sql
					if (count($deleteOids)) {
						foreach ($deleteOids as $deleteOid) {
//							$fm_b = $cm_b->getField($inverseAttr);
							$fm_b = $cm_b->getInverseFieldByFK($inverseAttr);
							$fk_b = $fm_b->getFK();
							$cm_b->getPKField();
							$sql = 'Update '.$this->quoteId($cm_b->getTable()).' SET ';

							for ($i = 0; $i < count($fk_b); $i ++) {
								$sql .= $this->quoteId($fk_b[$i]).' = NULL ';
								if ($i != count($fk_b) - 1) {
									$sql .= ', ';
								}
							} // end for
							
							//remove the index value in the object
							if($coll  = $fm->getCollection()){
								$sql .= ', ';
								$coll_type = $fm->getCollectionType();
								if($coll_type == 'string'){
									$sql .= $this->quoteId($coll).' = NULL ';
								}else{
									$sql .= $this->quoteId($coll).' = 0 ';
								}
							}
							
							//deleteOids
							$sql .= ' WHERE ';
							$sql .= epObj2Sql :: getSqlOidColumnNameAndValues($deleteOid, $cm_b, $this)/*.';'*/;
//							echo 'inverse delete sql : '.$sql.'<br>';
							$sqls[] = $sql;
						} //end foreach
					} //end count($deleteOids)
				} else
					if ($fk_a = $fm->getFK() and !$fm->getJointable()) {
						$oids_b = epObj2Sql :: getOidValues($val->epGetObjectId());
						$oid_vals_b = array_values($oids_b);
						$oid_attr_b = array_keys($oids_b);
						if (count($oid_vals_b)) {
							$sql = 'UPDATE '.$this->quoteId($cm->getTable()).' SET ';
							for ($i = 0; $i < count($fk_a); $i ++) {
								$fm_b = $cm->getField($oid_attr_b[$i]);
								$sql .= $this->quoteId($fk_a[$i]).' = '.$this->quote($oid_vals_b[$i], $fm_b);
								if ($i != count($fk_a) - 1) {
									$sql .= ', ';
								}
							}
							//update the collection between a and b
							$cm_a = $val->epGetClassMap();
//							$fm_a = $cm_a->getInverseField($var);
							// print_r($fm->getFK());
							$fm_a = $cm_a->getInverseFieldByInverse($fm->getFK());
							if($coll = $fm_a->getCollection()){
								$sql .= ', ' . $this->quoteId($coll) . ' = ';
								$var_a = $fm_a->getName();
								$index_val = $val->$var_a->getIndex($o);
								$sql .= $this->quoteByTypeValue($index_val, $fm_a->getCollectionType());
							}
							
							$sql .= ' WHERE ';
							$sql .= epObj2Sql :: getSqlOidColumnNameAndValues($o->epGetObjectId(), $cm, $this)/*.';'*/;
//							echo 'fk sql update : '.$sql.'<br>';
							$sqls[] = $sql;
						} else {
//							echo 'no oid for other side object <br>';
						}
					}
			} //End isPrimitive()

			// Automatic update Object Id after update relationship [Added 4 lines below]
			$lModifiedObjectIds = epObjectUtil::instance()->computeModifiedObjectId($o);
			if (isset($lModifiedObjectIds[$var])) {
				$o->epSetObjectId($lModifiedObjectIds[$var]);
			}
		}

		if (count($sqls)) {
			$this->_execute($sqls);
		}
	}

	public function updateJoinTableRelationShip($commitObj, $cm, $fm, $value, $cm_b) {

		$sql = epObj2Sql :: sqlGetAllRecordFromJoinTable($commitObj, $cm, $fm, $this);
//		$fm_b = $cm_b->getFieldByJoinTable($fm->getJointable());
		$fk_b = $fm->getInverseFK();
		$db_oids_b = $this->encodeDBJoinTableInf($sql, $cm_b, $fk_b);
		$store_in_a_oids_b = array ();

		foreach ($value as $val_b) {
			$store_in_a_oids_b[] = $val_b->epGetObjectId();
		}

		//if ($value->count() > 0) {
			$delete_b = array ();
			$insert_b = array ();

			//delete record after compare db --> store in object a
			foreach ($db_oids_b as $db_oid_b) {
				if (!in_array($db_oid_b, $store_in_a_oids_b)) {
					$delete_b[] = $db_oid_b;
				}
			}

			foreach ($store_in_a_oids_b as $store_in_a_oid_b) {
				if (!in_array($store_in_a_oid_b, $db_oids_b)) {
					$insert_b[] = $store_in_a_oid_b;
				}
			}
			
			//final
			if (count($insert_b)) {
				$sqls = epObj2Sql :: sqlInsertJoinTableRecord($commitObj, $cm, $fm, $insert_b, $cm_b, $this);
				$this->_execute($sqls);
			}
			
			if (count($delete_b)) {
				$sqls = epObj2Sql :: sqlRemoveJoinTableRecord($commitObj, $cm, $fm, $delete_b, $cm_b, $this);
				$this->_execute($sqls);
			}
			
		//}
	}

	/**
	 * remove not composed relationship
	 * remove the foreign key on many side
	 */
	public function removeRelationShip($delObj, $cm, $m) {
		if ($delObj->epGetObjectId()) {
			$non_primitive_fms = $cm->getNonPrimitive();
			foreach ($non_primitive_fms as $non_p_fm) {
				if ($inverseAttr = $non_p_fm->getInverseAttribute() and !$non_p_fm->isComposedOf()) {
					$cm_b = $m->getClassMap($non_p_fm->getClass());
					$inverse_fm_b = $cm_b->getInverseFieldByFK($inverseAttr);
					$sql = epObj2SQL :: sqlRemoveRelationship($this, $delObj, $non_p_fm, $inverse_fm_b, $cm_b);
					if (!is_null($sql)) {
						return $this->_execute($sql);
					}
				} else
					if ($jointable = $non_p_fm->getJointable()) {
						$sql = epObj2Sql :: sqlRemoveJointableRelationship($this, $jointable, $delObj, $cm, $non_p_fm);
						if (!is_null($sql)) {
							return $this->_execute($sql);
						}
					}
			}
		}
		return true;
	}

	/**
	 * change obj to (ClassName)PKValue; 
	 */
//	protected function encodeObjValue($fm, $val) {
//		foreach ($val as $obj) {
//			echo $obj->epGetObjectId();
//		}
//	}

	/**
	 * resultset to PKAttr:PKValue;
	 */
	protected function encodeOidValue($sql, $cm_b) {
		$this->_execute($sql);
		$okay = $this->db->rsRestart();
		$pk_fms_b = $cm_b->getPKField();
		$oids = array ();
		while ($okay) {
			$oid = null;
			if (count($pk_fms_b) > 0) {
				foreach ($pk_fms_b as $pk_fm_b) {
					$val = $this->db->rsGetCol($pk_fm_b->getColumnName());
					if (!is_null($val)) {
						$oid = $pk_fm_b->getName().':'.$val.';';
					}
				}
			} else {
				$val = $this->db->rsGetCol($cm_b->getOidColumn());
				if (!is_null($val)) {
					$oid = $cm_b->getOidColumn().':'.$val.';';
				}
			}

			if (!is_null($oid)) {
				$oids[] = $oid;
			}
			$okay = $this->db->rsNext();
		}
		return $oids;
	}

	/**
	 * generate the b side oid value
	 * many to many 
	 * a --> b
	 * from db to generate db 'b''oid value
	 */
	 
	protected function encodeDBJoinTableInf($sql, $cm_b, $fk_b) {
		$this->_execute($sql);
		$okay = $this->db->rsRestart();
		$oids_b = array ();
		$pks_b = $cm_b->getPKField();

		while ($okay) {
			$oid_b = null;
			if (count($pks_b)) {
				for ($i = 0; $i < count($fk_b); $i ++) {
					$val = $this->db->rsGetCol($fk_b[$i]);
					$oid_b .= $pks_b[$i]->getName().':'.$val.';';
				}
			} else {
				$val .= $this->db->rsGetCol($fk_b[0]);
				$oid_b .= $cm_b->getOidColumn().':'.$val.';';
			}
			$oids_b[] = $oid_b;
			$okay = $this->db->rsNext();
		}
		return $oids_b;
	}
	
	public function getDiscrimationValue($cm, $oid){
		$dis_column = epClassMap::getTopParent($cm)->getDiscriminatorColumn();
		$column_name = array_keys($dis_column);
		$sql .= 'SELECT ' . $this->quoteId($column_name[0]) . ' FROM ' . $cm->getTable() . ' WHERE ';
		$sql .= epObj2Sql::getSqlOidColumnNameAndValues($oid, $cm, $this);
//		$sql .= ';';
		$this->_execute($sql);
		
		
		$okay = $this->db->rsRestart();
		while ($okay){
			if($result = $this->db->rsGetCol($column_name[0])){
				return $result;	
			}
			$okay = $this->db->rsNext();	
		}
	}
	
	/**
	 * Create a table specified in class map if not exists
     * @param epClassMap $cm The class map 
     * @param bool $force Whether to force creating table 
	 * @return bool
	 */
	public function create($cm, $force = false) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            // if so, no need to actually create 
            return true;
        }
        
        // Do not use relationship table [Added 3 lines below]
        if ('epObjectRelation' == $cm->getName()) {
            return true;
        }
        
			// check if table exists
        if (!$force && $this->_tableExists($cm->getTable())) {
			return true;
		}

		// preapre sql statement
		$sql = epObj2Sql :: sqlCreate($this, $cm);
		if (!$sql) {
			return false;
		}

		// execute sql
		return ($r = $this->_execute($sql));
		//		return true;
	}

	/**
     * Drop a table specified in class map if exists
     * @param epClassMap $cm The class map 
     * @return bool
     */
    public function drop($cm) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            // if so, no need to actually create 
            return true;
        }

        // preapre sql statement
        $sql = epObj2Sql::sqlDrop($this, $cm);
        if (!$sql) {
            return false;
        }

        $this->db->clearTableExists($cm->getTable());

        // execute sql
        // if the table doesn't exist, it will throw an
        // exception which is ok
        try {
            $result = ($r = $this->_execute($sql));
        } catch (Exception $e) {
            return true;
        }

        return $result;
    }
    
    /**
     * Create indexes and uniques specified in class map 
     * @param epClassMap $cm The class map 
     * @param bool Whether to force to create table or not
     * @return bool
     */
    public function index($cm, $create = false) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            // if so, no need to actually create 
            return true;
        }
        
        // tabe not exists?
        if (!$this->_tableExists($cm->getTable())) {
            
            // done if -not- forced to create
            if (!$force) {
                return true;
            }

            // create (includes index creation. done.)
            return $this->create($cm, true);
        }

        // check if index exists already
        if (!($curIndexes = $this->checkIndex($cm))) {
            return false;
        }

        // preapre sql statement for creating index 
        $sqls = epObj2Sql::sqlCreateIndex($this, $cm, $curIndexes);

        if (!$sqls) {
            return false;
        }

        // execute sql
        return ($r = $this->_execute($sqls));
    }
    
    /**
     * Retrieves the current indexes in a given table
     * @param epClassMap $cm The class map 
     * @return bool
     */
    protected function checkIndex($cm) {
        
        // get the portable
        if (!($dbp = & epObj2Sql::getPortable($this->dbType()))) {
            return false;
        }

        // call portable to check index
        return $dbp->checkIndex($cm, $this->db);
    }

    /**
	 * Empty a table specified in class map
	 * @param epClassMap
	 * @return bool
	 */
	public function truncate($cm) {

        // check if class is abstract
        if ($cm->isAbstract()) {
            throw new epExceptionDbObject('Class [' . $cm->getName() . '] is abstract');
            return false;
        }
        
		// preapre sql statement
		$sql = epObj2Sql :: sqlTruncate($this, $cm);
		if (!$sql) {
			return false;
		}

		// execute sql
		return ($r = $this->_execute($sql));
	}

	/**
     * Updates relationships for a relationship var
     * @param epClassMap $cm the class map for epObjectRelationship
     * @param string $base_a name of base class a
     * @param string $class_a name of class a
     * @param integer $oid_a object id of the class a object
     * @param integer $var_a the relational field of object a
     * @param string $base_b name of base b
     * @param array $oids_b oids of the class b object related to the class a object
     * @return bool
     */
    public function updateRelationship($cm, $class_a, $var_a, $oid_a, $base_b, $oids_b) {
        
        // make sure the table is created
        if (!$this->create($cm, false)) {
            return false;
        }

        // make sql for relationship update
        $sql = epObj2Sql::sqlUpdateRelationship($this, $cm->getTable(), $class_a, $var_a, $oid_a, $base_b, $oids_b);
        if (!$sql) {
            return false;
        }
        
        // execute sql
        $this->_execute($sql);

        return true;
    }
    
    /**
     * Deletes relationships for an object or a class in relationship
     * @param epClassMap $cm the class map for epObjectRelationship
     * @param string $base_a name of base class a
     * @param integer $oid
     */
    public function deleteRelationship($cm, $class, $oid = null) {
        
        // Do not use relationship table
        if (!$this->_tableExists($cm->getTable())) {
            return true;
        }
        
        // make sure the table is created
        if (!$this->create($cm, false)) {
            return false;
        }

        // make sql for relationship update
        $sql = epObj2Sql::sqlDeleteRelationship($this, $cm->getTable(), $class, $oid);
        if (!$sql) {
            return false;
        }
        
        // execute sql
        $this->_execute($sql);
        
        return true;
    }

    /**
	 * Returns the last insert id
	 * @param string $oid the oid column
	 * @return integer
	 * @access public
	 */
	public function lastInsertId($oid = 'oid') {
		return $this->db->lastInsertId($this->table_last_inserted, $oid);
	}
	
	/**
	 * directly quote by type
	 */
	 public function quoteByTypeValue($v, $type = null){
	 	if($type){
	 		switch ($type) {
	 			case epFieldMap :: DT_BOOL :
				case epFieldMap :: DT_BOOLEAN :
				case epFieldMap :: DT_BIT :
					$v = $v ? 1 : 0;

				case epFieldMap :: DT_INT :
				case epFieldMap :: DT_INTEGER :
					// date, time, datetime treated as integer
					return (integer) $v;
					
				case epFieldMap :: DT_DATE :
				case epFieldMap :: DT_TIME :
				case epFieldMap :: DT_DATETIME :
//					return (integer) $v;
					return $this->db->getDateString($v);

				case epFieldMap :: DT_DECIMAL :
				case epFieldMap :: DT_FLOAT :
				case epFieldMap :: DT_REAL :
					$v = (float) $v;

				case epFieldMap :: DT_BLOB :
				case epFieldMap :: DT_TEXT :
					$v = epStr2Hex($v);
					break;
	 		}
	 	}
	 	return $this->db->quote($v);
	 }
	 
	/**
	 * Formats input so it can be safely used as a literal
	 * Wraps around {@link epDb::quote()}
	 * @param mixed $v
	 * @param epFieldMap 
	 * @return mixed
	 */
	public function quote($v, $fm = null) {
			// special treatment for blob
		if ($fm) {

			switch ($fm->getType()) {
				case epFieldMap :: DT_BOOL :
				case epFieldMap :: DT_BOOLEAN :
				case epFieldMap :: DT_BIT :
					$v = $v ? 1 : 0;

				case epFieldMap :: DT_INT :
				case epFieldMap :: DT_INTEGER :
					// date, time, datetime treated as integer
					return (integer) $v;
					
				
				case epFieldMap :: DT_TIME :
					return $this->db->getTimeString($v);
					
				case epFieldMap :: DT_DATE :
					return $this->db->getDateString($v);
					
				case epFieldMap :: DT_DATETIME :
					return $this->db->getDateTimeString($v);

				case epFieldMap::DT_FLOAT:
				case epFieldMap::DT_REAL:
					$v = empty($v) ? '0.0' : $v;
					break;

				case epFieldMap :: DT_BLOB :
				//case epFieldMap::DT_TEXT:
					$v = epStr2Hex($v);
					break;
			}
		}

		return $this->db->quote($v);
	}

	/**
	 * Wraps around {@link epDb::quoteId()}
	 * Formats a string so it can be safely used as an identifier (e.g. table, column names)
	 * @param string $id
	 * @return mixed
	 */
	public function quoteId($id) {
		return $this->db->quoteId($id);
	}

	/**
	 * Set whether to log queries
	 * @param boolean $log_queries
	 * @return boolean
	 */
	public function logQueries($log_queries = true) {
		return $this->db->logQueries($log_queries);
	}

	/**
	 * Returns queries logged
	 * @return array
	 */
	public function getQueries() {
		return $this->db->getQueries();
	}

	/**
     * Calls underlying db to check if table exists. Always returns
     * true if options check_table_exists is set to false
     * @param string $table
     * @return boolean
     */
    protected function _tableExists($table) {

        // if no checking of table existence
        if (!$this->check_table_exists) {
            // always assume table exists
            return true;
        }

        return $this->db->tableExists($table);
    }

    /**
	 * Executes multiple db queries
	 * @param string|array $sql either a single sql statement or an array of statemetns
	 * @return mixed
	 */
	public function _execute($sql) {

		// make sql into an array
		if (!is_array($sql)) {
			$sql = array ($sql);
		}

		// execute sql stmt one by one
		foreach ($sql as $sql_) {
			if($this->ep_m = &epManager::instance()){
				if($this->ep_m->getConfigOption('show_sql')){
					echo $sql_ . '<br>';	
				}
			}
			$r = $this->db->execute($sql_);
		}

		return $r;
	}

	/**
     * Returns the aggregate function result 
     * @return integer|float
     * @throws epExceptionDb
     */
    protected function _rs2aggr($aggr_func) {
        
        // get ready to ready query result
        $this->db->rsRestart();

        // return aggregation 'column'
        $aggr_alt = substr($aggr_func, 0, strpos($aggr_func, '('));
        return $this->db->rsGetCol($aggr_func, $aggr_alt);
    }

	/**
	 * Converts the last record set into epObject object(s) with class map
	 * @param epClassMap $cm the class map for the conversion
	 * @param array (of integers) object ids to be excluded
	 * @return false|array (of epObject)
	 * @throws epExceptionDbObject
	 */
    protected function _rs2obj($cm, $oids_ex = null) {
        
	 // Create object according to discriminator value
	 // Set fk to the object
	 // Modify composition id

        // !!!important!!! with a large db, the list of oid to be excluded
        // $oids_ex can grown really large and can significantly slow down 
        // queries. so it is suppressed in the select statement and moved 
        // to this method to process.

        // get epManager instance and cache it
        if (!$this->ep_m) {
            $this->ep_m = & epManager::instance();
        }

        // get the class name
        $class = $cm->getName();

        // get all mapped vars
        if (!($fms = $cm->getAllFields())) {
            return self::$false;
        }

        // reset counter and return value
        $ret = array();

        // go through reach record
        $okay = $this->db->rsRestart();
        
        while ($okay) {

            // get oid column 
            $oid = $this->db->rsGetCol($cn=$cm->getOidColumn(), $class.'.'.$cn);
            if ($this->isExistOrginalOid($cm)) {
                $oid = $cm->getOidColumn().':'.$this->db->rsGetCol($cm->getOidColumn()).';';
            }

            // exclude it?
            if ($oids_ex && in_array($oid, $oids_ex)) {
                
                // next row
                $okay = $this->db->rsNext();
                
                // exclude it
                continue;
            }

            // According to discriminator value to create object [Added 16 lines below]
            if ($cm->getDiscriminatorValue() && epClassMap::getTopParent($cm) == $cm) {
                $dis_col = $cm->getBaseSuperClassDiscriminatorColumn();
                $dis_col_name = array_keys($dis_col);
                $dis_val = $this->db->rsGetCol($dis_col_name[0]);
                $dis_cm = $this->ep_m->getDiscriminateValueCM($dis_val);
                $class = $dis_cm->getName();
                $fms = $dis_cm->getAllFields();
            } elseif ($cm->getDiscriminatorValue()) {
                $dis_col = $cm->getBaseSuperClassDiscriminatorColumn();
                $dis_col_name = array_keys($dis_col);
                $dis_val = $this->db->rsGetCol($dis_col_name[0]);
                $dis_cm = $this->ep_m->getDiscriminateValueCM($dis_val);
                if ($dis_cm != $cm) {
                    break;
                }
            }

            // call epManager to create an instance (false: no caching; false: no event dispatching)
            if (!($o = & $this->ep_m->_create($class, false, false))) {
                // next row
                $okay = $this->db->rsNext();
                continue;
            }
            
            // go through each field
            foreach($fms as $fname => $fm) {

                // skip non-primivite field
                if (!$fm->isPrimitive()) {
                    // Set the fk to the object [Added 16 lines below]
                    if ($fks = $fm->getFK()) {
                        // if many side contain the fk value of one side, use string tmp store the value.
                        (string) $fk_string = null;

                        foreach ($fks as $fk_col) {
                            $val = $this->db->rsGetCol($fk_col);
                            if (!is_null($val)) {
                                $fk_string .= /*$fk_col . ':' .*/ $val.';';
                            }
                        }

                        if (!is_null($fk_string)) {
                            $fk_string = '('.$fm->getClass().')'.$fk_string;
                            $o->epSet($fm->getName(), $fk_string, true);
                        }
                    }
                    continue;
                }

                // get var value and set to object
                // Get var value by column name
                $val = $this->db->rsGetCol(epUnquote($fm->getColumnName()));
                // $val = $this->db->rsGetCol($cn=$fm->getColumnName(),$class.'.'.$cn);

                // set value to var (true: no dirty flag change)
                $o->epSet($fm->getName(), $this->_castType($val, $fm->getType()), true);

                // Modify the composition id [Added 1 line below]
                $this->_rs2objModifyOid($oid, $fm, $val);
            }

            // Compute Object ID when it is null [Added 18 lines below]
            if (is_null($oid)) {
                $lIdFields = array();
                foreach ($fms as $lFieldMap) {
                    if ($lFieldMap->isID()) {
                        $lIdFields[] = $lFieldMap;
                    }
                }
                $oid = '';
                foreach ($lIdFields as $lIdField) {
                    // Handle primary key column with foreign Key constraint
                    $lFKs = $lIdField->getFK();
                    if (is_array($lFKs) && count($lFKs) > 0) {
                        $oid .= $lIdField->getColumnName().':'.$this->db->rsGetCol($lFKs[0]).';';
                    } else {
                        $oid .= $lIdField->getColumnName().':'.$this->db->rsGetCol($lIdField->getColumnName()).';';
                    }
                }
            }

            // set oid 
            $o->epSetObjectId($oid); 

            // collect return result
            $ret[] = $o;

            // next row
            $okay = $this->db->rsNext();
        }

        return $ret;
	}
	
	protected function addArrayValueToObj($o, $array_fm){
		$array = array();
		$okay = $this->db->rsRestart();
		while($okay){
			
			$index = $this->db->rsGetCol($array_fm->getArrayIndexColumn());
			$value = $this->db->rsGetCol($array_fm->getColumnName());
			// next row
			$array[$index] = $value;
			$okay = $this->db->rsNext();
		}
		$o->epSet($array_fm->getName(), $array, true);	
	}
	
	protected function addPartialTableValueToObj($o, $cm, $table){
		$p_fms = $cm->getPartialTableField($table);
		$okay = $this->db->rsRestart();
		while($okay){
			foreach($p_fms as $fm){
				$value = $this->db->rsGetCol($fm->getColumnName());
				$o->epSet($fm->getName(), $value, true);	
			}
			$okay = $this->db->rsNext();	
		}
	}

	protected function isExistOrginalOid($cm) {
		$fms = $cm->getAllFields();
		foreach ($fms as $fm) {
			if ($fm->isID()) {
				return false;
			}
		}
		return true;
	}

	protected function _rs2objModifyOid(& $oid, $fm, $val) {
		if ($fm->isID()) {
			$oid .= $fm->getName().':'.$val.';';
		}
	}
	/**
     * Converts the last record set into uoids
     * @param epClassMap $cm the class map for the conversion
     * @param array (of integers) object ids to be excluded
     * @return false|array (of uoids)
     * @throws epExceptionDbObject
     */
    protected function _rs2uoid($cm, $oids_ex = null) {
        
        // !!!important!!! with a large db, the list of oid to be excluded
        // $oids_ex can grown really large and can significantly slow down 
        // queries. so it is suppressed in the select statement and moved 
        // to this method to process.

        // get the class name
        $class = $cm->getName();

        // reset counter and return value
        $ret = array();

        // go through reach record
        $okay = $this->db->rsRestart();
        
        while ($okay) {

            // get oid column 
            $oid = $this->db->rsGetCol($cn=$cm->getOidColumn(), $class.'.'.$cn);

            // exclude it?
            if ($oids_ex && in_array($oid, $oids_ex)) {
                
                // next row
                $okay = $this->db->rsNext();
                
                // exclude it
                continue;
            }

            // get class_b
            $class_b = $this->db->rsGetCol('class_b',$class.'.'.'class_b');

            // get oid_b
            $oid_b = $this->db->rsGetCol('oid_b',$class.'.'.'oid_b');

            // collect return result
            $ret[] = $class_b . ':' . $oid_b;

            // next row
            $okay = $this->db->rsNext();
        }

        return $ret;
    }

    /**
	 * Cast type according to field type
	 * @param mixed $val 
	 * @param string $ftype
	 * @return mixed (casted value)
	 * @access protected
	 */
	protected function _castType(& $val, $ftype) {

		if (is_null($val)) {
			return $val;
		}
		switch ($ftype) {

			case epFieldMap :: DT_BOOL :
			case epFieldMap :: DT_BOOLEAN :
			case epFieldMap :: DT_BIT :
				$val = (boolean) $val;
				break;

			case epFieldMap :: DT_DECIMAL :
			case epFieldMap :: DT_CHAR :
			case epFieldMap :: DT_CLOB :
				$val = (string) $val;
				break;

			case epFieldMap :: DT_BLOB :
			//case epFieldMap :: DT_TEXT :
				$val = (string) epHex2Str($val);
				break;

			case epFieldMap :: DT_INT :
			case epFieldMap :: DT_INTEGER :
				$val = (integer) $val;
				break;

			case epFieldMap :: DT_FLOAT :
			case epFieldMap :: DT_REAL :
				$val = (float) $val;
				break;

			case epFieldMap :: DT_DATE :
			case epFieldMap :: DT_TIME :
			case epFieldMap :: DT_DATETIME :
//				$val = (integer) $val;
				$val = (integer)strtotime($val);
				break;
		}

		return $val;
	}
		}

/**
 * Exception class for {@link epDbFactory}
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.db 
 */
class epExceptionDbFactory extends epException {
}

/**
 * Class of database connection factory
 * 
 * The factory creates databases with given DSNs and maintains
 * a one(DSN)-to-one(epDbObject isntance) mapping.
 * 
 * @author Oak Nauhygon <ezpdo4php@gmail.com>
 * @version $Revision$ $Date: 2007-01-19 05:38:55 -0500 (Fri, 19 Jan 2007) $
 * @package ezpdo
 * @subpackage ezpdo.db
 */
class epDbFactory implements epFactory, epSingleton {

	/**#@+
	 * Consts for DB abstraction layer libs
	 */
	const DBL_ADODB = "adodb";
	const DBL_ADODB_PDO = "adodb_pdo";
	const DBL_PEARDB = "peardb";
	const DBL_PDO = "pdo";
	/**#@-*/

    /**#@+
     * Used for return value to avoid reference notice in 5.0.x and up
     * @var bool
     */
    static public $false = false;
    static public $true = true;
    static public $null = null;
    /**#@-*/

	/**
	 * The array of DBALs supported
	 * @var array
	 */
	static public $dbls_supported = array (self :: DBL_ADODB, self :: DBL_ADODB_PDO, self :: DBL_PEARDB, self :: DBL_PDO,);

	/**
	 * The current DB abstraction lib in use
	 */
	private $dbl = epDbFactory :: DBL_ADODB;

	/**
	 * db connections created
	 * @var array
	 */
	private $dbs = array ();

	/**
	 * Constructor
	 */
	private function __construct() {
	}

	/**
	 * Get the current DBA (DB abstraction lib)
	 * @return string
	 */
	function getDbLib() {
		return $this->dbl;
	}

	/**
	 * Set the current DBA (DB abstraction lib)
	 * @param string self::DBL_ADODB|self::DBL_PEARDB
	 * @return void
	 */
	function setDbLib($dbl) {

		// lower case dbl name
		$dbl = strtolower($dbl);

		// is dbl supported?
		if (!in_array($dbl, self :: $dbls_supported)) {
			throw new epExceptionDbFactory('Db library ['.$dbl.'] unsupported.');
		}

		// set the current dbl
		$this->dbl = $dbl;
	}

	/**
	 * Implements factory method {@link epFactory::make()}
	 * @param string $dsn
	 * @return epDbObject|null
	 * @access public
	 * @static
	 */
	public function & make($dsn) {
		return $this->get($dsn, false); // false: no tracking
	}

	/**
	 * Implement factory method {@link epFactory::track()}
	 * @param string $dsn
	 * @return epDbObject
	 * @access public
	 */
	public function & track() {
		$args = func_get_args();
		return $this->get($args[0], true); // true: tracking
	}

	/**
	 * Either create a class map (if not tracking) or retrieve it from cache 
	 * @param $dsn
	 * @param bool tracking or not
     * @return null|epDbObject
	 * @throws epExceptionDbFactory
	 */
	private function & get($dsn, $tracking = false) {

			// check if dsn is empty 
	if (empty ($dsn)) {
			throw new epExceptionDbFactory('DSN is empty');
            return self::$null;
		}

		// check if class map has been created
		if (isset ($this->dbs[$dsn])) {
			return $this->dbs[$dsn];
		}

		// check if it's in tracking mode
		if ($tracking) {
            return self::$null;
		}

		// otherwise create
		switch ($this->dbl) {

			case self :: DBL_ADODB :
				include_once (EP_SRC_DB.'/epDbAdodb.php');
				$this->dbs[$dsn] = new epDbObject(new epDbAdodb($dsn));
				break;

			case self :: DBL_ADODB_PDO :
				include_once (EP_SRC_DB.'/epDbAdodbPdo.php');
				$this->dbs[$dsn] = new epDbObject(new epDbAdodbPdo($dsn));
				break;

			case self :: DBL_PEARDB :
				include_once (EP_SRC_DB.'/epDbPeardb.php');
				$this->dbs[$dsn] = new epDbObject(new epDbPeardb($dsn));
				break;

			case self :: DBL_PDO :
				include_once (EP_SRC_DB.'/epDbPdo.php');
				$this->dbs[$dsn] = new epDbObject(new epDbPdo($dsn));
				break;
		}

		return $this->dbs[$dsn];
	}

	/**
	 * Implement factory method {@link epFactory::allMade()}
	 * Return all db connections made by factory
	 * @return array
	 * @access public
	 */
    public function allMade() {
		return array_values($this->dbs);
	}

	/**
	 * Implement factory method {@link epFactory::removeAll()}
	 * Remove all db connections made 
	 * @return void
	 */
	public function removeAll() {

		// close all db connections
		if ($this->dbs) {
			foreach ($this->dbs as $db) {
				$db->connection()->close();
			}
		}

		// wipe out all db connections
        $this->dbs = array();
	}

	/**
	 * Implements {@link epSingleton} interface
	 * @return epDbFactory
	 * @access public
	 */
	static public function & instance() {
		if (!isset (self :: $instance)) {
			self :: $instance = new self;
		}
		return self :: $instance;
	}

	/**
	 * Implement {@link epSingleton} interface
	 * Forcefully destroy old instance (only used for tests). 
	 * After reset(), {@link instance()} returns a new instance.
	 */
	static public function destroy() {
        if (self::$instance) {
            self::$instance->removeAll();
        }
		self :: $instance = null;
	}

	/**
	 * epDbFactory instance
	 */
	static private $instance;
}
?>
