/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by Brian Duff are Copyright
	(C)Brian Duff. All Rights Reserved.

	Contributor(s): Gerhard Leonhartsberger.
*/

package net.java.plaf.windows.xp;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.swing.AbstractAction;
import javax.swing.ActionMap;
import javax.swing.JComponent;
import javax.swing.JPopupMenu;
import javax.swing.plaf.ComponentUI;

import net.java.plaf.ClientProperties;

import com.sun.java.swing.plaf.windows.WindowsScrollBarUI;

/**
 * Patches scrollbars on Windows XP so that they have a context menu containing
 * the following menu items:
 * <pre>
 *   Scroll Here
 *   ---------------------
 *   Top / Left Edge
 *   Bottom / Right Edge
 *   ---------------------
 *   Page Up / Page Left
 *   Page Down / Page Right
 *   ----------------------
 *   Scroll Up / Scroll Left
 *   Scroll Down / Scroll Right
 * </pre>
 * You can prevent a specific scrollbar instance from displaying this menu
 * by setting the client property <code>NO_SCROLLBAR_POPUP</code> to 
 * <code>true</code>. e.g.
 * <pre>
 *      JScrollBar sb = new JScrollBar();
 *      sb.putClientProperty( net.java.plaf.ClientProperties.NO_SCROLLBAR_POPUP, 
 *                            Boolean.TRUE );
 *  </pre>
 * 
 * @author Brian.Duff@oracle.com
 */
public class XPScrollBarUI extends WindowsScrollBarUI {

	private ContextMenuListener contextMenuListener;
	private ActionMap actionMap;
	private JPopupMenu contextMenu;

	// BDUFF: need to check if this is really XP specific, or whether it exists
	// in 2000. Definitely not in NT4.
	public static ComponentUI createUI(JComponent comp) {
		return new XPScrollBarUI();
	}

	protected void createActionMap() {

		actionMap = new ActionMap();
		actionMap.put(
			PageDecreaseAction.ACTION_COMMAND,
			new PageDecreaseAction());
		actionMap.put(
			PageIncreaseAction.ACTION_COMMAND,
			new PageIncreaseAction());
		actionMap.put(
			ScrollIncreaseAction.ACTION_COMMAND,
			new ScrollIncreaseAction());
		actionMap.put(
			ScrollDecreaseAction.ACTION_COMMAND,
			new ScrollDecreaseAction());
		actionMap.put(ScrollHereAction.ACTION_COMMAND, new ScrollHereAction());
		actionMap.put(
			ScrollToEndAction.ACTION_COMMAND,
			new ScrollToEndAction());
		actionMap.put(
			ScrollToStartAction.ACTION_COMMAND,
			new ScrollToStartAction());
	}

	/* (non-Javadoc)
	 * @see javax.swing.plaf.ComponentUI#installUI(javax.swing.JComponent)
	 */
	public void installUI(JComponent c) {
		super.installUI(c);
		createActionMap();
	}

	protected void installListeners() {

		super.installListeners();

		contextMenuListener = new ContextMenuListener();
		scrollbar.addMouseListener(contextMenuListener);

		incrButton.addMouseListener(contextMenuListener);
		decrButton.addMouseListener(contextMenuListener);
	}

	protected void uninstallListeners() {

		super.uninstallListeners();
		scrollbar.removeMouseListener(contextMenuListener);
		incrButton.removeMouseListener(contextMenuListener);
		decrButton.removeMouseListener(contextMenuListener);
	}

	protected void fillContextMenu(JPopupMenu contextMenu) {

		contextMenu.add(actionMap.get(ScrollHereAction.ACTION_COMMAND));
		contextMenu.addSeparator();
		contextMenu.add(actionMap.get(ScrollToStartAction.ACTION_COMMAND));
		contextMenu.add(actionMap.get(ScrollToEndAction.ACTION_COMMAND));
		contextMenu.addSeparator();
		contextMenu.add(actionMap.get(PageDecreaseAction.ACTION_COMMAND));
		contextMenu.add(actionMap.get(PageIncreaseAction.ACTION_COMMAND));
		contextMenu.addSeparator();
		contextMenu.add(actionMap.get(ScrollDecreaseAction.ACTION_COMMAND));
		contextMenu.add(actionMap.get(ScrollIncreaseAction.ACTION_COMMAND));
	}

	private void showContextMenu(MouseEvent event) {

		// lazy creation
		if (contextMenu == null) {
			contextMenu = new JPopupMenu();
		}
		else {
			contextMenu.removeAll();
		}
        fillContextMenu(contextMenu);

		initScrollHereAction(event);

		// needed in order to determine the invoker component.
		// If the mouse event was fireded by a non awt component than
		// the invoker is the scrollbar. 
		Component invokerComponent =
			event.getSource() instanceof Component
				? (Component) event.getSource()
				: scrollbar;

		contextMenu.show(invokerComponent, event.getX(), event.getY());
	}

	private void initScrollHereAction(MouseEvent event) {

		// initialize scrollHereAction so that here action "knows" where 
		// "here" is.
		// We need to decide if the user right-clicked one of the scrollbar arrow
		// buttons or on the scrollbar.
		ScrollHereAction scrollHereAction =
			(ScrollHereAction) actionMap.get(ScrollHereAction.ACTION_COMMAND);
		int herePosition =
			scrollbar.getOrientation() == HORIZONTAL
				? event.getX()
				: event.getY();
		if (event.getSource().equals(incrButton)) {
			herePosition = scrollbar.getMaximum();
		}
		else
			if (event.getSource().equals(decrButton)) {
				herePosition = scrollbar.getMinimum();
			}
		scrollHereAction.setHerePosition(herePosition);
	}

	private boolean isPopupEnabled() {
		Boolean value =
			(Boolean) scrollbar.getClientProperty(
				ClientProperties.NO_SCROLLBAR_POPUP);
		if (value != null)
			return !value.booleanValue();

		return true;
	}

	private class ContextMenuListener extends MouseAdapter {

		public void mouseReleased(MouseEvent me) {

			if (me.isPopupTrigger()
				&& scrollbar.isEnabled()
				&& isPopupEnabled()) {

				showContextMenu(me);
			}
		}
	}

	private class ScrollHereAction extends AbstractAction {
		private int herePosition;

		public static final String ACTION_COMMAND = "scrollHereAction"; //$NON-NLS-1$

		public ScrollHereAction() {
			super(ACTION_COMMAND);
		}

		public void setHerePosition(int herePosition) {
			this.herePosition = herePosition;
		}

		public void actionPerformed(ActionEvent event) {

			int decrSize =
				scrollbar.getOrientation() == HORIZONTAL
					? decrButton.getWidth()
					: decrButton.getHeight();
			int incrSize =
				scrollbar.getOrientation() == HORIZONTAL
					? incrButton.getWidth()
					: incrButton.getHeight();

			int scrollbarSize =
				scrollbar.getOrientation() == HORIZONTAL
					? scrollbar.getWidth()
					: scrollbar.getHeight();

			double perc =
				(double) (herePosition - decrSize)
					/ (double) (scrollbarSize - incrSize - decrSize);
			int newPos =
				(int) (perc
					* (scrollbar.getMaximum() - scrollbar.getMinimum()));

			if (newPos < scrollbar.getMinimum())
				newPos = scrollbar.getMinimum();
			if (newPos > scrollbar.getMaximum())
				newPos = scrollbar.getMaximum();
			scrollbar.setValue(newPos);
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				return Messages.getText(MessageConstants.SCROLL_HERE);
			}
			return super.getValue(key);
		}
	}

	private class ScrollToStartAction extends AbstractAction {

		public static final String ACTION_COMMAND = "scrollToStartAction"; //$NON-NLS-1$

		public ScrollToStartAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			scrollbar.setValue(scrollbar.getMinimum());
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.LEFT_EDGE);
				}
				else {
					return Messages.getText(MessageConstants.TOP);
				}
			}
			return super.getValue(key);
		}
	}

	private class ScrollToEndAction extends AbstractAction {

		public static final String ACTION_COMMAND = "scrollToEndAction"; //$NON-NLS-1$

		public ScrollToEndAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			scrollbar.setValue(scrollbar.getMaximum());
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.RIGHT_EDGE);
				}
				else {
					return Messages.getText(MessageConstants.BOTTOM);
				}
			}
			return super.getValue(key);
		}
	}

	private class PageDecreaseAction extends AbstractAction {

		public static final String ACTION_COMMAND = "pageDecreaseAction"; //$NON-NLS-1$

		public PageDecreaseAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			int blockIncrement = scrollbar.getBlockIncrement(-1);
			int newValue = scrollbar.getValue() - blockIncrement;
			scrollbar.setValue(
				newValue < scrollbar.getMinimum()
					? scrollbar.getMinimum()
					: newValue);
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.PAGE_LEFT);
				}
				else {
					return Messages.getText(MessageConstants.PAGE_UP);
				}
			}
			return super.getValue(key);
		}
	}

	private class PageIncreaseAction extends AbstractAction {

		public static final String ACTION_COMMAND = "pageIncreaseAction"; //$NON-NLS-1$

		public PageIncreaseAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			int blockIncrement = scrollbar.getBlockIncrement(1);
			int newValue = scrollbar.getValue() + blockIncrement;
			scrollbar.setValue(
				newValue > scrollbar.getMaximum()
					? scrollbar.getMaximum()
					: newValue);
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.PAGE_RIGHT);
				}
				else {
					return Messages.getText(MessageConstants.PAGE_DOWN);
				}
			}
			return super.getValue(key);
		}
	}

	private class ScrollDecreaseAction extends AbstractAction {

		public static final String ACTION_COMMAND = "scrollDecreaseAction"; //$NON-NLS-1$

		public ScrollDecreaseAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			int increment = scrollbar.getUnitIncrement(-1);
			int newValue = scrollbar.getValue() - increment;
			scrollbar.setValue(
				newValue < scrollbar.getMinimum()
					? scrollbar.getMinimum()
					: newValue);
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.SCROLL_LEFT);
				}
				else {
					return Messages.getText(MessageConstants.SCROLL_UP);
				}
			}
			return super.getValue(key);
		}
	}

	private class ScrollIncreaseAction extends AbstractAction {

		public static final String ACTION_COMMAND = "scrollIncreaseAction"; //$NON-NLS-1$

		public ScrollIncreaseAction() {
			super(ACTION_COMMAND);
		}

		public void actionPerformed(ActionEvent ae) {
			int increment = scrollbar.getUnitIncrement(1);
			int newValue = scrollbar.getValue() + increment;
			scrollbar.setValue(
				newValue > scrollbar.getMaximum()
					? scrollbar.getMaximum()
					: newValue);
		}

		public Object getValue(String key) {
			if (NAME.equals(key)) {
				if (scrollbar.getOrientation() == HORIZONTAL) {
					return Messages.getText(MessageConstants.SCROLL_RIGHT);
				}
				else {
					return Messages.getText(MessageConstants.SCROLL_DOWN);
				}
			}
			return super.getValue(key);
		}
	}

	private static class Messages {

		private static ResourceBundle resource = null;

		// private constructor in order to avoid instantiation.
		private Messages() {
		}

		public static String getText(String key) {
			try {
				StringBuffer sb = new StringBuffer("XPScrollBarUI."); //$NON-NLS-1$
				sb.append(key);
				return getResourceBundle().getString(sb.toString());
			}
			catch (MissingResourceException e) {
				return "!" + key + "!"; //$NON-NLS-1$//$NON-NLS-2$
			}
		}

		private static ResourceBundle getResourceBundle() {

			if (resource == null) {
				resource = ResourceBundle.getBundle("net.java.plaf.windows.xp.XPResources"); //$NON-NLS-1$
			}

			return resource;
		}
	}

	private interface MessageConstants {
		public final static String SCROLL_HERE = "ScrollHere"; //$NON-NLS-1$
		public final static String SCROLL_LEFT = "ScrollLeft"; //$NON-NLS-1$
		public final static String SCROLL_RIGHT = "ScrollRight"; //$NON-NLS-1$
		public final static String SCROLL_DOWN = "ScrollDown"; //$NON-NLS-1$
		public final static String SCROLL_UP = "ScrollUp"; //$NON-NLS-1$
		public final static String LEFT_EDGE = "LeftEdge"; //$NON-NLS-1$
		public final static String RIGHT_EDGE = "RightEdge"; //$NON-NLS-1$
		public final static String TOP = "Top"; //$NON-NLS-1$
		public final static String BOTTOM = "Bottom"; //$NON-NLS-1$
		public final static String PAGE_LEFT = "PageLeft"; //$NON-NLS-1$
		public final static String PAGE_UP = "PageUp"; //$NON-NLS-1$
		public final static String PAGE_RIGHT = "PageRight"; //$NON-NLS-1$
		public final static String PAGE_DOWN = "PageDown"; //$NON-NLS-1$
	}
}