/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by Brian Duff are Copyright
	(C)Brian Duff. All Rights Reserved.

	Contributor(s): Gerhard Leonhartsberger (GHL)
*/

package net.java.plaf.windows.xp;

import javax.swing.BorderFactory;
import javax.swing.UIDefaults;
import javax.swing.UIManager;
import javax.swing.border.Border;
import javax.swing.plaf.InsetsUIResource;

import net.java.plaf.Environment;
import net.java.plaf.LookAndFeelPatch;

/**
 * Menu popups under the Windows look and feel on Windows XP do not have the
 * correct appearance. They ought to have a "flat" style border.
 * This patch fixes the following winlaf issues:
 * <ul>
 * <li><b>Issue 4:</b> Spacing and appearance of menus does not match 
 *    Windows XP</li>
 * <li><b>Issue 26:</b> Menus on a menubar contained by a JFrame are not drawn 
 *                      in "de-activated" style when JFrame is deactivated.
 * </ul>
 * 
 * @author Brian.Duff@oracle.com
 * @author Gerhard.Leonhartsberger@scch.at
 */
public class XPMenuPatch implements LookAndFeelPatch {
	public boolean isApplicable(Environment env) {
		return env.isWindowsLookAndFeel() && env.isWindowsXPThemed();
	}

	public void patch(Environment env) {
		UIDefaults defaults = UIManager.getDefaults();

		// Remove the 3d border, replace it with a flat one that has more spacing.
		defaults.put("PopupMenu.border", createPopupMenuBorder()); //$NON-NLS-1$

		// Patch the separator so that it's a single line rather than an etched
		// 3D effect one. We patch both SeparatorUI (which corresponds to 
		// JSeparator(), and PopupMenuSeparatorUI (which corresponds to the 
		// package protected subclass of JSeparator that is used when you do
		// JMenu.addSeparator()).
		defaults.put("SeparatorUI", "net.java.plaf.windows.xp.XPSeparatorUI"); //$NON-NLS-1$//$NON-NLS-2$
		defaults.put("PopupMenuSeparatorUI", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPPopupMenuSeparatorUI"); //$NON-NLS-1$

		// Patch the menu in order to paint the correct forground color
        // when active window is changed.
        // DisabledForeground color is not set by WindowsLookAndFeel class.
        defaults.put("Menu.disabledForeground", defaults.get("MenuItem.disabledForeground")); //$NON-NLS-1$ //$NON-NLS-2$
		defaults.put("MenuUI", "net.java.plaf.windows.xp.XPMenuUI");  //$NON-NLS-1$//$NON-NLS-2$

		// Fix the margin on menu items. Menu items are 2 pixels higher than
		// their native counterparts by default.
		// bduff: check this on < 1.4.2.
		defaults.put("MenuItem.margin", new InsetsUIResource(1, 2, 1, 2)); //$NON-NLS-1$
		defaults.put("Menu.margin", new InsetsUIResource(1, 2, 1, 2)); //$NON-NLS-1$
		defaults.put("CheckBoxMenuItem.margin", //$NON-NLS-1$
		new InsetsUIResource(1, 2, 1, 2));
		defaults.put("RadioButtonMenuItem.margin", //$NON-NLS-1$
		new InsetsUIResource(1, 4, 1, 2));

		// Wouldn't it be cool if we could add a drop shadow now? :P
	}

	private Border createPopupMenuBorder() {
		return BorderFactory.createCompoundBorder(BorderFactory.createMatteBorder(1, 1, 1, 1, UIManager.getColor("controlShadow")), //$NON-NLS-1$
		BorderFactory.createEmptyBorder(2, 2, 2, 2));
	}

	public void unpatch() {

	}

}