/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Gerhard Leonhartsberger. Portions created by 
	Gerhard Leonhartsberger are Copyright(C) Gerhard Leonhartsberger. 
	All Rights Reserved.

	Contributor(s): Gerhard Leonhartsberger.
*/
package net.java.plaf.windows.xp;

import java.awt.Color;
import java.awt.Component;
import java.awt.KeyboardFocusManager;
import java.awt.Window;

import javax.swing.Icon;
import javax.swing.JList;
import javax.swing.UIManager;

/**
 * Class <code>DefaultListCellRenderer</code> is reimplemented because the
 * former <code>javax.swing.list.DefaultListCellRenderer</code> does not
 * consider focus permantently lost selection state.
 * <p>
 * This class fixes issue: 
 * <ul>
 * <li><b>Issue 28:</b> JList does not update selection when loosing focus to 
 * 						non-focus selection color.</li>
 * <li><b>Issue 31:</b> A selected list item does not change selection color 
 *                      when the JFrame where the list is added looses 
 *                      focus.</li>
 * </ul>
 */
public class DefaultListCellRenderer
	extends javax.swing.DefaultListCellRenderer {

	private JList list;
	private boolean cellHasFocus;

	/* (non-Javadoc)
	 * @see javax.swing.ListCellRenderer#getListCellRendererComponent(javax.swing.JList, java.lang.Object, int, boolean, boolean)
	 */
	public Component getListCellRendererComponent(
		JList list,
		Object value,
		int index,
		boolean isSelected,
		boolean cellHasFocus) {

		this.list = list;
		this.cellHasFocus = cellHasFocus;

		setComponentOrientation(list.getComponentOrientation());
		if (isSelected && cellHasFocus) {
			setBackground(list.getSelectionBackground());
			setForeground(list.getSelectionForeground());
		}
		else
			// We need to consider if the focus is lost permanently or just
			// temporarily.
			// if permanent lost we need to change the selection color.
			// Issue 28: JList does not update selection when loosing focus to 
			//           non-focus selection color.
			if (isFocusPermanentlyLost() && isSelected) {
				Color focusLostSelectionColor = UIManager.getColor("Label.background"); //$NON-NLS-1$
				setBackground(focusLostSelectionColor);
				setForeground(list.getForeground());
			}
			else {
				setBackground(list.getBackground());
				setForeground(list.getForeground());
			}

		if (value instanceof Icon) {
			setIcon((Icon) value);
			setText(""); //$NON-NLS-1$
		}
		else {
			setIcon(null);
			setText((value == null) ? "" : value.toString()); //$NON-NLS-1$
		}

		setEnabled(list.isEnabled());
		setFont(list.getFont());
		setBorder((cellHasFocus) ? UIManager.getBorder("List.focusCellHighlightBorder") //$NON-NLS-1$
		: noFocusBorder);

		return this;
	}

	/**
	 * Returns if the focus for this component is permanently lost or 
	 * temporarily lost.
	 * 
	 * @return <code>true</code> if the focus for this component is permanently
	 * 		   lost, else <code>false</code>. 
	 */
	protected boolean isFocusPermanentlyLost() {

		KeyboardFocusManager kfm =
			KeyboardFocusManager.getCurrentKeyboardFocusManager();

		Component component = kfm.getPermanentFocusOwner();
		if (component == null) {
			return hasCellFocus();
		}

        // Issue 31:
		Window window = windowForComponent(getList());
        Window activeWindow = kfm.getActiveWindow();
        if (activeWindow == null) {
            // The active window is not a window or dialog of the calling 
            // thread's context. 
            return true;
        }
		return !component.equals(getList())
			|| !activeWindow.equals(window);
	}

	protected Window windowForComponent(Component aComponent) {

		if (aComponent == null) {
			return null;
		}
		Component parent = aComponent.getParent();
		Window window = null;
		boolean found = false;
		while ((parent != null) && (!found)) {
			if (parent instanceof Window) {
				window = (Window) parent;
				found = true;
			}
			else {
				parent = parent.getParent();
			}
		}

		return window;
	}

	protected JList getList() {
		return list;
	}

	protected boolean hasCellFocus() {
		return cellHasFocus;
	}
}
