/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by 
	Brian Duff are Copyright(C) Brian Duff. 
	All Rights Reserved.

	Contributor(s): ___________________________.
*/

package net.java.plaf.util;

import java.awt.Component;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Toolkit;
import java.awt.image.ImageObserver;

import java.util.MissingResourceException;
import java.util.ResourceBundle;

import javax.accessibility.AccessibleContext;

import javax.swing.ImageIcon;

/**
 * An icon set is an ImageIcon that delegates to another ImageIcon instance. 
 * The specific icon used can differ based on the DPI and color depth of the
 * current display. This provides something similar to the multiple icon support
 * of Windows .ico files for Java.
 * 
 * A further benefit of this class is that icon loading is as lazy as possible.
 * Icon loading can constitute a significant portion of the startup time of a
 * swing application; making image loading lazy can lead to significant
 * startup performance gains.
 * 
 * The implementation uses ResourceBundles. You specify a base key and a 
 * bundle. When required, the implementation will look up the correct icon
 * using the following keys, stopping when one is found:
 * 
 * baseKey.depth.dpi
 * baseKey.depth
 * baseKey
 * 
 * e.g. on a 96 dpi 32 bit color system*, with a baseKey of myicon:
 * 
 * myicon.24.96
 * myicon.24
 * myicon
 * 
 * The value of the key should be a relative resource URL to load the 
 * icon from on the classpath.
 * 
 * *The value for the color depth used is the return value of 
 * Toolkit.getDefaultToolkit().getColorModel().getPixelSize(). On a 32-bit
 * color system, this returns 24.
 * 
 * @author Brian.Duff@oracle.com
 */
public class FlexibleImageIcon extends ImageIcon
{
  // TODO: can we listen for DPI / color depth changes?

  private ImageIcon m_delegateIcon;
  private String m_baseKey;
  private Class m_baseClass;
  private ResourceBundle m_bundle;
  
  public FlexibleImageIcon( ResourceBundle bundle, Class baseClass, String baseKey )
  {
    m_bundle = bundle;
    m_baseKey = baseKey;
    m_baseClass = baseClass;
  }
  
  public static void main( String[] args )
  {
    FlexibleImageIcon fii = new FlexibleImageIcon( 
      ResourceBundle.getBundle( "net.java.plaf.windows.xp.XPResources" ),
      FlexibleImageIcon.class,
      "testkey"
    );
    
    Toolkit.getDefaultToolkit().getDesktopProperty( "BLAH" );
    
    fii.getIconWidth();
  }
  
  private ImageIcon getDelegateIcon()
  {
    if ( m_delegateIcon == null )
    {
      String colorDepth = String.valueOf(
        Toolkit.getDefaultToolkit().getColorModel().getPixelSize()
      );
      
      String dpi = String.valueOf(
        Toolkit.getDefaultToolkit().getScreenResolution()
      );
      
      StringBuffer keyBuffer = new StringBuffer( 
        m_baseKey.length() + colorDepth.length() + dpi.length() 
      );
      
      keyBuffer.append( m_baseKey );
      keyBuffer.append( "." );
      keyBuffer.append( colorDepth );
      keyBuffer.append( "." );
      keyBuffer.append( dpi );
      
      String value;
      try
      {
        value = m_bundle.getString( keyBuffer.toString() );
      }
      catch ( MissingResourceException mre )
      {
        try
        {
          keyBuffer.setLength( m_baseKey.length() + colorDepth.length() + 1 );        
          value = m_bundle.getString( keyBuffer.toString() );
        }
        catch ( MissingResourceException mre1 )
        {
          keyBuffer.setLength( m_baseKey.length() );
          value = m_bundle.getString( keyBuffer.toString() );          
        }
      }
      
      m_delegateIcon = new ImageIcon( m_baseClass.getResource( value ) );
    }
    return m_delegateIcon;
  }
  
  public AccessibleContext getAccessibleContext()
  {
    return getDelegateIcon().getAccessibleContext();
  }
  
  public String getDescription()
  {
    return getDelegateIcon().getDescription();
  }
  
  public int getIconHeight()
  {
    return getDelegateIcon().getIconHeight();
  }
  
  public int getIconWidth()
  {
    return getDelegateIcon().getIconWidth();
  }
  
  public Image getImage()
  {
    return getDelegateIcon().getImage();
  }
  
  public int getImageLoadStatus()
  {
    return getDelegateIcon().getImageLoadStatus();
  }
  
  public ImageObserver getImageObserver()
  {
    return getDelegateIcon().getImageObserver();
  }
  
  public void paintIcon( Component c, Graphics g, int x, int y )
  {
    getDelegateIcon().paintIcon( c, g, x, y );
  }
  
  public void setDescription( String description )
  {
    getDelegateIcon().setDescription( description );
  }
  
  public void setImage( Image i )
  {
    getDelegateIcon().setImage( i );
  }
  
  public void setImageObserver( ImageObserver io )
  {
    getDelegateIcon().setImageObserver( io );
  }
  
  public String toString()
  {
    return getDelegateIcon().toString();
  }
  

}