/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by Brian Duff are Copyright
	(C)Brian Duff. All Rights Reserved.

	Contributor(s): Gerhard Leonhartsberger.
*/

package net.java.plaf.test;

import java.awt.BorderLayout;
import java.awt.Container;
import java.awt.Dimension;
import java.net.URL;

import java.util.ResourceBundle;
import javax.swing.BorderFactory;
import javax.swing.Icon;
import javax.swing.ImageIcon;
import javax.swing.JCheckBoxMenuItem;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;
import javax.swing.JRadioButtonMenuItem;
import javax.swing.JScrollPane;
import javax.swing.JSplitPane;
import javax.swing.JTree;
import javax.swing.SwingConstants;
import javax.swing.UIManager;
import javax.swing.event.EventListenerList;
import javax.swing.event.TreeModelEvent;
import javax.swing.event.TreeModelListener;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreeModel;
import javax.swing.tree.TreePath;

import net.java.plaf.LookAndFeelPatchManager;
import net.java.plaf.util.FlexibleImageIcon;
import net.java.plaf.windows.xp.DefaultTreeCellRenderer;

// In order to add additional test patch modules extend method 
// createTestPatchModules().

public class TestApp extends JFrame {

	private static final String ICON_LOCATION = "net/java/plaf/test/"; //$NON-NLS-1$

	private JTree patchTestTree;
	private PatchTestTreeModel patchTestTreeModel;
	private PatchTestModule rootPatchTestModule;
	private TreeSelectionHandler treeSelectionHandler;

	//
	// Main
	//

	public static void main(String[] args) throws Exception {

		UIManager.setLookAndFeel(UIManager.getSystemLookAndFeelClassName());
		LookAndFeelPatchManager.initialize();

		TestApp testApp = new TestApp("WinLAF Patch Explorer");
		testApp.setSize(700, 500);
		testApp.setLocationRelativeTo(null);
		testApp.setVisible(true);
	}

	public TestApp(String title) {
		super(title);

		setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);

		createUI();
		createListeners();
		installListeners();
	}

	protected void createUI() {

		createContentPane();
		createMenuBar();

		createPatchTestModules();
	}

	protected void createContentPane() {

		JSplitPane contentPane = new JSplitPane();
		contentPane.setOrientation(JSplitPane.HORIZONTAL_SPLIT);
		setContentPane(contentPane);

		JPanel rightPanel = new JPanel();
		rightPanel.setPreferredSize(new Dimension(300, 300));
		contentPane.setRightComponent(rightPanel);
		rightPanel.setLayout(new BorderLayout());
		JLabel label = new JLabel("Select a patch to see a demo.");
		label.setHorizontalAlignment(SwingConstants.CENTER);
		rightPanel.add(label, BorderLayout.CENTER);

		createPatchTestTree(contentPane);
	}

	protected void createPatchTestTree(JSplitPane contentPane) {

		rootPatchTestModule = new PatchTestModule("Swing Patches");
		patchTestTreeModel = new PatchTestTreeModel(rootPatchTestModule);
		patchTestTree = new JTree();
		patchTestTree.setModel(patchTestTreeModel);
		patchTestTree.setRootVisible(true);
		patchTestTree.setShowsRootHandles(true);

		// I include the patch for the tree already here.
		DefaultTreeCellRenderer r = new DefaultTreeCellRenderer();
		r.setLeafIcon(
			new FlexibleImageIcon(
				ResourceBundle.getBundle("net.java.plaf.test.TestResources"),
				getClass(),
				"fixicon"));
		patchTestTree.setCellRenderer(r);

		JPanel patchTestTreePane = new JPanel(new BorderLayout());
		patchTestTreePane.add(patchTestTree, BorderLayout.CENTER);
		JScrollPane scrollPane = new JScrollPane(patchTestTreePane);
		scrollPane.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 0));
		contentPane.setLeftComponent(scrollPane);
	}

	protected void createMenuBar() {

		JMenu m = new JMenu("File");
    
    // Testing for issue 35
    JMenu newMenu = new JMenu( "New" );
    newMenu.add( new JMenuItem( "Issue" ) );
    newMenu.add( new JMenuItem( "Testcase" ) );
    m.add( newMenu );
    
		final JMenuItem open = new JMenuItem("Open");
		m.add(open);
		m.add("Explore");
		m.add("Search");
		m.addSeparator();
		m.add("Sharing and Security...");

		JMenuBar bar = new JMenuBar();
		bar.add(m);

		JMenuItem cut = new JMenuItem("Cut", getIcon("Cut16.gif"));
		JMenuItem copy = new JMenuItem("Copy", getIcon("Copy16.gif"));
		JMenuItem paste = new JMenuItem("Paste", getIcon("Paste16.gif"));
		JMenuItem delete = new JMenuItem("Delete", getIcon("Delete16.gif"));
		JMenuItem selectAll = new JMenuItem("Select All");
		JMenu editMenu = new JMenu("Edit");
		editMenu.add(cut);
		editMenu.add(copy);
		editMenu.add(paste);
		editMenu.addSeparator();
		editMenu.add(delete);
		editMenu.add(selectAll);

		bar.add(editMenu);

		JMenu view = new JMenu("View");
		JMenu toolbars = new JMenu("Toolbars");
		toolbars.add(new JCheckBoxMenuItem("Standard Buttons"));
		toolbars.add(new JCheckBoxMenuItem("Address Bar"));
		toolbars.add(new JCheckBoxMenuItem("Links"));
		toolbars.addSeparator();
		toolbars.add(new JCheckBoxMenuItem("Lock the Toolbars"));
		toolbars.add("Customize...");

		view.add(toolbars);
		view.add(new JCheckBoxMenuItem("Status Bar"));

		view.addSeparator();
		view.add(new JRadioButtonMenuItem("Thumbnails"));
		view.add(new JRadioButtonMenuItem("Tiles"));
		view.add(new JRadioButtonMenuItem("Icons"));
		view.add(new JRadioButtonMenuItem("List"));
		view.add(new JRadioButtonMenuItem("Details"));
		view.addSeparator();

		JMenu arrange = new JMenu("Arrange Icons by");
		arrange.add(new JRadioButtonMenuItem("Name"));
		arrange.add(new JRadioButtonMenuItem("Size"));
		arrange.add(new JRadioButtonMenuItem("Type"));
		arrange.add(new JRadioButtonMenuItem("Modified"));
		arrange.addSeparator();
		arrange.add(new JCheckBoxMenuItem("Show in Groups"));
		arrange.add("Auto Arrange").setEnabled(false);
		JCheckBoxMenuItem grid = new JCheckBoxMenuItem("Align to Grid");
		grid.setEnabled(false);
		arrange.add(grid);
		view.add(arrange);

		bar.add(view);
		setJMenuBar(bar);
	}

	protected static Icon getIcon(String name) {

		Icon icon = null;

		StringBuffer sb = new StringBuffer(ICON_LOCATION);
		sb.append(name);

		URL iconURL = ClassLoader.getSystemResource(sb.toString());
		if (iconURL != null) {
			icon = new ImageIcon(iconURL);
		}

		return icon;
	}

	protected void createPatchTestModules() {

		// Add future patch test modules here.
		addPatchTestModule(new TreePatchTestModule());
		addPatchTestModule(new ComboBoxMarginPatchTestModule());
		addPatchTestModule(new ListSelectionPatchTestModule());
		addPatchTestModule(new TextFieldMarginPatchTestModule());
		addPatchTestModule(new IconPatchTestModule());
        addPatchTestModule(new ScrollBarPatchTestModule());

		Object[] path = { rootPatchTestModule };
		patchTestTreeModel.fireTreeStructureChanged(this, path, null, null);
	}

	protected void addPatchTestModule(IPatchTestModule module) {
		rootPatchTestModule.addSubPatchTestModule(module);
	}

	protected void createListeners() {
		treeSelectionHandler = new TreeSelectionHandler();
	}

	protected void installListeners() {
		patchTestTree.addTreeSelectionListener(treeSelectionHandler);
	}

	protected static class PatchTestTreeModel implements TreeModel {

		protected EventListenerList listenerList;
		protected IPatchTestModule rootPatchTestModule;

		public PatchTestTreeModel(IPatchTestModule rootPatchTestModule) {
			this.rootPatchTestModule = rootPatchTestModule;
			listenerList = new EventListenerList();
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#addTreeModelListener(javax.swing.event.TreeModelListener)
		 */
		public void addTreeModelListener(TreeModelListener l) {
			listenerList.add(TreeModelListener.class, l);
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#getChild(java.lang.Object, int)
		 */
		public Object getChild(Object parent, int index) {
			return ((IPatchTestModule) parent).getSubPatchTestModuleAt(index);
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#getChildCount(java.lang.Object)
		 */
		public int getChildCount(Object parent) {
			return ((IPatchTestModule) parent).getSubPatchTestModuleCount();
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#getIndexOfChild(java.lang.Object, java.lang.Object)
		 */
		public int getIndexOfChild(Object parent, Object child) {
			return ((IPatchTestModule) parent).getIndexOf(
				(IPatchTestModule) child);
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#getRoot()
		 */
		public Object getRoot() {
			return rootPatchTestModule;
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#isLeaf(java.lang.Object)
		 */
		public boolean isLeaf(Object node) {
			return ((IPatchTestModule) node).getSubPatchTestModuleCount() == 0;
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#removeTreeModelListener(javax.swing.event.TreeModelListener)
		 */
		public void removeTreeModelListener(TreeModelListener l) {
			listenerList.remove(TreeModelListener.class, l);
		}

		/* (non-Javadoc)
		 * @see javax.swing.tree.TreeModel#valueForPathChanged(javax.swing.tree.TreePath, java.lang.Object)
		 */
		public void valueForPathChanged(TreePath path, Object newValue) {
		}

		/**
		 * Notifies all listeners that have registered interest for
		 * notification on this event type.  The event instance 
		 * is lazily created using the parameters passed into 
		 * the fire method.
		 *
		 * @param source the node being changed
		 * @param path the path to the root node
		 * @param childIndices the indices of the changed elements
		 * @param children the changed elements
		 * @see EventListenerList
		 */
		protected void fireTreeNodesChanged(
			Object source,
			Object[] path,
			int[] childIndices,
			Object[] children) {
			// Guaranteed to return a non-null array
			Object[] listeners = listenerList.getListenerList();
			TreeModelEvent e = null;
			// Process the listeners last to first, notifying
			// those that are interested in this event
			for (int i = listeners.length - 2; i >= 0; i -= 2) {
				if (listeners[i] == TreeModelListener.class) {
					// Lazily create the event:
					if (e == null)
						e =
							new TreeModelEvent(
								source,
								path,
								childIndices,
								children);
					((TreeModelListener) listeners[i + 1]).treeNodesChanged(e);
				}
			}
		}

		/**
		 * Notifies all listeners that have registered interest for
		 * notification on this event type.  The event instance 
		 * is lazily created using the parameters passed into 
		 * the fire method.
		 *
		 * @param source the node where new elements are being inserted
		 * @param path the path to the root node
		 * @param childIndices the indices of the new elements
		 * @param children the new elements
		 * @see EventListenerList
		 */
		protected void fireTreeNodesInserted(
			Object source,
			Object[] path,
			int[] childIndices,
			Object[] children) {
			// Guaranteed to return a non-null array
			Object[] listeners = listenerList.getListenerList();
			TreeModelEvent e = null;
			// Process the listeners last to first, notifying
			// those that are interested in this event
			for (int i = listeners.length - 2; i >= 0; i -= 2) {
				if (listeners[i] == TreeModelListener.class) {
					// Lazily create the event:
					if (e == null)
						e =
							new TreeModelEvent(
								source,
								path,
								childIndices,
								children);
					((TreeModelListener) listeners[i + 1]).treeNodesInserted(e);
				}
			}
		}

		/**
		 * Notifies all listeners that have registered interest for
		 * notification on this event type.  The event instance 
		 * is lazily created using the parameters passed into 
		 * the fire method.
		 *
		 * @param source the node where elements are being removed
		 * @param path the path to the root node
		 * @param childIndices the indices of the removed elements
		 * @param children the removed elements
		 * @see EventListenerList
		 */
		protected void fireTreeNodesRemoved(
			Object source,
			Object[] path,
			int[] childIndices,
			Object[] children) {
			// Guaranteed to return a non-null array
			Object[] listeners = listenerList.getListenerList();
			TreeModelEvent e = null;
			// Process the listeners last to first, notifying
			// those that are interested in this event
			for (int i = listeners.length - 2; i >= 0; i -= 2) {
				if (listeners[i] == TreeModelListener.class) {
					// Lazily create the event:
					if (e == null)
						e =
							new TreeModelEvent(
								source,
								path,
								childIndices,
								children);
					((TreeModelListener) listeners[i + 1]).treeNodesRemoved(e);
				}
			}
		}

		/**
		 * Notifies all listeners that have registered interest for
		 * notification on this event type.  The event instance 
		 * is lazily created using the parameters passed into 
		 * the fire method.
		 *
		 * @param source the node where the tree model has changed
		 * @param path the path to the root node
		 * @param childIndices the indices of the affected elements
		 * @param children the affected elements
		 * @see EventListenerList
		 */
		protected void fireTreeStructureChanged(
			Object source,
			Object[] path,
			int[] childIndices,
			Object[] children) {
			// Guaranteed to return a non-null array
			Object[] listeners = listenerList.getListenerList();
			TreeModelEvent e = null;
			// Process the listeners last to first, notifying
			// those that are interested in this event
			for (int i = listeners.length - 2; i >= 0; i -= 2) {
				if (listeners[i] == TreeModelListener.class) {
					// Lazily create the event:
					if (e == null)
						e =
							new TreeModelEvent(
								source,
								path,
								childIndices,
								children);
					(
						(TreeModelListener) listeners[i
							+ 1]).treeStructureChanged(
						e);
				}
			}
		}
	}

	protected class TreeSelectionHandler implements TreeSelectionListener {

		/* (non-Javadoc)
		* @see javax.swing.event.TreeSelectionListener#valueChanged(javax.swing.event.TreeSelectionEvent)
		*/
		public void valueChanged(TreeSelectionEvent e) {

			TreePath path = e.getPath();
			if (path.getPathCount() == 1) {
				return;
			}

			IPatchTestModule module =
				(IPatchTestModule) path.getLastPathComponent();

			JSplitPane contentPane = (JSplitPane) getContentPane();
			TreePath oldPath = e.getOldLeadSelectionPath();
			if (oldPath != null) {
				IPatchTestModule oldModule =
					(IPatchTestModule) oldPath.getLastPathComponent();
				contentPane.remove(oldModule.getContentPane());
			}

			((Container) contentPane.getRightComponent()).remove(
				((Container) contentPane.getRightComponent()).getComponent(0));
			((Container) contentPane.getRightComponent()).add(
				module.getContentPane(),
				BorderLayout.CENTER);
			contentPane.repaint();
			contentPane.validate();
		}
	}
}