/*
  The contents of this file are subject to the Sun Public License
	Version 1.0 (the "License"); you may not use this file except in
	compliance with the License. A copy of the License is available at
	http://www.sun.com/

	The Original Code is winlaf. The Initial Developer of the
	Original Code is Brian Duff. Portions created by Brian Duff are Copyright
	(C)Brian Duff. All Rights Reserved.

	Contributor(s): ______________________________________.
*/

package net.java.plaf;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.StringTokenizer;

/**
 * Use the LookAndFeelPatchRegistry to install patches for look and feel issues.
 * Your application should call the initialize() method as eary as possible
 * during application startup.
 * 
 * @author Brian.Duff@oracle.com
 */
public class LookAndFeelPatchManager {
	private static final String SYSPROP_PATCHES = "net.java.plaf.patches"; //$NON-NLS-1$
	private static final String SYSPROP_ADDITIONAL_PATCHES = "net.java.plaf.additionalpatches"; //$NON-NLS-1$

	private static final String[] LAF_PATCHES = new String[] { "net.java.plaf.windows.common.WindowsMenuPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPMenuPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPMenuItemPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPTextFieldMarginPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPComboBoxMarginPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPPasswordFieldPatch", //$NON-NLS-1$
		"net.java.plaf.windows.common.WindowsTextFieldPatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPIconPatch", //$NON-NLS-1$
		"net.java.plaf.windows.common.WindowsOptionPanePatch", //$NON-NLS-1$
		"net.java.plaf.windows.xp.XPScrollBarPatch" //$NON-NLS-1$
	};

	private static Collection m_allPatches;
	private static Collection m_installedPatches;

	/**
	 * Initialize the look and feel patch manager. The manager will:
	 * 
	 *  - Install applicable look and feel fixes for the current look and feel
	 *  - Add a listener so that when the look and feel changes, any existing
	 *    patches are uninstalled and new ones are installed for the new look
	 *    and feel.
	 *    
	 */
	public static void initialize() {
		ArrayList patchClasses = new ArrayList(10);

		String patches = System.getProperty(SYSPROP_PATCHES);
		if (patches != null) {
			addClasses(patchClasses, patches);
		}
		else {
			patchClasses.addAll(Arrays.asList(LAF_PATCHES));
		}

		String extraPatches = System.getProperty(SYSPROP_ADDITIONAL_PATCHES);
		if (extraPatches != null) {
			addClasses(patchClasses, extraPatches);
		}

		// Convert each patch in the array into a proper instance of the patch
		// class.
		m_allPatches = new ArrayList(patchClasses.size());
		for (Iterator i = patchClasses.iterator(); i.hasNext();) {
			String patchName = (String) i.next();
			try {
				Class c = Class.forName(patchName);
				Object o = c.newInstance();
				if (o instanceof LookAndFeelPatch) {
					m_allPatches.add(o);
				}
			}
			catch (Exception e) {
				System.err.println("Failed to install patch " + patchName); //$NON-NLS-1$
				e.printStackTrace();
			}
		}

		patch();
	}

	private static void patch() {
		m_installedPatches = new ArrayList();
		Environment env = new Environment();
		for (Iterator i = m_allPatches.iterator(); i.hasNext();) {
			LookAndFeelPatch patch = (LookAndFeelPatch) i.next();

			if (patch.isApplicable(env)) {
				patch.patch(env);
				m_installedPatches.add(patch);
			}
		}
	}

	private static void addClasses(Collection coll, String classList) {
		StringTokenizer tok =
			new StringTokenizer(
				classList,
				System.getProperty("path.separator")); //$NON-NLS-1$
		while (tok.hasMoreTokens()) {
			coll.add(tok.nextToken());
		}
	}
}