/*
 * Decompiled with CFR 0.152.
 */
package com.android.tools.idea.welcome.config;

import com.google.common.base.Function;
import com.google.common.base.Strings;
import com.google.common.collect.Iterables;
import com.intellij.openapi.progress.ProgressIndicator;
import com.intellij.openapi.progress.Task;
import com.intellij.openapi.projectRoots.JavaSdk;
import com.intellij.openapi.projectRoots.JavaSdkVersion;
import com.intellij.openapi.projectRoots.JdkUtil;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.text.StringUtil;
import java.io.File;
import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.concurrent.atomic.AtomicBoolean;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

public class JdkDetection {
    private static final AtomicBoolean myJdkDetectionInProgress = new AtomicBoolean();

    private JdkDetection() {
    }

    public static void start(JdkDetectionResult result) {
        new DetectJdkTask(result, true).queue();
    }

    public static void startWithProgressIndicator(JdkDetectionResult result) {
        new DetectJdkTask(result, false).queue();
    }

    @Nullable
    public static String validateJdkLocation(@Nullable File location) {
        if (location == null) {
            return "Path is empty";
        }
        if (!JdkUtil.checkForJdk((File)location)) {
            return "Path specified is not a valid JDK location";
        }
        if (!JdkDetection.isJdk7(location)) {
            return "JDK 7.0 or newer is required";
        }
        return null;
    }

    private static boolean isJdk7(@NotNull File path) {
        JavaSdkVersion version;
        String jdkVersion = JavaSdk.getJdkVersion((String)path.getAbsolutePath());
        return jdkVersion == null || (version = JavaSdk.getInstance().getVersion(jdkVersion)) == null || version.isAtLeast(JavaSdkVersion.JDK_1_7);
    }

    private static class DetectJdkTask
    extends Task.Modal {
        private static final String MAC_JDKS_DIR = "/Library/Java/JavaVirtualMachines/";
        private static final String WINDOWS_JDKS_DIR = "C:\\Program Files\\Java";
        private static final String LINUX_SDK_DIR = "/usr/lib/jvm";
        private final AtomicBoolean myCancelled = new AtomicBoolean(false);
        private final JdkDetectionResult myResult;
        private final boolean myHeadless;
        private String myPath = null;

        public DetectJdkTask(JdkDetectionResult result, boolean headless) {
            super(null, "Detect JDK", true);
            this.myResult = result;
            this.myHeadless = headless;
        }

        public void run(@NotNull ProgressIndicator indicator) {
            indicator.setIndeterminate(true);
            if (myJdkDetectionInProgress.compareAndSet(false, true)) {
                try {
                    this.myPath = DetectJdkTask.detectJdkPath(this.myCancelled);
                }
                finally {
                    myJdkDetectionInProgress.set(false);
                }
            }
            while (myJdkDetectionInProgress.get()) {
                try {
                    Thread.sleep(300L);
                }
                catch (InterruptedException interruptedException) {}
            }
        }

        public boolean isHeadless() {
            if (this.myHeadless) {
                return false;
            }
            return super.isHeadless();
        }

        public void onSuccess() {
            this.myResult.onSuccess(this.myPath);
        }

        public void onCancel() {
            this.myCancelled.set(true);
            this.myResult.onCancel();
        }

        @Nullable
        private static String detectJdkPath(@Nullable AtomicBoolean cancellationFlag) {
            String topVersion = null;
            String chosenPath = null;
            for (String path : DetectJdkTask.getCandidatePaths()) {
                if (cancellationFlag != null && cancellationFlag.get()) {
                    return null;
                }
                if (!StringUtil.isEmpty((String)JdkDetection.validateJdkLocation(new File(path)))) continue;
                String version = JavaSdk.getInstance().getVersionString(path);
                if (topVersion != null && version != null && topVersion.compareTo(version) >= 0) continue;
                topVersion = version;
                chosenPath = path;
            }
            return chosenPath;
        }

        @NotNull
        private static Iterable<String> getCandidatePaths() {
            return Iterables.concat(DetectJdkTask.deduceFromJavaHome(), DetectJdkTask.deduceFromPath(), DetectJdkTask.deduceFromCurrentJvm(), DetectJdkTask.getOsSpecificCandidatePaths());
        }

        @NotNull
        private static Iterable<String> deduceFromJavaHome() {
            String javaHome = System.getenv("JAVA_HOME");
            return Strings.isNullOrEmpty((String)javaHome) ? Collections.emptySet() : Collections.singleton(javaHome);
        }

        @NotNull
        private static Iterable<String> deduceFromPath() {
            String[] pathEntries;
            String path = System.getenv("PATH");
            if (Strings.isNullOrEmpty((String)path)) {
                return Collections.emptyList();
            }
            for (String entry : pathEntries = path.split(File.pathSeparator)) {
                if (Strings.isNullOrEmpty((String)entry)) continue;
                File javaParentDir = new File(entry);
                File javaFile = new File(javaParentDir, "java");
                if (!javaParentDir.isDirectory() || !javaFile.isFile()) continue;
                try {
                    File canonicalJavaFile = javaFile.getCanonicalFile();
                    return DetectJdkTask.forJavaBinParent(canonicalJavaFile.getParentFile());
                }
                catch (IOException iOException) {
                    // empty catch block
                }
            }
            return Collections.emptyList();
        }

        @NotNull
        private static Iterable<String> deduceFromCurrentJvm() {
            String javaHome = System.getProperty("java.home");
            return Strings.isNullOrEmpty((String)javaHome) ? Collections.emptySet() : DetectJdkTask.forJavaBinParent(new File(javaHome));
        }

        @NotNull
        private static Iterable<String> forJavaBinParent(@NotNull File javaBinParent) {
            if (!javaBinParent.isDirectory()) {
                return Collections.emptySet();
            }
            if (!"jre".equals(javaBinParent.getName())) {
                return Collections.singleton(javaBinParent.getAbsolutePath());
            }
            File parentFile = javaBinParent.getParentFile();
            if (parentFile.isDirectory()) {
                return Collections.singleton(parentFile.getAbsolutePath());
            }
            return Collections.emptySet();
        }

        @NotNull
        private static Iterable<String> getOsSpecificCandidatePaths() {
            if (SystemInfo.isMac) {
                return DetectJdkTask.getMacCandidateJdks();
            }
            if (SystemInfo.isWindows) {
                return DetectJdkTask.getWindowsCandidateJdks();
            }
            if (SystemInfo.isLinux) {
                return DetectJdkTask.getLinuxCandidateJdks();
            }
            return Collections.emptyList();
        }

        @NotNull
        private static Iterable<String> getMacCandidateJdks() {
            return DetectJdkTask.getCandidatePaths(MAC_JDKS_DIR, "/Contents/Home");
        }

        @NotNull
        private static Iterable<String> getWindowsCandidateJdks() {
            return DetectJdkTask.getCandidatePaths(WINDOWS_JDKS_DIR, "");
        }

        @NotNull
        private static Iterable<String> getLinuxCandidateJdks() {
            return DetectJdkTask.getCandidatePaths(LINUX_SDK_DIR, "");
        }

        private static Iterable<String> getCandidatePaths(String basedir, final String suffix) {
            final File location = new File(basedir);
            if (location.isDirectory()) {
                return Iterables.transform(Arrays.asList(location.list()), (Function)new Function<String, String>(){

                    public String apply(@Nullable String dir) {
                        return new File(location, dir + suffix).getAbsolutePath();
                    }
                });
            }
            return Collections.emptyList();
        }
    }

    public static interface JdkDetectionResult {
        public void onSuccess(String var1);

        public void onCancel();
    }
}

