/*
 * Decompiled with CFR 0.152.
 */
package com.android.tools.idea.ui.validation.validators;

import com.android.repository.io.FileOp;
import com.android.repository.io.FileOpUtils;
import com.android.tools.idea.ui.validation.Validator;
import com.google.common.base.CharMatcher;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.intellij.openapi.application.Application;
import com.intellij.openapi.application.ApplicationNamesInfo;
import com.intellij.openapi.application.PathManager;
import com.intellij.openapi.util.SystemInfo;
import com.intellij.openapi.util.io.FileUtil;
import java.io.File;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import net.jcip.annotations.Immutable;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

@Immutable
public final class PathValidator
implements Validator<File> {
    public static final Rule IS_EMPTY = new SimpleRule(){

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return file.getName().isEmpty();
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("Please specify a %1$s.", fieldName);
        }
    };
    public static final Rule INVALID_SLASHES = new SimpleRule(){

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            String path = file.getPath();
            return File.separatorChar == '/' && path.contains("\\") || File.separatorChar == '\\' && path.contains("/");
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            if (File.separatorChar == '\\') {
                return String.format("Your %1$s contains incorrect slashes ('/').", fieldName);
            }
            return String.format("Your %1$s contains incorrect slashes ('\\').", fieldName);
        }
    };
    public static final Rule WHITESPACE = new RecursiveRule(){

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return CharMatcher.WHITESPACE.matchesAnyOf((CharSequence)file.getName());
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("%1$s should not contain whitespace, as this can cause problems with the NDK tools.", fieldName);
        }
    };
    public static final Rule NON_ASCII_CHARS = new RecursiveRule(){

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return !CharMatcher.ASCII.matchesAllOf((CharSequence)file.getName());
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("Your %1$s contains non-ASCII characters.", fieldName);
        }
    };
    public static final Rule PARENT_DIRECTORY_NOT_WRITABLE = new RecursiveRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            File parent = file.getParentFile();
            return !fileOp.exists(file) && parent != null && fileOp.exists(parent) && !fileOp.canWrite(parent);
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The path '%1$s' is not writable. Please choose a new location.", file.getParentFile().getPath());
        }
    };
    public static final Rule LOCATION_IS_A_FILE = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return fileOp.isFile(file);
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The %1$s specified already exists.", fieldName);
        }
    };
    public static final Rule LOCATION_IS_ROOT = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return file.getParent() == null;
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The %1$s cannot be at the filesystem root.", fieldName);
        }
    };
    public static final Rule PARENT_IS_NOT_A_DIRECTORY = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            File parent = file.getParentFile();
            return parent != null && fileOp.exists(parent) && !fileOp.isDirectory(parent);
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The %1$s's parent must be a directory, not a plain file.", fieldName);
        }
    };
    public static final Rule PATH_NOT_WRITABLE = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return fileOp.exists(file) && !fileOp.canWrite(file);
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The path '%1$s' is not writable. Please choose a new location.", file.getPath());
        }
    };
    public static final Rule PATH_INSIDE_ANDROID_STUDIO = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            String installLocation = PathManager.getHomePathFor(Application.class);
            return installLocation != null && FileUtil.isAncestor((File)new File(installLocation), (File)file, (boolean)false);
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            String applicationName = ApplicationNamesInfo.getInstance().getProductName();
            return String.format("The %1$s is inside %2$s's install location.", fieldName, applicationName);
        }
    };
    public static final Rule NON_EMPTY_DIRECTORY = new SimpleRule(){

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return fileOp.listFiles(file).length > 0;
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("A non-empty directory already exists at the specified %1$s.", fieldName);
        }
    };
    public static final Rule ILLEGAL_FILENAME = new RecursiveRule(){
        private final Set<String> RESERVED_WINDOWS_FILENAMES = ImmutableSet.of((Object)"con", (Object)"prn", (Object)"aux", (Object)"clock$", (Object)"nul", (Object)"com1", (Object[])new String[]{"com2", "com3", "com4", "com5", "com6", "com7", "com8", "com9", "lpt1", "lpt2", "lpt3", "lpt4", "lpt5", "lpt6", "lpt7", "lpt8", "lpt9", "$mft", "$mftmirr", "$logfile", "$volume", "$attrdef", "$bitmap", "$boot", "$badclus", "$secure", "$upcase", "$extend", "$quota", "$objid", "$reparse"});

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return fileOp.isWindows() && this.RESERVED_WINDOWS_FILENAMES.contains(file.getName().toLowerCase(Locale.US));
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("Illegal filename in %1$s path: %2$s.", fieldName, file.getName());
        }
    };
    public static final Rule PATH_TOO_LONG = new SimpleRule(){
        private static final int WINDOWS_PATH_LENGTH_LIMIT = 100;

        @Override
        protected boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return file.getAbsolutePath().length() > 100;
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            return String.format("The length of the %1$s exceeds the limit of %2$d characters.", fieldName, 100);
        }
    };
    public static final Rule ILLEGAL_CHARACTER = new RecursiveRule(){
        private final CharMatcher ILLEGAL_CHARACTER_MATCHER = CharMatcher.anyOf((CharSequence)"[/\\\\?%*:|\"<>!;]");

        @Override
        public boolean matches(@NotNull FileOp fileOp, @NotNull File file) {
            return this.ILLEGAL_CHARACTER_MATCHER.matchesAnyOf((CharSequence)file.getName());
        }

        @Override
        @NotNull
        public String getMessage(@NotNull File file, @NotNull String fieldName) {
            String name = file.getName();
            char illegalChar = name.charAt(this.ILLEGAL_CHARACTER_MATCHER.indexIn((CharSequence)name));
            return String.format("Illegal character in %1$s path: '%2$c' in filename %3$s.", fieldName, Character.valueOf(illegalChar), name);
        }
    };
    @NotNull
    private final String myPathName;
    @NotNull
    private final Iterable<Rule> myErrors;
    @NotNull
    private final Iterable<Rule> myWarnings;
    @NotNull
    private final FileOp myFileOp;

    private PathValidator(@NotNull String pathName, @NotNull Iterable<Rule> errors, @NotNull Iterable<Rule> warnings, @NotNull FileOp fileOp) {
        this.myPathName = pathName;
        this.myErrors = errors;
        this.myWarnings = warnings;
        this.myFileOp = fileOp;
    }

    public static PathValidator createDefault(@NotNull String pathName) {
        return new Builder().withAllRules(Validator.Severity.ERROR).build(pathName);
    }

    @Override
    @NotNull
    public Validator.Result validate(@NotNull File file) {
        Validator.Result result = this.validate(file, Validator.Severity.ERROR);
        if (result != Validator.Result.OK) {
            return result;
        }
        result = this.validate(file, Validator.Severity.WARNING);
        if (result != Validator.Result.OK) {
            return result;
        }
        return Validator.Result.OK;
    }

    @NotNull
    private Validator.Result validate(@NotNull File projectFile, @NotNull Validator.Severity severity) {
        assert (severity != Validator.Severity.OK);
        Iterable<Rule> rules = severity == Validator.Severity.ERROR ? this.myErrors : this.myWarnings;
        for (Rule rule : rules) {
            File matchingFile = rule.getMatchingFile(this.myFileOp, projectFile);
            if (matchingFile == null) continue;
            return new Validator.Result(severity, rule.getMessage(matchingFile, this.myPathName));
        }
        return Validator.Result.OK;
    }

    private static abstract class RecursiveRule
    implements Rule {
        private RecursiveRule() {
        }

        @Override
        @Nullable
        public final File getMatchingFile(@NotNull FileOp fileOp, @NotNull File file) {
            for (File currFile = file; currFile != null; currFile = currFile.getParentFile()) {
                if (!this.matches(fileOp, currFile)) continue;
                return currFile;
            }
            return null;
        }

        protected abstract boolean matches(@NotNull FileOp var1, @NotNull File var2);
    }

    private static abstract class SimpleRule
    implements Rule {
        private SimpleRule() {
        }

        @Override
        @Nullable
        public final File getMatchingFile(@NotNull FileOp fileOp, @NotNull File file) {
            return this.matches(fileOp, file) ? file : null;
        }

        protected abstract boolean matches(@NotNull FileOp var1, @NotNull File var2);
    }

    public static class Builder {
        private final List<Rule> myErrors = Lists.newArrayList();
        private final List<Rule> myWarnings = Lists.newArrayList();

        @NotNull
        public Builder withAllRules(@NotNull Validator.Severity pathNotWritable) {
            this.withRule(IS_EMPTY, Validator.Severity.ERROR);
            this.withRule(PATH_NOT_WRITABLE, pathNotWritable);
            this.withRule(NON_EMPTY_DIRECTORY, Validator.Severity.WARNING);
            return this.withCommonRules();
        }

        @NotNull
        public Builder withCommonRules() {
            this.withRule(INVALID_SLASHES, Validator.Severity.ERROR);
            this.withRule(ILLEGAL_CHARACTER, Validator.Severity.ERROR);
            this.withRule(ILLEGAL_FILENAME, Validator.Severity.ERROR);
            this.withRule(WHITESPACE, Validator.Severity.WARNING);
            this.withRule(NON_ASCII_CHARS, SystemInfo.isWindows ? Validator.Severity.ERROR : Validator.Severity.WARNING);
            this.withRule(PARENT_DIRECTORY_NOT_WRITABLE, Validator.Severity.ERROR);
            this.withRule(PATH_TOO_LONG, Validator.Severity.ERROR);
            this.withRule(LOCATION_IS_A_FILE, Validator.Severity.ERROR);
            this.withRule(LOCATION_IS_ROOT, Validator.Severity.ERROR);
            this.withRule(PARENT_IS_NOT_A_DIRECTORY, Validator.Severity.ERROR);
            this.withRule(PATH_INSIDE_ANDROID_STUDIO, Validator.Severity.ERROR);
            return this;
        }

        public Builder withRule(@NotNull Rule rule, @NotNull Validator.Severity severity) {
            switch (severity) {
                case ERROR: {
                    this.myErrors.add(rule);
                    break;
                }
                case WARNING: {
                    this.myWarnings.add(rule);
                    break;
                }
                default: {
                    throw new IllegalArgumentException(String.format("Can't create rule with invalid severity %1$s", new Object[]{severity}));
                }
            }
            return this;
        }

        @NotNull
        public PathValidator build(@NotNull String pathName) {
            return this.build(pathName, FileOpUtils.create());
        }

        @NotNull
        public PathValidator build(@NotNull String pathName, @NotNull FileOp fileOp) {
            return new PathValidator(pathName, this.myErrors, this.myWarnings, fileOp);
        }
    }

    public static interface Rule {
        @Nullable
        public File getMatchingFile(@NotNull FileOp var1, @NotNull File var2);

        @NotNull
        public String getMessage(@NotNull File var1, @NotNull String var2);
    }
}

