#!/bin/bash

##
## This script generates a YCP file that contains only translations.
## Pot files are generated by common 'make pot' command along
## with generating all other common pot-files.
##
## More information about this feature:
## #300687: Ports for SuSEfirewall added via packages
##

# configuration
# DISTRO: distribution for `pdb` command
DISTRO="stable-i386"

# must NOT begin with slash, needs to be relative!
SUSEFWFILES="etc/sysconfig/SuSEfirewall2.d/*"
# where new translations are stored
Y2_NEWYCPFILE="yast2-services-translations.ycp"
# ycpfile textdomain ('base' is yast2.rpm)
TEXTDOMAIN="firewall-services"

CURRENTDIR=`pwd`

# check whether needed packages/scripts/binaries are installed
PDBINSTALLED=`which pdb`
if [ "$PDBINSTALLED" == "" ]; then
    echo "Please install 'pdb-commandline' package"
    echo
    exit 1
fi
MKTEMPINSTALLED=`which mktemp`
if [ "$MKTEMPINSTALLED" == "" ]; then
    echo "Please install 'mktemp' package"
    echo
    exit 2
fi
RPM2CPIOINSTALLED=`which rpm2cpio`
if [ "$RPM2CPIOINSTALLED" == "" ]; then
    echo "Please install 'rpm' package"
    echo
    exit 3
fi
CPIOINSTALLED=`which cpio`
if [ "$CPIOINSTALLED" == "" ]; then
    echo "Please install 'cpio' package"
    echo
    exit 4
fi
YCPCINSTALLED=`which ycpc`
if [ "$YCPCINSTALLED" == "" ]; then
    echo "Please install 'yast2-core' package"
    echo
    exit 5
fi

# definition of used directories
RPMDIRECTORY=$1

if [ "$RPMDIRECTORY" == "" ]; then
    echo "Please use $0 /path/to/a/mounted/installation/DVD/with/related/product"
    echo
    exit 6
fi

if [ ! -d $RPMDIRECTORY ]; then
    echo "Directory $RPMDIRECTORY does not exist"
    exit 6
fi

if [ ! -d $RPMDIRECTORY/suse ]; then
    echo "Directory $RPMDIRECTORY does not contain SUSE distribution"
    exit 6
fi

# checks PDB for packages containing such files
# PDB certificate is invalid, PERL_LWP_SSL_VERIFY_HOSTNAME must be set to 0
AFFECTED=`PERL_LWP_SSL_VERIFY_HOSTNAME=0 pdb query --release stable-i386 --filter "rpmfile:/$SUSEFWFILES" --attribs packname | grep -v '^SuSEfirewall2[\t ]*$'`

# create temporary directory
TMPDIR=`mktemp -d`
if [ "$TMPDIR" == "" ]; then
    echo "Cannot create temporary directory..."
    echo
    exit 21
fi
echo "Using temporary directory: "$TMPDIR
echo
cd $TMPDIR

ls -1 $TMPDIR/$SUSEFWFILES > $TMPDIR/last

# installs these files into a temporary directory
for ONEPACKAGE in $AFFECTED; do
    echo "Package: "$ONEPACKAGE
    rpm2cpio $RPMDIRECTORY/suse/*/$ONEPACKAGE-*.rpm | cpio -idm --no-absolute-filenames "./$SUSEFWFILES" "$SUSEFWFILES" 2> /dev/null
    ls -1 $TMPDIR/$SUSEFWFILES > $TMPDIR/now

    for CPIOEDFILE in `diff -u $TMPDIR/last $TMPDIR/now | grep "^\+[^\+]" | sed 's/\+//'`; do
	echo "	File: ${CPIOEDFILE}"
	echo "${CPIOEDFILE}_:_$ONEPACKAGE" >> $TMPDIR/FileBelongsTo
    done

    mv $TMPDIR/now $TMPDIR/last
done
echo

cd $SUSEFWFILES
echo "Current working directory is: "`pwd`
echo

# open YCP file
echo "{
  /**
   * This file contains only translations for
   * FATE #300687: Ports for SuSEfirewall added via packages
   *
   * Translations are grabbed when 'make pot' is called.
   */

  textdomain \""$TEXTDOMAIN"\";

  string tmpstring = nil;
" > $Y2_NEWYCPFILE

for FILENAME in `ls -1 | grep -v $Y2_NEWYCPFILE`; do
    #echo "Transforming file: "$FILENAME

    RPMNAME=`grep "${FILENAME}_:_" $TMPDIR/FileBelongsTo | sed s'/.*_:_//'`

    # ... ## Name:
    TAG_NAME=`grep -i ".*##[\t ]*Name:[\t ]*" $FILENAME | sed 's/^.*##[\t ]*[Nn][Aa][Mm][Ee]:[\t ]//' | sed 's/"/\\"/'`
    if [ "$TAG_NAME" == "" ]; then
	echo "Something is wrong with file '"$FILENAME"', no 'Name:' tag found!"
    elif [ "$TAG_NAME" == "template service" ]; then
	echo "'Name:' not adjusted correctly in the '"$FILENAME"' file!"
    else
	echo "  // TRANSLATORS: Name of Service (File name: "$FILENAME", RPM: "$RPMNAME"), can be used as check box, item in multiple selection box..." >> $Y2_NEWYCPFILE
	echo "  tmpstring = _(\""$TAG_NAME"\");" >> $Y2_NEWYCPFILE
	echo >> $Y2_NEWYCPFILE
    fi

    # ... ## Description:
    TAG_DESCRIPTION=`grep -i ".*##[\t ]*Description:[\t ]*" $FILENAME | sed 's/^.*##[\t ]*[Dd][Ee][Ss][Cc][Rr][Ii][Pp][Tt][Ii][Oo][Nn]:[\t ]//' | sed 's/"/\\"/'`
    if [ "$TAG_DESCRIPTION" == "" ]; then
	echo "Something is wrong with file '"$FILENAME"', no 'Description:' tag found!"
    elif [ "$TAG_DESCRIPTION" == "opens ports for foo in order to allow bar" ]; then
	echo "'Description:' not adjusted correctly in the '"$FILENAME"' file!"
    else
	echo "  // TRANSLATORS: Description of a Service (File name: "$FILENAME", RPM: "$RPMNAME"), used as a common label or an item in table" >> $Y2_NEWYCPFILE
	echo "  tmpstring = _(\""$TAG_DESCRIPTION"\");" >> $Y2_NEWYCPFILE
	echo >> $Y2_NEWYCPFILE
    fi
done
# close YCP file
echo "}" >> $Y2_NEWYCPFILE
echo

cp $Y2_NEWYCPFILE $CURRENTDIR/$Y2_NEWYCPFILE.new

# cleanup
echo "Removing  temporary directory: "$TMPDIR
rm -rf "'"$TMPDIR"'"
echo

cd $CURRENTDIR

CHECKYCP=`ycpc -E -q $Y2_NEWYCPFILE.new`
if [ "$CHECKYCP" != "" ]; then
    echo "Invalid YCP syntax ("$Y2_NEWYCPFILE.new")!"
    echo
    exit 41
fi

# check whether it differs
FILEDIFF=`diff -u $Y2_NEWYCPFILE $Y2_NEWYCPFILE.new`
if [ "$FILEDIFF" == "" ]; then
    echo "No updates needed"
    echo
    rm -rf $Y2_NEWYCPFILE.new
else
    echo "Here comes the diff:"
    diff -u $Y2_NEWYCPFILE $Y2_NEWYCPFILE.new
    mv --force $Y2_NEWYCPFILE.new $Y2_NEWYCPFILE
    echo
fi

exit 0
