unit SDUFileIterator_U;
// Description: File Iterator
// By Sarah Dean
//
// -----------------------------------------------------------------------------

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  SDUDirIterator_U;

type

  TSDUFileIterator = class(TComponent)
  private
    currDir: string;
    currDirsFiles: TStringList;
    currFileIdx: integer;
    dirIterator: TSDUDirIterator;
    procedure ReadFilenames(theDir: string);
    function CountFilenames(theDir: string): integer;
  protected
    FRootDirectory: string;
    FFileMask: string;
    FSorted: boolean;
    FIncludeSubDirs: boolean;
    FSupressDotSlash: boolean;
    FIncludeDirNames: boolean;
    function SlashSep(const Path, S: String): String;
    procedure SetRootDirectory(const theDir: string);
    procedure SetFileMask(const theFileMask: string);
    procedure SortedFilenames(sort: boolean);
  public
    constructor Create(AOwner: TComponent); override;
    destructor Destroy(); override;

    procedure Reset();
    function  Next(): string;
    function  Count(): integer;
  published
    property Directory: string read FRootDirectory write SetRootDirectory;
    property FileMask: string read FFileMask write SetFileMask;
    property Sorted: boolean read FSorted write SortedFilenames;
    property IncludeSubDirs: boolean read FIncludeSubDirs write FIncludeSubDirs;
    property SupressDotSlash: boolean read FSupressDotSlash write FSupressDotSlash;

    // If this is set, the directory names will be output, as well as filenames
    // Set to FALSE by default
    property IncludeDirNames: boolean read FIncludeDirNames write FIncludeDirNames;
  end;

procedure Register;

implementation

procedure Register;
begin
  RegisterComponents('SDeanUtils', [TSDUFileIterator]);
end;

constructor TSDUFileIterator.Create(AOwner: TComponent);
begin
  inherited;
  currDirsFiles := TStringList.Create();
  currFileIdx:=0;

  FRootDirectory := '.';
  FFileMask := '*.*';

  FSupressDotSlash := TRUE;
  FIncludeDirNames := FALSE;

  dirIterator:= TSDUDirIterator.Create(nil);

end;

destructor TSDUFileIterator.Destroy();
begin
  dirIterator.Free();
  currDirsFiles.Free();
  inherited;
  
end;

procedure TSDUFileIterator.Reset();
begin
  dirIterator.Directory := FRootDirectory;
  dirIterator.Reset();
  ReadFilenames(dirIterator.Next());

end;

function TSDUFileIterator.Count(): integer;
var
  dIt: TSDUDirIterator;
  cnt: integer;
  currDir: string;
begin
  cnt := 0;

  if not(fIncludeSubDirs) then
    begin
    inc(cnt, CountFilenames(FRootDirectory));
    end
  else
    begin
    dIt := TSDUDirIterator.Create(nil);
    try
      dIt.Directory := FRootDirectory;
      dIt.ReverseFormat := TRUE;
      dIt.IncludeStartDir := TRUE;
      dIt.Reset();

      currDir := dIt.Next();
      while currDir<>'' do
        begin
        inc(cnt, CountFilenames(currDir));
        currDir := dIt.Next();
        end;

    finally
      dIt.Free();
    end;

    end;

  Result := cnt;

end;


function TSDUFileIterator.Next(): string;
var
  nextDir: string;
  returnFilename: string;
begin
  Result := '';

  while currFileIdx>=currDirsFiles.count do
    begin
    if not(FIncludeSubDirs) then
      begin
      exit;
      end;

    nextDir := dirIterator.Next();
    if nextDir='' then
      begin
      exit;
      end
    else
      begin
      ReadFilenames(nextDir);
      end;
    end;

  returnFilename := SlashSep(currDir, currDirsFiles[currFileIdx]);
  inc(currFileIdx);

  if SupressDotSlash then
    begin
    if Pos('.\', returnFilename)=1 then
      begin
      Delete(returnFilename, 1, 2);
      end;
    end;

  Result := returnFilename;

end;


function TSDUFileIterator.SlashSep(const Path, S: String): String;
begin
  if AnsiLastChar(Path)^ <> '\' then
    Result := Path + '\' + S
  else
    Result := Path + S;
end;


procedure TSDUFileIterator.ReadFilenames(theDir: string);
var
  SearchRec: TSearchRec;
  Status: integer;
begin
  currDirsFiles.Clear();

  if theDir<>'' then
    begin
    Status := FindFirst(SlashSep(theDir, FFileMask), faAnyFile, SearchRec);

    while (Status=0) do
      begin
      if (FIncludeDirNames or (SearchRec.Attr AND faDirectory <> faDirectory)) then
        begin
        currDirsFiles.add(SearchRec.Name);
        end;
      Status := FindNext(SearchRec);
      end;

    FindClose(SearchRec);
    end;

  currDirsFiles.Sorted := FSorted;

  currFileIdx:=0;
  currDir := theDir;

end;


function TSDUFileIterator.CountFilenames(theDir: string): integer;
var
  SearchRec: TSearchRec;
  Status: integer;
  cnt: integer;
begin
  cnt:= 0;

  if theDir<>'' then
    begin
    Status := FindFirst(SlashSep(theDir, FFileMask), faAnyFile, SearchRec);

    while (Status=0) do
      begin
      if (SearchRec.Attr AND faDirectory <> faDirectory) then
        begin
        inc(cnt);
        end;
      Status := FindNext(SearchRec);
      end;

    FindClose(SearchRec);
    end;

  Result := cnt;

end;


procedure TSDUFileIterator.SetRootDirectory(const theDir: string);
begin
  FRootDirectory := theDir;
  Reset();

end;

procedure TSDUFileIterator.SetFileMask(const theFileMask: string);
begin
  Directory := ExtractFilePath(theFileMask);
  if (Directory='') then
    begin
    Directory := '.';
    end;

  FFileMask := ExtractFileName(theFileMask);
  Reset();

end;

procedure TSDUFileIterator.SortedFilenames(sort: boolean);
begin
  FSorted:= sort;
  currDirsFiles.Sorted := FSorted;

end;

END.

