program SDHash;

uses
  SysUtils, // needed for Format
  Windows, // needed for SYSTEMTIME
  SDUGeneral,
  SDUFileIterator_U,
  HashAlg_U,
  HashAlgGOST_U,
  HashAlgMD2_U,
  HashAlgMD4_U,
  HashAlgMD5_U,
  HashAlgSHA_U,
  HashAlgSHA1_U,
  HashAlgRIPEMD_U;

type
  // Note: KEEP THESE BOTH IN THE SAME ORDER!!
  fhHashType  = (
                 hashGOST,
                 hashMD2,
                 hashMD4,
                 hashMD5,
                 hashSHA,
                 hashSHA1,
                 hashRIPEMD_128,
                 hashRIPEMD_160,
                 hashRIPEMD_256,
                 hashRIPEMD_320
                 );
                 
const
  fhHashNames: array [fhHashType] of string = (
                                               'GOST',
                                               'MD2',
                                               'MD4',
                                               'MD5',
                                               'SHA',
                                               'SHA-1',
                                               'RIPEMD-128',
                                               'RIPEMD-160',
                                               'RIPEMD-256',
                                               'RIPEMD-320'
                                               );


var
  hashObjects: array [fhHashType] of THashAlg;
  hashLoop: fhHashType;
  filename: string;
  hash: THashArray;
  doAllHashes: boolean;
  doTestVector: boolean;
  doRecursive: boolean;
  doLowercase: boolean;
  numHashesToDo: integer;
  fileIterator: TSDUFileIterator;
  fileSpec: string;
  cntFiles: integer;
  exeFilename: string;
  exeFilenameExt: string;
  progStartTime: TTimeStamp;
  hashStartTime: TTimeStamp;
  doTimed: boolean;

procedure CreateHashObjects();
begin
  hashObjects[hashGOST] := THashAlgGOST.Create(nil);
  hashObjects[hashMD2]  := THashAlgMD2.Create(nil);
  hashObjects[hashMD4]  := THashAlgMD4.Create(nil);
  hashObjects[hashMD5]  := THashAlgMD5.Create(nil);
  hashObjects[hashSHA]  := THashAlgSHA.Create(nil);
  hashObjects[hashSHA1] := THashAlgSHA1.Create(nil);
  hashObjects[hashRIPEMD_128] := THashAlgRIPEMD128.Create(nil);
  hashObjects[hashRIPEMD_160] := THashAlgRIPEMD160.Create(nil);
  hashObjects[hashRIPEMD_256] := THashAlgRIPEMD256.Create(nil);
  hashObjects[hashRIPEMD_320] := THashAlgRIPEMD320.Create(nil);

end;


// Return the current timestamp (high-res)
function GetCurrentTimeStamp(): TTimeStamp;
var
  currTime: SYSTEMTIME;
begin
  GetSystemTime(currTime);

  Result.Date := 0;
  Result.Time := (currTime.wHour*60*60*1000)+
                 (currTime.wMinute*60*1000)+
                 (currTime.wSecond*1000)+
                 (currTime.wMilliseconds);

end;


// Return the difference between two times, prettily formatted.
function FormattedTimeDiff(beginTime: TTimeStamp; endTime: TTimeStamp): string;
var
  duration: cardinal;
begin
  duration := endTime.time - beginTime.time;
  Result := Format('%.2d:%.2d.%.3d', [((duration div 1000) div 60),
                                 ((duration div 1000) mod 60),
                                 (duration mod 1000)]);
end;


begin
  if (ParamCount<1) then
    begin
    exeFilename := ExtractFilename(ParamStr(0));
    exeFilenameExt := ExtractFileExt(exeFilename);
    delete(exeFilename, length(exeFilename)-length(exeFilenameExt)+1, length(exeFilenameExt));

    writeln('SDHash: Sarah Dean''s Hash Generator (v1.2)');
    writeln('');
    writeln('Usage:');
    writeln('  '+exeFilename+' <hashtype>... [/timed] [/lowercase] [/recursive] <filename>');
    writeln('  '+exeFilename+' <hashtype>... [/timed] [/lowercase] /test <test vector>');
    writeln('');
    writeln('  Where <hashtype> is zero or more of:');
    for hashLoop:=low(hashObjects) to high(hashObjects) do
      begin
      writeln('    /'+lowercase(fhHashNames[hashLoop])+' ');
      end;
    writeln('');
    writeln('  <filename> is either a filename or filemask.');
    writeln('');
    writeln('  /timed - optionally display time taken to compute each hash, and total');
    writeln('           execution time [mins:secs.ms]');
    writeln('');
    writeln('  /lowercase - display hashes in lowercase');
    writeln('');
    writeln('  e.g.');
    writeln('    '+exeFilename+' /MD5 /SHA-1 aFile.txt');
    writeln('    '+exeFilename+' /MD5 /SHA-1 /recursive *.*');
    writeln('    '+exeFilename+' /MD5 /test 1234567890');
    writeln('');
    exit;
    end
  else
    begin
    progStartTime := GetCurrentTimeStamp();
    doTestVector := SDUCommandLineSwitch('test');
    doRecursive := SDUCommandLineSwitch('recursive');
    doTimed := SDUCommandLineSwitch('timed');
    doLowercase := SDUCommandLineSwitch('lowercase');

    fileSpec := ParamStr(ParamCount);

    if doTestVector then
      begin
      fileIterator := nil;
      cntFiles := 0;
      end
    else
      begin
      fileIterator := TSDUFileIterator.Create(nil);
      fileIterator.Sorted := TRUE;
      fileIterator.IncludeSubDirs := doRecursive;

      fileIterator.FileMask := fileSpec;
      fileIterator.Reset();

      cntFiles := fileIterator.Count;

      if (cntFiles=0) then
        begin
        writeln('File not found.');
        exit;
        end;
      end;

    CreateHashObjects();

    numHashesToDo:= 0;
    for hashLoop:=low(hashObjects) to high(hashObjects) do
      begin
      if SDUCommandLineSwitch(fhHashNames[hashLoop]) then
        begin
        inc(numHashesToDo);
        end;
      end;

    doAllHashes := TRUE;
    if (numHashesToDo>0) then
      begin
      doAllHashes := FALSE;
      end
    else
      begin
      for hashLoop:=low(hashObjects) to high(hashObjects) do
        begin
        inc(numHashesToDo);
        end;
      end;


    if doTestVector then
      begin
      for hashLoop:=low(hashObjects) to high(hashObjects) do
        begin
        if SDUCommandLineSwitch(fhHashNames[hashLoop]) OR
           doAllHashes                                 then
          begin
          hashStartTime := GetCurrentTimeStamp();
          hash := hashObjects[hashLoop].HashString(fileSpec);
          
          if numHashesToDo>1 then
            begin
            write(Format('%-5s', [fhHashNames[hashLoop]])+': ');
            end;

          if doTimed then
            begin
            write(' ['+FormattedTimeDiff(hashStartTime, GetCurrentTimeStamp())+'] ');
            end;

          if doLowercase then
            begin
            writeln(lowercase(hashObjects[hashLoop].HashToDisplay(hash)));
            end
          else
            begin
            writeln(uppercase(hashObjects[hashLoop].HashToDisplay(hash)));
            end;
          end;
        end;
      end
    else
      begin
      filename := fileIterator.Next();
      while (filename<>'') do
        begin
        for hashLoop:=low(hashObjects) to high(hashObjects) do
          begin
          if SDUCommandLineSwitch(fhHashNames[hashLoop]) OR
             doAllHashes                                 then
            begin
            hashStartTime := GetCurrentTimeStamp();
            hashObjects[hashLoop].HashFile(filename, hash);

            if cntFiles>1 then
              begin
              write(filename+': ');
              end;

            if numHashesToDo>1 then
              begin
              write(Format('%-5s', [fhHashNames[hashLoop]])+': ');
              end;

            if doTimed then
              begin
              write(' ['+FormattedTimeDiff(hashStartTime, GetCurrentTimeStamp())+'] ');
              end;

            if doLowercase then
              begin
              writeln(lowercase(hashObjects[hashLoop].HashToDisplay(hash)));
              end
            else
              begin
              writeln(uppercase(hashObjects[hashLoop].HashToDisplay(hash)));
              end;
            end;
          end;
        filename := fileIterator.Next();
        end;
      end;

    for hashLoop:=low(hashObjects) to high(hashObjects) do
      begin
      hashObjects[hashLoop].Free();
      end;

    if doTimed then
      begin
      writeln('Program execution: '+FormattedTimeDiff(progStartTime, GetCurrentTimeStamp));
      end;

    end;

END.

