unit Shredder;
// Description: File/Disk Free Space Shredder (overwriter)
// By Sarah Dean
//
// -----------------------------------------------------------------------------
//

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  SDUGeneral, SDUProgress_U,
  FileList_U;

type
  TShredDetails = array [0..2] of byte;
  TShredBlock = array of byte;
  TShredFreeSpaceBlock = array of byte;

  TShredFreeSpaceBlockObj = class
  public
    BLANK_FREESPACE_BLOCK: TShredFreeSpaceBlock;
  end;

  TShredder = class(TComponent)
  private
    procedure GetGutmannBlock(passNum: integer; var outputBlock: TShredBlock);
    function  GetGutmannChars(passNum: integer): TShredDetails;
    function  SetGutmannDetails(nOne: integer; nTwo: integer; nThree: integer): TShredDetails;
    procedure CreateEmptyFile(filename: string; size: int64; blankArray: TShredFreeSpaceBlockObj);
    procedure DeleteTempFiles(tempDriveDir: string; numberOfFiles: integer);
    function  GetTempFilename(driveDir: string; serialNo: integer): string;
    function  WipeFileSlacks(dirName: string; progressDlg: TSDUProgress_F; problemFiles: TStringList): integer;
    function  WipeSpecifiedFileSlack(filename: string): boolean;
    function  CountFiles(dirName: string): integer;
    function  GenerateRndDotFilename(path: string; origFilename: string): string;
  protected
    { Protected declarations }
  public
    // General Shredder Methods
    OptShredFileDirUseInt: boolean;
    OptShredFreeUseInt: boolean;
    OptShredSlack: boolean;
    OptShredExtFileExe : string;
    OptShredExtDirExe : string;
    OptShredExtFreeSpaceExe: string;
    OptShredExtDirDiff : boolean; // TRUE if should use different command line for shredding dirs
    OptShredExtShredFilesThenDir : boolean; // TRUE if should shred all files in a dir before shredding the dir
    OptShredIntFirstBytes : longint;
    OptShredIntPasses : integer;
    OptShredIntGutmann: boolean;
    OptShredIntShowProgress: boolean;
    OptShredIntFreeSpcFileSize: integer;
    OptShredIntFreeSpcFileCreationBlkSize: integer;
    OptShredIntFreeSpcBufferSize: integer;
    OptShredIntFileBufferSize: integer;
    OptShredIntResetFileDateStamps: boolean;

    function  ShredFreeSpace(driveLetter: char; silent: boolean = FALSE): integer;
    function  WipeAllFileSlacks(driveLetter: char; silent: boolean = FALSE): integer;
    procedure ShredDir(dirname: string; showProgress: boolean);
    procedure DestroyFileOrDir(itemname : string; quickShred: boolean; showProgress: boolean);
    procedure DeleteFileOrDir(itemname: string);
    function  InternalShredFile(filename : string; quickShred: boolean; showProgress: boolean; leaveFile: boolean = FALSE): integer;
    procedure DestroyRegKey(key: string);
  published


  end;

procedure Register;

implementation

uses Math,
     SDUFileIterator_U, SDUDirIterator_U,
     Registry;

procedure Register;
begin
  RegisterComponents('SDeanSecurity', [TShredder]);
end;

// Destroy a given file/dir, using the method specified in the INI file
// NOTE: When destroying dirs, QUICKSHRED IS ALWAYS FALSE - set in the dir
//       shredding procedure
// itemname       - the file/dir to be destroyed
// quickShred - ignored if not using internal shredding, otherwise if set to
//              FALSE then delete whole file; setting it to TRUE will only
//              delete the first n bytes
procedure TShredder.DestroyFileOrDir(itemname : string; quickShred: boolean; showProgress: boolean);
var
  fileAttributes : integer;
  shredderCommandLine : string;
begin
  // Remove any hidden, system or readonly file attribs
  fileAttributes := FileGetAttr(itemname);
  fileAttributes := fileAttributes AND not(faReadOnly);
  fileAttributes := fileAttributes AND not(faHidden);
  fileAttributes := fileAttributes AND not(faSysFile);
  FileSetAttr(itemname, fileAttributes);

  if (fileAttributes AND faDirectory)<>0 then
    begin
    ShredDir(itemname, showProgress);
    end
  else
    begin
    itemname := SDUConvertLFNToSFN(itemname);
    if OptShredFileDirUseInt then
      begin
      InternalShredFile(itemname, quickShred, showProgress);
      end
    else
      begin
      shredderCommandLine := format(OptShredExtFileExe, [itemname]);
      winexec(PChar(shredderCommandLine), SW_MINIMIZE);
      end;
    end;

end;

// leaveFile - set to TRUE to just leave the file, after shredding; don't
//             delete it (default = FALSE)
// Returns: -1 on error
//          -2 on user cancel
//          1 On success
function TShredder.InternalShredFile(filename : string;
                                     quickShred: boolean;
                                     showProgress: boolean;
                                     leaveFile: boolean = FALSE): integer;
var
  fileHandle : THandle;
  i: integer;
  j : integer;
  blankingBytes: TShredBlock;
  bytesToShred: longint;
  bytesToShredHi: DWORD;
  bytesWritten: DWORD;
  bytesInBlock: DWORD;
  numPasses: integer;
  progressDlg: TSDUProgress_F;
  totalBlocks: integer;
  progressTitle: string;
begin
  bytesToShred := -1;
  bytesInBlock := OptShredIntFileBufferSize;
  SetLength(blankingBytes, OptShredIntFileBufferSize);
  randomize();

  numPasses := OptShredIntPasses;
  if OptShredIntGutmann then
    begin
    numPasses := 35;
    end;
  progressDlg:= TSDUProgress_F.create(nil);
  try
    if showProgress then
      begin
      progressDlg.Show();
      end;
    for i:=1 to numPasses do
      begin
      progressTitle := 'Shredding (';

      // Fill a block with random garbage - needs to be random in order to
      // increase chances of successful overwriting on compressed drives;
      // in which case, make OptShredIntFileBufferSize as large as
      // possible.
      if OptShredIntGutmann then
        begin
        progressTitle := progressTitle + 'Gutmann ';
        GetGutmannBlock(i, blankingBytes);
        end
      else
        begin
        GetGutmannBlock(1, blankingBytes); // random pass on Gutmann
        end;

      progressDlg.Caption := progressTitle + 'pass '+inttostr(i)+'/'+inttostr(numPasses)+') '+filename;

      try
        fileHandle := CreateFile(PChar(filename),
                                 GENERIC_READ or GENERIC_WRITE,
                                 0,
                                 nil,
                                 OPEN_EXISTING,
                                 FILE_ATTRIBUTE_NORMAL or FILE_FLAG_WRITE_THROUGH,
                                 0);
        try
          if bytesToShred=-1 then
            begin
            // WARNING - THIS CAN ONLY SHRED FILES OF LESS THAN 4GB
            bytesToShred := GetFileSize(fileHandle, @bytesToShredHi);
            end;
          if quickShred then
            begin
            bytesToShred := min(bytesToShred, OptShredIntFirstBytes);
            end;

          if bytesToShredHi>0 then
            begin
            CloseHandle(fileHandle);
            Result := -1;
            exit;
            end;

          SetFilePointer(fileHandle, 0, nil, FILE_BEGIN);
          totalBlocks := (bytesToShred div OptShredIntFileBufferSize)+1;
          progressDlg.i64Min := 0;
          progressDlg.i64Max := totalBlocks;
          progressDlg.i64Position := 0;

          for j:=1 to totalBlocks do
            begin
            WriteFile(fileHandle, blankingBytes[0], bytesInBlock, bytesWritten, nil);
            // Ensure that the buffer is flushed to disk (even through disk caching
            // software) [from Borland FAQ]
            FlushFileBuffers(fileHandle);
            if showProgress then
              begin
              progressDlg.i64IncPosition();
              Application.ProcessMessages();
              if progressDlg.Cancel then
                begin
                Result := -2;
                CloseHandle(fileHandle);
                exit;
                end;
              end;
            end;
        finally
          FlushFileBuffers(fileHandle);
          CloseHandle(fileHandle);
        end;
      except
        // Nothing (i.e. ignore all exceptions, e.g. can't open file)
      else
        // Nothing (i.e. ignore all exceptions, e.g. can't open file)
      end;

      end; // for each pass
  finally
    progressDlg.Free();
  end;

  if not(leaveFile) then
    begin
    DeleteFileOrDir(filename);
    end;

  Result := 1;

end;

// Rename a file/dir and then delete it.
procedure TShredder.DeleteFileOrDir(itemname: string);
const
  // This should be enough to overwrite any LFN directory entries (about 255 chars long)
  MASSIVE_FILENAME = 'aaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaaa.aaa';
var
  j: integer;
  deleteFilename: string;
  testRenameFilename: string;
  fileAttributes: integer;
  largeFilename: string;
  fileHandle : THandle;
  tmpFsp: string;
begin
  try
    tmpFsp := GenerateRndDotFilename(ExtractFilePath(itemname), ExtractFileName(itemname));
    tmpFsp := ExtractFilePath(itemname)+tmpFsp;
    if tmpFsp<>'' then
      begin
      if RenameFile(itemname, tmpFsp) then
        begin
        itemname := tmpFsp;
        end;
      end;

    deleteFilename := itemname;
    if (Win32Platform=VER_PLATFORM_WIN32_NT) then
      begin
      for j:=ord('a') to ord('z') do
        begin
        testRenameFilename := ExtractFilePath(itemname) + chr(j)+'.';
        if not(fileexists(testRenameFilename)) then
          begin
          deleteFilename := testRenameFilename;
          break;
          end;
        end;

      largeFilename := ExtractFilePath(itemname) + MASSIVE_FILENAME;
      if length(largeFilename)>MAX_PATH then
        begin
        Delete(largeFilename, MAX_PATH-1, length(largeFilename)-MAX_PATH+1);
        end;


      if RenameFile(itemname, largeFilename) then
        begin
        if not(RenameFile(largeFilename, deleteFilename)) then
          begin
          deleteFilename := largeFilename;
          end;
        end
      else
        begin
        deleteFilename := itemname;
        end;;

      end;

    fileAttributes := FileGetAttr(deleteFilename);
    if (fileAttributes AND faDirectory)=0 then
      begin
      // Truncate the file to 0 bytes
      fileHandle := CreateFile(PChar(deleteFilename),
                               GENERIC_READ or GENERIC_WRITE,
                               0,
                               nil,
                               OPEN_EXISTING,
                               FILE_ATTRIBUTE_NORMAL or FILE_FLAG_WRITE_THROUGH,
                               0);
      SetFilePointer(fileHandle, 0, nil, FILE_BEGIN);
      SetEndOfFile(fileHandle);

      if OptShredIntResetFileDateStamps then
        begin
        SDUSetAllFileTimes(fileHandle, DateTimeToFileDate(encodedate(1980, 1, 1)));
        end;

      CloseHandle(fileHandle);

      DeleteFile(deleteFilename);
      end
    else
      begin
      RemoveDir(deleteFilename);
      end;
  except
    // Nothing (i.e. ignore all exceptions, e.g. can't open file)
  else
    // Nothing (i.e. ignore all exceptions, e.g. can't open file)
  end;

end;

procedure TShredder.GetGutmannBlock(passNum: integer; var outputBlock: TShredBlock);
var
  i: integer;
  passDetails: TShredDetails;
begin
  if (passNum<5) OR (passNum>31) then
    begin
    randomize;
    for i:=0 to OptShredIntFileBufferSize-1 do
      begin
      outputBlock[i] := random(256);
      end;
    end
  else
    begin
    passDetails := GetGutmannChars(passNum);
    for i:=0 to OptShredIntFileBufferSize-1 do
      begin
      outputBlock[i] := passDetails[i mod 3];
      end;
    end;

end;

function TShredder.GetGutmannChars(passNum: integer): TShredDetails;
begin
  case passnum of
   5: Result := SetGutmannDetails($55, $55, $55);
   6: Result := SetGutmannDetails($aa, $aa, $aa);
   7: Result := SetGutmannDetails($92, $49, $24);
   8: Result := SetGutmannDetails($49, $24, $92);
   9: Result := SetGutmannDetails($24, $92, $49);
  10: Result := SetGutmannDetails($00, $00, $00);
  11: Result := SetGutmannDetails($11, $11, $11);
  12: Result := SetGutmannDetails($22, $22, $22);
  13: Result := SetGutmannDetails($33, $33, $33);
  14: Result := SetGutmannDetails($44, $44, $44);
  15: Result := SetGutmannDetails($55, $55, $55);
  16: Result := SetGutmannDetails($66, $66, $66);
  17: Result := SetGutmannDetails($77, $77, $77);
  18: Result := SetGutmannDetails($88, $88, $88);
  19: Result := SetGutmannDetails($99, $99, $99);
  20: Result := SetGutmannDetails($aa, $aa, $aa);
  21: Result := SetGutmannDetails($bb, $bb, $bb);
  22: Result := SetGutmannDetails($cc, $cc, $cc);
  23: Result := SetGutmannDetails($dd, $dd, $dd);
  24: Result := SetGutmannDetails($ee, $ee, $ee);
  25: Result := SetGutmannDetails($ff, $ff, $ff);
  26: Result := SetGutmannDetails($92, $49, $24);
  27: Result := SetGutmannDetails($49, $24, $92);
  28: Result := SetGutmannDetails($24, $92, $49);
  29: Result := SetGutmannDetails($6d, $b6, $db);
  30: Result := SetGutmannDetails($b6, $db, $6d);
  31: Result := SetGutmannDetails($db, $6d, $b6);
  end;

end;

function TShredder.SetGutmannDetails(nOne: integer; nTwo: integer; nThree: integer): TShredDetails;
begin
  Result[0] := nOne;
  Result[1] := nTwo;
  Result[2] := nThree;

end;

// Returns: -1 on error
//          -2 on user cancel
//          1 On success
function TShredder.ShredFreeSpace(driveLetter: char; silent: boolean): integer;
var
  drive: string;
  tempDriveDir: string;
  freeSpace: int64;
  fileNumber: integer;
  currFilename: string;
  blankArray: TShredFreeSpaceBlockObj;
  i: integer;
  lastFilename: string;
  shredderCommandLine: string;
  progressDlg: TSDUProgress_F;
  diskNumber: integer;
begin
  Result := -1;
  if not(OptShredFreeUseInt) then
    begin
    shredderCommandLine := format(OptShredExtFreeSpaceExe, [driveLetter]);
    if (WinExec(PChar(shredderCommandLine), SW_RESTORE))<31 then
      begin
      messageDlg('Error running external (3rd party) free space shredder',
                 mtError,
                 [mbOK],
                 0);
      end;
    end
  else
    begin
    progressDlg := TSDUProgress_F.Create(nil);
    blankArray := TShredFreeSpaceBlockObj.Create();
    try
      SetLength(blankArray.BLANK_FREESPACE_BLOCK, OptShredIntFreeSpcFileCreationBlkSize);
      for i:=0 to OptShredIntFreeSpcFileCreationBlkSize-1 do
        begin
        blankArray.BLANK_FREESPACE_BLOCK[i] := 0;
        end;

      // create a subdir
      drive := uppercase(driveLetter);
      driveLetter := drive[1];
      randomize();
      tempDriveDir := driveLetter + ':\~STUfree'+inttostr(random(10000))+'.tmp';
      diskNumber := ord(drive[1])-ord('A')+1;

      CreateDir(tempDriveDir);
      fileNumber := 0;

      // While there is OptShredIntFreeSpcFileSize bytes diskspace
      // left, create a file OptShredIntFreeSpcFileSize big
      freeSpace := DiskFree(diskNumber);
      progressDlg.Caption := 'Shredding free space on drive '+driveLetter+':';
      progressDlg.i64Max := freeSpace;
      progressDlg.i64Min := 0;
      progressDlg.i64Position := 0;
      if not(silent) then
        begin
        progressDlg.Show();
        end;
      // xxx should this be >= and  not > ?
      while (freeSpace>OptShredIntFreeSpcFileSize) do
        begin
        inc(fileNumber);
        currFilename := GetTempFilename(tempDriveDir, fileNumber);
        CreateEmptyFile(currFilename, OptShredIntFreeSpcFileSize, blankArray);

        // Shred the file, but _don't_ _delete_ _it_
        InternalShredFile(currFilename, FALSE, FALSE, TRUE);
        // Slack on that file needs to be wiped, as the internal shred may have
        // increased the size of the file, giving file slack that has not been
        // shredded
        WipeSpecifiedFileSlack(currFilename);

        freeSpace := DiskFree(diskNumber);
        progressDlg.i64InversePosition := freeSpace;
        if not(silent) then
          begin
          Application.ProcessMessages();
          if progressDlg.Cancel then
            begin
            Result := -2;
            // Remove any files that were created, together with the dir
            DeleteTempFiles(tempDriveDir, fileNumber);
            DeleteFileOrDir(tempDriveDir);
            exit;
            end;
          end;
        end;

      // Create a file with the remaining disk bytes
      lastFilename := GetTempFilename(tempDriveDir, fileNumber+1);
      CreateEmptyFile(lastFilename, freeSpace, blankArray);

      // Delete all the other files, to ensure that when this one's shredded,
      // the drive doesn't run out of space due to
      // OptShredIntFileBufferSize overwriting beyond the file
      // length to overwrite it's slack space
      DeleteTempFiles(tempDriveDir, fileNumber);

      // Shred the last file
      InternalShredFile(lastFilename, FALSE, FALSE, FALSE);
      // Slack on that file needs to be wiped, as the internal shred may have
      // increased the size of the file, giving file slack that has not been
      // shredded
      WipeSpecifiedFileSlack(lastFilename);

      // Remove the temp dir
      DeleteFileOrDir(tempDriveDir);
    finally
      progressDlg.Free();
      blankArray.Free();
    end;

  Result := 1;
  end; // use internal free space shredder

end;

procedure TShredder.DeleteTempFiles(tempDriveDir: string; numberOfFiles: integer);
var
  currFilename: string;
  i: integer;
begin
  for i:=1 to numberOfFiles do
    begin
    currFilename := GetTempFilename(tempDriveDir, i);
    DeleteFileOrDir(currFilename);
    end;
end;


procedure TShredder.CreateEmptyFile(filename: string; size: int64; blankArray: TShredFreeSpaceBlockObj);
var
  fileHandle : THandle;
  bytesWritten: DWORD;
  bytesInBlock: DWORD;
  totalBytesWritten: integer;
begin
  fileHandle := CreateFile(PChar(filename),
                           GENERIC_READ or GENERIC_WRITE,
                           0,
                           nil,
                           CREATE_ALWAYS,
                           FILE_ATTRIBUTE_NORMAL or FILE_FLAG_WRITE_THROUGH,
                           0);

  // Fill out the file to the required size
  totalBytesWritten := 0;
  while (totalBytesWritten<size) do
    begin
    bytesInBlock := min((size-totalBytesWritten), OptShredIntFreeSpcFileCreationBlkSize);
    WriteFile(fileHandle, blankArray.BLANK_FREESPACE_BLOCK[0], bytesInBlock, bytesWritten, nil);
    inc(totalBytesWritten, bytesWritten);
    end;

  // Ensure that the buffer is flushed to disk (even through disk caching
  // software) [from Borland FAQ]
  FlushFileBuffers(fileHandle);

  CloseHandle(fileHandle);
end;


function TShredder.GetTempFilename(driveDir: string; serialNo: integer): string;
begin
  Result := driveDir + '\~STUtmp.'+inttostr(serialNo);
end;



// Returns: -1 on error
//          -2 on user cancel
//          1 On success
function TShredder.WipeAllFileSlacks(driveLetter: char; silent: boolean): integer;
var
  progressDlg: TSDUProgress_F;
  rootDir: string;
  problemFiles: TStringList;
  reportDlg: TFileList_F;
  drive: string;
begin
  drive := uppercase(driveLetter);
  driveLetter := drive[1];

  progressDlg := TSDUProgress_F.Create(nil);
  problemFiles:= TStringList.create();
  try
    rootDir:= driveLetter+':\';

    progressDlg.Caption := 'Shredding file slack on drive '+driveLetter+':';
    progressDlg.i64Max := CountFiles(rootDir);
    progressDlg.i64Min := 0;
    progressDlg.i64Position := 0;
    if not(silent) then
      begin
      progressDlg.Show();
      end;

    Result := WipeFileSlacks(rootDir, progressDlg, problemFiles);

    if not(silent) AND (problemFiles.count>0) then
      begin
      reportDlg := TFileList_F.Create(nil);
      try
        reportDlg.lbFiles.visible := TRUE;
        reportDlg.lblTitle.caption := 'The following files could not have their slack space shredded:';
        reportDlg.lbFiles.items.assign(problemFiles);
        reportDlg.showmodal;
      finally
        reportDlg.Free();
      end;
      end;
  finally
    problemFiles.Free();
    progressDlg.Free();
  end;

end;

// Perform file slack shredding
// Returns: -1 on error
//          -2 on user cancel
//          1 on success
function TShredder.WipeFileSlacks(dirName: string; progressDlg: TSDUProgress_F; problemFiles: TStringList): integer;
var
  slackFile: string;
  fileIterator: TSDUFileIterator;
  currFile: string;
begin
  fileIterator := TSDUFileIterator.Create(nil);
  try
    fileIterator.Directory := dirName;
    fileIterator.IncludeSubDirs := TRUE;
    fileIterator.Reset();

    currFile := fileIterator.Next();
    while currFile<>'' do
      begin
      slackFile := SDUConvertLFNToSFN(currFile);
      if not(WipeSpecifiedFileSlack(slackFile)) then
        begin
        problemFiles.add(slackFile);
        end;

      progressDlg.i64IncPosition();
      if ProgressDlg.Cancel then
        begin
        Result := -2;
        exit;
        end;
      currFile := fileIterator.Next();
      end;

  finally
    fileIterator.Free();
  end;

  Result := 1;
end;


function TShredder.WipeSpecifiedFileSlack(filename: string): boolean;
var
  fileHandle: THandle;
  fileLengthLo: DWORD;
  fileLengthHi: DWORD;
  bytesInBlock: integer;
  numPasses: integer;
  i: integer;
  blankingBytes: TShredBlock;
  bytesWritten: DWORD;
  fileDateStamps: integer;
  fileAttributes : integer;
  error: boolean;
begin
  Result := FALSE;
  error := FALSE;

  SetLength(blankingBytes, OptShredIntFileBufferSize);

  // Record any file attributes and remove any hidden, system or readonly file
  // attribs
  fileAttributes := FileGetAttr(filename);
  if fileAttributes=-1 then
    begin
    error:=TRUE;
    end;

  if not(error) then
    begin
    error := (FileSetAttr(filename, faArchive) = -1);
    end;

  // This line not needed, but gets rid of compiler warning
  fileHandle:=INVALID_HANDLE_VALUE;

  if not(error)then
    begin
    fileHandle := CreateFile(PChar(filename),
                             GENERIC_READ or GENERIC_WRITE,
                             0,
                             nil,
                             OPEN_EXISTING,
                             FILE_ATTRIBUTE_NORMAL or FILE_FLAG_WRITE_THROUGH,
                             0);
    error := (fileHandle=INVALID_HANDLE_VALUE);
    end;

  if not(error) then
    begin
    fileDateStamps := FileGetDate(fileHandle);

    fileLengthLo := GetFileSize(fileHandle, @fileLengthHi);

    // xxx - We do not handle files larger than 4GB
    if fileLengthHi<=0 then
      begin
      // Perform the actual slack wipe operation
      bytesInBlock := OptShredIntFileBufferSize;
      randomize();

      numPasses := OptShredIntPasses;
      if OptShredIntGutmann then
        begin
        numPasses := 35;
        end;
      for i:=1 to numPasses do
        begin
        // Fill a block with random garbage - needs to be random in order to
        // increase chances of successful overwriting on compressed drives;
        // in which case, make OptShredIntFileBufferSize as large
        // as possible.
        if OptShredIntGutmann then
          begin
          GetGutmannBlock(i, blankingBytes);
          end
        else
          begin
          GetGutmannBlock(1, blankingBytes); // random pass on Gutmann
          end;

        // Set the file pointer to the end of the file
        SetFilePointer(fileHandle, fileLengthLo, @fileLengthHi, FILE_BEGIN);
        // Write from there, pass data (e.g. a block of Gutmann)
        WriteFile(fileHandle, blankingBytes[0], bytesInBlock, bytesWritten, nil);
        // Ensure that the buffer is flushed to disk (even through disk caching
        // software) [from Borland FAQ]
        FlushFileBuffers(fileHandle);

        end; // for each pass

      // truncate the file to the correct length
      SetFilePointer(fileHandle, fileLengthLo, @fileLengthHi, FILE_BEGIN);
      SetEndOfFile(fileHandle);
      end; // if fileLengthHi<=0 then

    // reset the date/timestamps
    FileSetDate(fileHandle, fileDateStamps);

    // flush and close
    FlushFileBuffers(fileHandle);
    CloseHandle(fileHandle);

    // Reset the file attributes
    FileSetAttr(filename, fileAttributes);

    Result := (fileLengthHi<=0);
    end; // CreateFile opened file file correctly

end;

// Perform dir shredding, using the method specified in the INI file
// filename   - the file to be destroyed
procedure TShredder.ShredDir(dirname: string; showProgress: boolean);
var
  dirToDestroy: string;
  shredderCommandLine: string;
  fileIterator: TSDUFileIterator;
  dirIterator: TSDUDirIterator;
  currFile: string;
  currDir: string;
begin
  if (not(OptShredFileDirUseInt))         AND
     (not(OptShredExtShredFilesThenDir))  AND
     (OptShredExtDirExe<>'')              then
    begin
    // i.e. we using an external shredder which doesn't need all files to be
    // removed before it can be used
    dirname := SDUConvertLFNToSFN(dirname);
    shredderCommandLine := format(OptShredExtDirExe, [dirname]);
    winexec(PChar(shredderCommandLine), SW_MINIMIZE);
    end
  else
    begin
    fileIterator := TSDUFileIterator.Create(nil);
    try
      fileIterator.Directory := dirname;
      fileIterator.IncludeSubDirs := TRUE;
      fileIterator.Reset();

      currFile := fileIterator.Next();
      while currFile<>'' do
        begin
        DestroyFileOrDir(currFile,
                         FALSE,
                         showProgress);
        currFile := fileIterator.Next();
        end;
    finally
      fileIterator.Free();
    end;


    dirIterator := TSDUDirIterator.Create(nil);
    try
      dirIterator.Directory := dirName;
      dirIterator.ReverseFormat := TRUE;
      dirIterator.IncludeStartDir := TRUE;
      dirIterator.Reset();

      // Now do the dir structure
      currDir := dirIterator.Next();
      while currDir<>'' do
        begin
        // And finally, remove the current dir...
        //   if external shredder handles dirs, use it
        //   else pass the dirname to the internal shredder for shredding
        if (not(OptShredFileDirUseInt)) AND
           (OptShredExtDirExe<>'')      then
          begin
          // i.e. we using an external shredder which doesn't need all files to be
          // removed before it can be used
          dirToDestroy := SDUConvertLFNToSFN(currDir);
          shredderCommandLine := format(OptShredExtDirExe, [dirToDestroy]);
          winexec(PChar(shredderCommandLine), SW_MINIMIZE);
          end
        else
          begin
          // Fallback to simply removing the dir using internal method
          DeleteFileOrDir(currDir);
          end;

        currDir := dirIterator.Next();
        end;
    finally
      dirIterator.Free();
    end;

    end; // External shredder not being used/needs files deleted first

end;

procedure TShredder.DestroyRegKey(key: string);
var
  registry: TRegistry;
  NTsubkeys: TStrings;
  keyValues: TStrings;
  valueInfo: TRegDataInfo;
  rootStr: string;
  i: integer;
  j: integer;
  buffer: array of byte;
begin
  registry := TRegistry.create();
  try
    if Pos('HKCR\', key)=1 then
      begin
      registry.RootKey := HKEY_CLASSES_ROOT;
      end
    else if Pos('HKCU\', key)=1 then
      begin
      registry.RootKey := HKEY_CURRENT_USER;
      end
    else if Pos('HKLM\', key)=1 then
      begin
      registry.RootKey := HKEY_LOCAL_MACHINE;
      end
    else if Pos('HKU \', key)=1 then
      begin
      registry.RootKey := HKEY_USERS;
      end
    else if Pos('HKCC\', key)=1 then
      begin
      registry.RootKey := HKEY_CURRENT_CONFIG;
      end
    else if Pos('HKDD\', key)=1 then
      begin
      registry.RootKey := HKEY_DYN_DATA;
      end;

    rootStr := Copy(key, 1, 5);
    Delete(key, 1, 5);

    if (Win32Platform=VER_PLATFORM_WIN32_NT) then
      begin
      NTsubkeys:=TStringList.Create();
      try
        if registry.OpenKey(key, FALSE) then
          begin
          if registry.HasSubkeys() then
            begin
            registry.GetKeyNames(NTsubkeys);
            end;

          keyValues := TStringList.Create();
          try
            registry.GetValueNames(keyValues);
            for i:=0 to (keyValues.count-1) do
              begin
              registry.GetDataInfo(keyValues[i], valueInfo);
              case valueInfo.RegData of
              rdString:
                begin
                registry.WriteString(keyValues[i],
                                      Format('%-'+inttostr(valueInfo.DataSize-1)+'.'+inttostr(valueInfo.DataSize-1)+'s', ['']));
                end;

              rdExpandString:
                begin
                registry.WriteExpandString(keyValues[i],
                                      Format('%-'+inttostr(valueInfo.DataSize-1)+'.'+inttostr(valueInfo.DataSize-1)+'s', ['']));
                end;

              rdInteger:
                begin
                registry.WriteInteger(keyValues[i], 0);
                end;

              rdBinary:
                begin
                setlength(buffer, valueInfo.DataSize);
                for j:=0 to (valueInfo.DataSize-1) do
                  begin
                  buffer[j] := $FF;
                  end;
                registry.WriteBinaryData(keyValues[i],
                                         buffer[0],
                                         valueInfo.DataSize);
                end;

              rdUnknown:
                begin
                // Nada - don't know how to overwrite!
                end;

              else
                begin
                // Nada - don't know how to overwrite!
                end;

              end;
              end;

          finally
            keyValues.Free();
          end;

          registry.CloseKey();

          for i:=0 to (NTsubkeys.count-1) do
            begin
            DestroyRegKey(rootStr+NTsubkeys[i]);
            end;
          end;
      finally
        NTsubkeys.Free();
      end;
      end;

    registry.DeleteKey(key);
  finally
    registry.Free();
  end;

end;

// Generate a random filename of the same length as the one supplied, but
// preserving the last "." in the filename
// Give it 5 tries to find a filename that doesn't already exist, if we don't
// find one, just return ''
function TShredder.GenerateRndDotFilename(path: string; origFilename: string): string;
var
  i: integer;
  fndLastDot: boolean;
  finished: boolean;
  count: integer;
begin
  count := 0;
  finished:= FALSE;
  while not(finished) do
    begin
    fndLastDot := FALSE;
    for i:=length(origFilename) downto 1 do
      begin
      if fndLastDot then
        begin
        origFilename[i] := char(ord('A')+random(26));
        end
      else
        begin
        if origFilename[i]='.' then
          begin
          fndLastDot := TRUE;
          end
        else
          begin
          origFilename[i] := char(ord('A')+random(26));
          end;
        end;
      end; // for i:=length(origFilename) downto 1 do

    finished := not(FileExists(path+origFilename));
    if not(finished) then
      begin
      inc(count);
      if count=5 then
        begin
        origFilename := '';
        end;
      end;
    end; // while not(finished) do

  Result := origFilename;

end;

function TShredder.CountFiles(dirName: string): integer;
var
  fileIterator: TSDUFileIterator;
  cnt: integer;
begin
  fileIterator:= TSDUFileIterator.Create(nil);
  try
    fileIterator.Directory := dirName;
    fileIterator.IncludeSubDirs := TRUE;
    fileIterator.Reset();

    cnt := fileIterator.Count();
    
  finally
    fileIterator.Free();
  end;

  Result := cnt;

end;


END.

