unit OTFE_U;
// Description: Abstract base class for OTF Encryption systems
// By Sarah Dean
//
// Email: sdean12@mailcity.com
// WWW:   http://www.fortunecity.com/skyscraper/true/882/
//
// -----------------------------------------------------------------------------
// Version history:
// 0.0.1b - 12th August 1999
//          Initial draft version
//
// 0.9.0  - 6th November 1999
//          Initial beta release
//


// -----------------------------------------------------------------------------
interface

// -----------------------------------------------------------------------------
uses classes,
     sysUtils,
     OTFEConsts_U;

// -----------------------------------------------------------------------------
type

  TOTFE = class(TComponent)
  private
    { private declarations here}
  protected
    FActive: boolean;
    FLastErrCode: integer; // See OTFEConsts_U.pas
    procedure DestroyString(var theString: string);
    procedure DestroyTStringList(theTStringList: TStringList);
    function  SortString(theString: string): string;

    // Set the component active/inactive
    procedure SetActive(status: Boolean); virtual;

    // Raises exception if the component isn't active
    // Returns FActive (TRUE/FALSE)
    function  CheckActive(): boolean;

    // Work out the order in which to dismount drives, taking into account
    // the potential for one drive to be mounted within another
    // Returns "dismountDrives" in the order in which they should be dismounted
    function  DismountOrder(dismountDrives: string): string;

  public
    constructor Create(AOwner: TComponent); override;
    destructor  Destroy(); override;

    // === Mounting, dismounting drives ========================================

    // Prompt the user for a password (and drive letter if necessary), then
    // mount the specified volume file
    // Returns the drive letter of the mounted volume on success, #0 on failure
    function  Mount(volumeFilename: string; readonly: boolean = FALSE): char; overload; virtual; abstract;

    // As previous Mount, but more than one volumes is specified. Volumes are
    // mounted using the same password
    // Sets mountedAs to the drive letters of the mounted volumes, in order.
    // Volumes that could not be mounted have #0 as their drive letter
    // Returns TRUE if any of the volumes mounted correctly, otherwise FALSE
    function  Mount(volumeFilenames: TStringList; var mountedAs: string; readonly: boolean = FALSE): boolean; overload; virtual; abstract;
    // Example:
    //   Set:
    //     volumeFilenames[0] = c:\test0.dat
    //     volumeFilenames[1] = c:\test1.dat
    //     volumeFilenames[2] = c:\test2.dat
    //   Call Mount described above in which:
    //     volume test0.dat was sucessfully mounted as W:
    //     volume test1.dat failed to mount
    //     volume test2.dat was sucessfully mounted as X:
    //   Then this function should set:
    //     mountedAs = 'W.X' (where '.' is #0)

    // Dismount by volume filename
    // Returns TRUE on success, otherwise FALSE
    function  Dismount(volumeFilename: string; emergency: boolean = FALSE): boolean; overload; virtual; abstract;

    // Dismount by drive letter
    // Returns TRUE on success, otherwise FALSE
    function  Dismount(driveLetter: char; emergency: boolean = FALSE): boolean; overload; virtual; abstract;

    // Dismount all mounted drives
    // Returns a string containing the drive letters for all drives that could not be dismounted
    function  DismountAll(emergency: boolean = FALSE): string; virtual;

    // === Miscellaneous =======================================================

    // Returns TRUE if the underlying driver is installed and the component can
    // be used, otherwise FALSE
    function  IsDriverInstalled(): boolean; overload; virtual; 

    // Returns version ID of underlying driver ($FFFFFFFF on error)
    // (Returns cardinal and not string representation in order to facilitate
    // processing of the version number by the calling function)
    function  Version(): cardinal; virtual; abstract;

    // Returns version ID of underlying driver ('' on error)
    function  VersionStr(): string; virtual; abstract;

    // Returns TRUE if the file specified appears to be an encrypted volume file
    function  IsEncryptedVolFile(volumeFilename: string): boolean; virtual; abstract;

    // Returns a string containing the drive letters (in uppercase) of all
    // mounted drives (e.g. 'DGH') in alphabetical order
    function  DrivesMounted(): string; virtual; abstract;

    // Returns a count of drives mounted (included for completeness)
    function  CountDrivesMounted(): integer; virtual;

    // Returns the volume filename for a given mounted drive
    // (empty string on failure)
    function  GetVolFileForDrive(driveLetter: char): string; virtual; abstract;

    // Returns the drive letter for a mounted volume file
    // (#0 on failure)
    function  GetDriveForVolFile(volumeFilename: string): char; virtual; abstract;

    // Check to see if there are any volume files mounted on the given drive
    // Returns the drive letters of any mounted volumes
    function  VolsMountedOnDrive(driveLetter: char): string; virtual;

    // Test to see if the specified drive is readonly
    // Returns TRUE if the drive is readonly, otherwise FALSE
    function  IsDriveReadonly(driveLetter: char): boolean; virtual;

    // Returns a string describing the last error
    function  GetLastErrorMsg(): string; virtual;

    // Returns a string containing the main executable to the OTF crypto system
    // (the one used mounting, if there's more than one executable)
    // Returns "" on error
    function  GetMainExe(): string; virtual; abstract;

    // Returns a disk-encryption specific record containing information on the
    // specified encrypted drive
    //
    // (Can't represent in the base class, but...)
    // function GetDriveInfo(driveLetter: char): TDiskInfo; virtual; abstract;

  published
    property Active: boolean read FActive write SetActive;
    property LastErrorCode: integer read FLastErrCode write FLastErrCode default OTFE_ERR_SUCCESS;

  end;

// -----------------------------------------------------------------------------
implementation

uses Windows; // needed for UINT

// -----------------------------------------------------------------------------
constructor TOTFE.Create(AOwner: TComponent);
begin
  inherited;
  FActive := FALSE;
  FLastErrCode:= OTFE_ERR_SUCCESS;

end;

// -----------------------------------------------------------------------------
destructor  TOTFE.Destroy();
begin
  inherited;

end;

// -----------------------------------------------------------------------------
// Destroy the contents of the supplied TStringList
procedure TOTFE.DestroyTStringList(theTStringList: TStringList);
var
  i: integer;
  j: integer;
begin
  randomize();
  for i:=0 to (theTStringList.Count-1) do
    begin
    for j:=0 to length(theTStringList[i]) do
      begin
      theTStringList[i] := format('%'+inttostr(length(theTStringList[i]))+'s', [chr(random(255))]);
      end;
    end;

  theTStringList.Clear();

end;

// -----------------------------------------------------------------------------
// Destroy the contents of the supplied string
procedure TOTFE.DestroyString(var theString: string);
var
  i: integer;
begin
  randomize();
  for i:=0 to length(theString) do
    begin
    theString := format('%'+inttostr(length(theString))+'s', [chr(random(255))]);
    end;

end;

// -----------------------------------------------------------------------------
// Returns a string describing the last error
// If no error, returns an empty string
function TOTFE.GetLastErrorMsg(): string;
begin
  Result := ''; // OTFE_ERR_SUCCESS

  case FLastErrCode of

    // See OTFEConsts_U.pas for descriptions

    OTFE_ERR_NOT_ACTIVE                : Result := 'Component not active';
    OTFE_ERR_DRIVER_FAILURE            : Result := 'Driver failure';
    OTFE_ERR_USER_CANCEL               : Result := 'User cancelled operation';
    OTFE_ERR_WRONG_PASSWORD            : Result := 'Wrong password entered';
    OTFE_ERR_VOLUME_FILE_NOT_FOUND     : Result := 'Volume file not found';
    OTFE_ERR_INVALID_DRIVE             : Result := 'Invalid drive';
    OTFE_ERR_MOUNT_FAILURE             : Result := 'Mount failure';
    OTFE_ERR_DISMOUNT_FAILURE          : Result := 'Dismount failure';
    OTFE_ERR_FILES_OPEN                : Result := 'Files open on volume';
    OTFE_ERR_STREAMING_DATA            : Result := 'Can''t dismount while still streaming data, or was doing so in the last few seconds';
    OTFE_ERR_FILE_NOT_ENCRYPTED_VOLUME : Result := 'File is not an encrypted volume';
    OTFE_ERR_UNABLE_TO_LOCATE_FILE     : Result := 'Unable to locate file';
    OTFE_ERR_DISMOUNT_RECURSIVE        : Result := 'Dismounting recursivly mounted drive';
    OTFE_ERR_INSUFFICENT_RIGHTS        : Result := 'Insufficient rights';
    OTFE_ERR_NOT_W9X                   : Result := 'Operation not available under Windows 95/98/ME';
    OTFE_ERR_NOT_WNT                   : Result := 'Operation not available under Windows NT/2000';
    OTFE_ERR_NO_FREE_DRIVE_LETTERS     : Result := 'There are no free drive letters';

    // BestCrypt
    OTFE_ERR_UNKNOWN_ALGORITHM         : Result := 'Unknown algorithm';
    OTFE_ERR_UNKNOWN_KEYGEN            : Result := 'Unknown key generator';

    // ScramDisk
    OTFE_ERR_UNABLE_MOUNT_COMPRESSED   : Result := 'Can''t mount compressed volume';

    // PANIC!
    OTFE_ERR_UNKNOWN_ERROR             : Result := 'Unknown error';
  end;

end;

// -----------------------------------------------------------------------------
// Returns the number of drives mounted
function TOTFE.CountDrivesMounted(): integer;
begin
  FLastErrCode:= OTFE_ERR_SUCCESS;
  Result := length(DrivesMounted());

end;

// -----------------------------------------------------------------------------
// Dismounts all mounted drives
function TOTFE.DismountAll(emergency: boolean = FALSE): string;
var
  drvsMounted: string;
  badUnmount: string;
  i: integer;
begin
  FLastErrCode:= OTFE_ERR_SUCCESS;
  drvsMounted := DrivesMounted();
  badUnmount := '';

  // Dismount in correct order!
  drvsMounted := DismountOrder(drvsMounted);

  for i:=1 to length(drvsMounted) do
    begin
    if not(Dismount(drvsMounted[i], emergency)) then
      begin
      FLastErrCode:= OTFE_ERR_DISMOUNT_FAILURE;
      badUnmount := badUnmount + drvsMounted[i];
      end;
    end;

  Result := badUnmount;

end;

// -----------------------------------------------------------------------------
// Returns TRUE if the underlying driver is installed and the component can
// be used, otherwise FALSE
function TOTFE.IsDriverInstalled(): boolean;
begin
  Result := TRUE;
  FLastErrCode:= OTFE_ERR_SUCCESS;

  if not(Active) then
    begin
    try
      Active := TRUE;
    except
    end;

    Result := Active;
    Active := FALSE;
    end;

  if not(Result) then
    begin
    FLastErrCode:= OTFE_ERR_DRIVER_FAILURE;
    end;

end;

// -----------------------------------------------------------------------------
// Set the component active/inactive
// [IN] status - the status of the component
procedure TOTFE.SetActive(status: Boolean);
begin
  FLastErrCode:= OTFE_ERR_SUCCESS;
  FActive := status;

end;

// -----------------------------------------------------------------------------
// Sort the contents of a string into alphabetical order, assuming each char
// only appears once.
function TOTFE.SortString(theString: string): string;
var
  output: string;
  i: integer;
begin
  output := '';
  for i:=ord('A') to ord('Z') do
    begin
    if pos(chr(i), theString)>0 then
      begin
      output := output + chr(i);
      end;
    end;

  Result := output;

end;

// -----------------------------------------------------------------------------
// Raises exception if the component isn't active
// Returns FActive (TRUE/FALSE)
function TOTFE.CheckActive(): boolean;
begin
  FLastErrCode:= OTFE_ERR_SUCCESS;
  if not(FActive) then
    begin
    FLastErrCode := OTFE_ERR_NOT_ACTIVE;
    raise EOTFEException.Create(OTFE_EXCPT_NOT_ACTIVE);
    end;

  Result := FActive;

end;

// -----------------------------------------------------------------------------
// Check to see if there are any volume files mounted on the given drive
// Returns the drive letters of any mounted volumes
function TOTFE.VolsMountedOnDrive(driveLetter: char): string;
var
  mountedDrives: string;
  volFilename: string;
  retval: string;
  i: integer;
begin
  retval := '';

  driveLetter := upcase(driveLetter);

  mountedDrives := DrivesMounted();

  for i:=1 to length(mountedDrives) do
    begin
    volFilename := GetVolFileForDrive(mountedDrives[i]);
    volFilename := uppercase(volFilename);

    if (length(volFilename)>=3) then
      begin
      if (volFilename[2]=':') and
         (volFilename[3]='\') or (volFilename[3]='/') then
        begin
        if (volFilename[1]=driveLetter) then
          begin
          retVal := retVal + mountedDrives[i];
          end;
        end;

      end;
    end;

  Result := retval;

end;

// -----------------------------------------------------------------------------
// Work out the order in which to dismount drives, taking into account
// the potential for one drive to be mounted within another
// Returns "dismountDrives" in the order in which they should be dismounted
function TOTFE.DismountOrder(dismountDrives: string): string;
var
  unorderedDrives: string;
  unorderedDrivesHosts: string;
  sortedDrives: string;
  sortedDrivesHosts: string;
  tmpDrives: string;
  tmpDrivesHosts: string;
  i: integer;
  volFilename: string;
  hostDrive: char;
begin
  unorderedDrives := dismountDrives;
  // Get all host drive letters for recursivly mounted drives
  unorderedDrivesHosts := '';
  for i:=1 to length(dismountDrives) do
    begin
    volFilename := GetVolFileForDrive(dismountDrives[i]);
    volFilename := uppercase(volFilename);

    hostDrive := #0;
    if (length(volFilename)>=3) then
      begin
      if (volFilename[2]=':') and
         (volFilename[3]='\') or (volFilename[3]='/') then
        begin
        hostDrive := volFilename[1];
        end;

      end;

    unorderedDrivesHosts := unorderedDrivesHosts + hostDrive;
    end;

  // Finally, get the drives into the order in which they should be dismounted
  // in...
  sortedDrives := '';
  sortedDrivesHosts:= '';
  while (length(unorderedDrives)>0) do
    begin
    tmpDrives := '';
    tmpDrivesHosts:= '';

    for i:=1 to length(unorderedDrives) do
      begin
      if ((Pos(unorderedDrivesHosts[i], dismountDrives)=0) or
          (Pos(unorderedDrivesHosts[i], sortedDrives)>0)) then
        begin
        sortedDrives := unorderedDrives[i] + sortedDrives;
        sortedDrivesHosts := unorderedDrivesHosts[i] + sortedDrivesHosts;
        end
      else
        begin
        tmpDrives := unorderedDrives[i] + tmpDrives;
        tmpDrivesHosts := unorderedDrivesHosts[i] + tmpDrivesHosts;
        end;

      end;

    unorderedDrives := tmpDrives;
    unorderedDrivesHosts := tmpDrivesHosts;

    end;

  Result := sortedDrives;

end;

// -----------------------------------------------------------------------------
// Test to see if the specified drive is readonly
// Returns TRUE if the drive is readonly, otherwise FALSE
function  TOTFE.IsDriveReadonly(driveLetter: char): boolean;
  function GenerateRandomFilename(): string;
    var
      outputFilename: string;
      i: integer;
    begin
      outputFilename := '';
      for i:=1 to 20 do
        begin
        outputFilename := outputFilename + chr(ord('A')+random(26));
        end;

      Result := outputFilename;
    end;

var
  F: TextFile;
  OldMode: UINT;
  I: Integer;
  rndFilename: string;
begin
  Result := FALSE;
  // This function has been tested, and works OK if the disk is full
  // This function has been tested, and works OK if the root directory is full of
  // files, and no more can be added

  randomize();
  rndFilename := driveLetter + ':\'+ GenerateRandomFilename();
  while FileExists(rndFilename) do
    begin
    rndFilename := driveLetter + ':\'+ GenerateRandomFilename();
    end;

  OldMode := SetErrorMode(SEM_FAILCRITICALERRORS);
{$I-}
  AssignFile(F, rndFilename);  // File you want to write to here.
  Rewrite(F);
{$I+}
  I := IOResult;
  SetErrorMode(OldMode);
  if I <> 0 then
    begin
    Result := ((I AND $FFFFFFFF)=ERROR_WRITE_PROTECT);
    end
  else
    begin
    CloseFile(F);
    DeleteFile(PChar(rndFilename));
    end;

end;

// -----------------------------------------------------------------------------

END.


