// -*- C++ -*-
#include "Rivet/Analysis.hh"
#include "Rivet/Projections/ChargedFinalState.hh"

namespace Rivet {


  /// @brief Charged multiplicity below Z pole based on ALEPH Z pole analysis
  ///
  /// @author Peter Richardson
  class AMY_1990_I295160 : public Analysis {
  public:

    /// Constructor
    RIVET_DEFAULT_ANALYSIS_CTOR(AMY_1990_I295160);


    /// @name Analysis methods
    /// @{

    /// Book histograms and initialise projections before the run
    void init() {
      const ChargedFinalState cfs;
      declare(cfs, "CFS");
      unsigned int offset = 0;
      if(isCompatibleWithSqrtS(50.0*GeV)) {
        offset = 1;
      }
      else if(isCompatibleWithSqrtS(52.0*GeV)) {
        offset = 2;
      }
      else if(isCompatibleWithSqrtS(55.0*GeV)) {
        offset = 3;
      }
      else if(isCompatibleWithSqrtS(56.0*GeV)) {
        offset = 4;
      }
      else if(isCompatibleWithSqrtS(57.0*GeV)) {
        offset = 5;
      }
      else if(isCompatibleWithSqrtS(60.0*GeV)) {
        offset = 6;
      }
      else if(isCompatibleWithSqrtS(60.8*GeV)) {
        offset = 7;
      }
      else if(isCompatibleWithSqrtS(61.4*GeV)) {
        offset = 8;
      }
      else {
        MSG_WARNING("CoM energy of events sqrt(s) = " << sqrtS()/GeV
                    << " doesn't match any available analysis energy .");
      }
      book(_histChTotal, 1, 1, offset);
      book(_histTotal, 2, 1, 1);
      if (offset==5) {
        book(_histChAver, 1, 1, 9);
      }
    }

    /// Perform the per-event analysis
    void analyze(const Event& event) {
      const string ECMS = _histTotal->bin(ecmsAxis.index(sqrtS()/GeV)).xEdge();
      const FinalState& cfs = apply<FinalState>(event, "CFS");
      MSG_DEBUG("Total charged multiplicity = " << cfs.size());
      _histChTotal->fill(cfs.size());
      _histTotal->fill(ECMS, cfs.size());
      if (_histChAver) {
        _histChAver->fill(cfs.size());
        _histTotal->fill(string("50.0 - 61.4"), cfs.size());
      }
    }

    /// Normalise histograms etc., after the run
    void finalize() {
      scale(_histChTotal, 100.0/sumOfWeights()); // %age (100)
      if (_histChAver) scale(_histChAver, 100.0/sumOfWeights());
    }

    /// @}


  private:

    /// @name Histograms
    /// @{
    BinnedHistoPtr<int> _histChTotal, _histChAver;
    BinnedProfilePtr<string> _histTotal;
    YODA::Axis<double> ecmsAxis{49., 51., 53., 55.5, 56.5, 58., 60.5, 61., 62.};
    /// @}

  };


  RIVET_DECLARE_PLUGIN(AMY_1990_I295160);

}
