/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.xpack.ml.job.process.autodetect.output;

import java.time.Clock;
import java.time.Duration;
import java.util.Collections;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.logging.log4j.message.Message;
import org.apache.logging.log4j.message.ParameterizedMessage;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.ActionRequest;
import org.elasticsearch.action.ActionType;
import org.elasticsearch.action.DocWriteResponse;
import org.elasticsearch.action.bulk.BulkResponse;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.action.support.WriteRequest;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.xpack.core.ClientHelper;
import org.elasticsearch.xpack.core.ml.MachineLearningField;
import org.elasticsearch.xpack.core.ml.action.PutJobAction;
import org.elasticsearch.xpack.core.ml.action.UpdateJobAction;
import org.elasticsearch.xpack.core.ml.annotations.Annotation;
import org.elasticsearch.xpack.core.ml.job.config.JobUpdate;
import org.elasticsearch.xpack.core.ml.job.messages.Messages;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.output.FlushAcknowledgement;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.CategorizerStats;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSizeStats;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.ModelSnapshot;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.Quantiles;
import org.elasticsearch.xpack.core.ml.job.process.autodetect.state.TimingStats;
import org.elasticsearch.xpack.core.ml.job.results.AnomalyRecord;
import org.elasticsearch.xpack.core.ml.job.results.Bucket;
import org.elasticsearch.xpack.core.ml.job.results.CategoryDefinition;
import org.elasticsearch.xpack.core.ml.job.results.Forecast;
import org.elasticsearch.xpack.core.ml.job.results.ForecastRequestStats;
import org.elasticsearch.xpack.core.ml.job.results.Influencer;
import org.elasticsearch.xpack.core.ml.job.results.ModelPlot;
import org.elasticsearch.xpack.ml.annotations.AnnotationPersister;
import org.elasticsearch.xpack.ml.job.persistence.JobResultsPersister;
import org.elasticsearch.xpack.ml.job.persistence.TimingStatsReporter;
import org.elasticsearch.xpack.ml.job.process.autodetect.AutodetectProcess;
import org.elasticsearch.xpack.ml.job.process.autodetect.output.FlushListener;
import org.elasticsearch.xpack.ml.job.process.normalizer.Renormalizer;
import org.elasticsearch.xpack.ml.job.results.AutodetectResult;
import org.elasticsearch.xpack.ml.notifications.AnomalyDetectionAuditor;

public class AutodetectResultProcessor {
    private static final Logger LOGGER = LogManager.getLogger(AutodetectResultProcessor.class);
    private final Client client;
    private final AnomalyDetectionAuditor auditor;
    private final String jobId;
    private final Renormalizer renormalizer;
    private final JobResultsPersister persister;
    private final AutodetectProcess process;
    private final TimingStatsReporter timingStatsReporter;
    private final Clock clock;
    final CountDownLatch completionLatch = new CountDownLatch(1);
    final Semaphore updateModelSnapshotSemaphore = new Semaphore(1);
    private final FlushListener flushListener;
    private volatile boolean processKilled;
    private volatile boolean failed;
    private final Map<String, ForecastRequestStats> runningForecasts;
    private final long priorRunsBucketCount;
    private long currentRunBucketCount;
    private final JobResultsPersister.Builder bulkResultsPersister;
    private final AnnotationPersister.Builder bulkAnnotationsPersister;
    private boolean deleteInterimRequired;
    private volatile ModelSizeStats latestModelSizeStats;

    public AutodetectResultProcessor(Client client, AnomalyDetectionAuditor auditor, String jobId, Renormalizer renormalizer, JobResultsPersister persister, AnnotationPersister annotationPersister, AutodetectProcess process, ModelSizeStats latestModelSizeStats, TimingStats timingStats) {
        this(client, auditor, jobId, renormalizer, persister, annotationPersister, process, latestModelSizeStats, timingStats, Clock.systemUTC(), new FlushListener());
    }

    AutodetectResultProcessor(Client client, AnomalyDetectionAuditor auditor, String jobId, Renormalizer renormalizer, JobResultsPersister persister, AnnotationPersister annotationPersister, AutodetectProcess autodetectProcess, ModelSizeStats latestModelSizeStats, TimingStats timingStats, Clock clock, FlushListener flushListener) {
        this.client = Objects.requireNonNull(client);
        this.auditor = Objects.requireNonNull(auditor);
        this.jobId = Objects.requireNonNull(jobId);
        this.renormalizer = Objects.requireNonNull(renormalizer);
        this.persister = Objects.requireNonNull(persister);
        this.process = Objects.requireNonNull(autodetectProcess);
        this.flushListener = Objects.requireNonNull(flushListener);
        this.latestModelSizeStats = Objects.requireNonNull(latestModelSizeStats);
        this.bulkResultsPersister = persister.bulkPersisterBuilder(jobId).shouldRetry(this::isAlive);
        this.bulkAnnotationsPersister = annotationPersister.bulkPersisterBuilder(jobId).shouldRetry(this::isAlive);
        this.timingStatsReporter = new TimingStatsReporter(timingStats, this.bulkResultsPersister);
        this.clock = Objects.requireNonNull(clock);
        this.deleteInterimRequired = true;
        this.priorRunsBucketCount = timingStats.getBucketCount();
        this.runningForecasts = new ConcurrentHashMap<String, ForecastRequestStats>();
    }

    public void process() {
        try {
            this.readResults();
            try {
                if (!this.processKilled) {
                    this.timingStatsReporter.finishReporting();
                    this.bulkResultsPersister.executeRequest();
                    this.bulkAnnotationsPersister.executeRequest();
                }
            }
            catch (Exception e) {
                LOGGER.warn((Message)new ParameterizedMessage("[{}] Error persisting autodetect results", (Object)this.jobId), (Throwable)e);
            }
            LOGGER.info("[{}] {} buckets parsed from autodetect output", (Object)this.jobId, (Object)this.currentRunBucketCount);
        }
        catch (Exception e) {
            this.failed = true;
            if (this.processKilled) {
                LOGGER.warn("[{}] some results not processed due to the process being killed", (Object)this.jobId);
            } else if (!this.process.isProcessAliveAfterWaiting()) {
                LOGGER.warn("[{}] some results not processed due to the termination of autodetect", (Object)this.jobId);
            } else {
                LOGGER.error((Message)new ParameterizedMessage("[{}] error parsing autodetect output", (Object)this.jobId), (Throwable)e);
            }
        }
        finally {
            this.flushListener.clear();
            this.handleOpenForecasts();
            this.completionLatch.countDown();
        }
    }

    private void readResults() {
        this.currentRunBucketCount = 0L;
        try {
            Iterator<AutodetectResult> iterator = this.process.readAutodetectResults();
            while (iterator.hasNext()) {
                try {
                    AutodetectResult result = iterator.next();
                    this.processResult(result);
                    if (result.getBucket() == null) continue;
                    LOGGER.trace("[{}] Bucket number {} parsed from output", (Object)this.jobId, (Object)this.currentRunBucketCount);
                }
                catch (Exception e) {
                    if (!this.isAlive()) {
                        throw e;
                    }
                    LOGGER.warn((Message)new ParameterizedMessage("[{}] Error processing autodetect result", (Object)this.jobId), (Throwable)e);
                }
            }
        }
        finally {
            this.process.consumeAndCloseOutputStream();
        }
    }

    public void setProcessKilled() {
        this.processKilled = true;
        try {
            this.renormalizer.shutdown();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new RuntimeException(e);
        }
    }

    void handleOpenForecasts() {
        try {
            if (!this.runningForecasts.isEmpty()) {
                LOGGER.warn("[{}] still had forecasts {} executing. Attempting to set them to failed.", (Object)this.jobId, this.runningForecasts.keySet());
                this.bulkResultsPersister.clearBulkRequest();
                for (ForecastRequestStats forecastRequestStats : this.runningForecasts.values()) {
                    ForecastRequestStats failedStats = new ForecastRequestStats(forecastRequestStats);
                    failedStats.setStatus(ForecastRequestStats.ForecastRequestStatus.FAILED);
                    failedStats.setMessages(Collections.singletonList("forecast unable to complete as native process was killed."));
                    this.bulkResultsPersister.persistForecastRequestStats(failedStats);
                }
                this.bulkResultsPersister.executeRequest();
            }
        }
        catch (Exception ex) {
            LOGGER.warn((Message)new ParameterizedMessage("[{}] failure setting running forecasts to failed.", (Object)this.jobId), (Throwable)ex);
        }
    }

    void processResult(AutodetectResult result) {
        FlushAcknowledgement flushAcknowledgement;
        Quantiles quantiles;
        ModelSnapshot modelSnapshot;
        ModelSizeStats modelSizeStats;
        ForecastRequestStats forecastRequestStats;
        Forecast forecast;
        Annotation annotation;
        ModelPlot modelPlot;
        CategorizerStats categorizerStats;
        CategoryDefinition categoryDefinition;
        List<Influencer> influencers;
        List<AnomalyRecord> records;
        if (this.processKilled) {
            return;
        }
        Bucket bucket = result.getBucket();
        if (bucket != null) {
            if (this.deleteInterimRequired) {
                LOGGER.trace("[{}] Deleting interim results", (Object)this.jobId);
                this.persister.deleteInterimResults(this.jobId);
                this.deleteInterimRequired = false;
            }
            this.timingStatsReporter.reportBucket(bucket);
            this.bulkResultsPersister.persistBucket(bucket).executeRequest();
            this.bulkAnnotationsPersister.executeRequest();
            ++this.currentRunBucketCount;
        }
        if ((records = result.getRecords()) != null && !records.isEmpty()) {
            this.bulkResultsPersister.persistRecords(records);
        }
        if ((influencers = result.getInfluencers()) != null && !influencers.isEmpty()) {
            this.bulkResultsPersister.persistInfluencers(influencers);
        }
        if ((categoryDefinition = result.getCategoryDefinition()) != null) {
            this.persister.persistCategoryDefinition(categoryDefinition, this::isAlive);
        }
        if ((categorizerStats = result.getCategorizerStats()) != null) {
            this.bulkResultsPersister.persistCategorizerStats(categorizerStats);
        }
        if ((modelPlot = result.getModelPlot()) != null) {
            this.bulkResultsPersister.persistModelPlot(modelPlot);
        }
        if ((annotation = result.getAnnotation()) != null) {
            this.bulkAnnotationsPersister.persistAnnotation(annotation);
            this.notifyCategorizationStatusChange(annotation);
        }
        if ((forecast = result.getForecast()) != null) {
            this.bulkResultsPersister.persistForecast(forecast);
        }
        if ((forecastRequestStats = result.getForecastRequestStats()) != null) {
            LOGGER.trace("Received Forecast Stats [{}]", (Object)forecastRequestStats.getId());
            this.bulkResultsPersister.persistForecastRequestStats(forecastRequestStats);
            if (forecastRequestStats.getStatus().isAnyOf(new ForecastRequestStats.ForecastRequestStatus[]{ForecastRequestStats.ForecastRequestStatus.FAILED, ForecastRequestStats.ForecastRequestStatus.FINISHED})) {
                this.runningForecasts.remove(forecastRequestStats.getForecastId());
            } else {
                this.runningForecasts.put(forecastRequestStats.getForecastId(), forecastRequestStats);
            }
            switch (forecastRequestStats.getStatus()) {
                case OK: 
                case STARTED: {
                    break;
                }
                default: {
                    this.bulkResultsPersister.executeRequest();
                }
            }
        }
        if ((modelSizeStats = result.getModelSizeStats()) != null) {
            this.processModelSizeStats(modelSizeStats);
        }
        if ((modelSnapshot = result.getModelSnapshot()) != null) {
            BulkResponse bulkResponse = this.persister.persistModelSnapshot(modelSnapshot, WriteRequest.RefreshPolicy.IMMEDIATE, this::isAlive);
            assert (bulkResponse.getItems().length == 1);
            IndexResponse indexResponse = (IndexResponse)bulkResponse.getItems()[0].getResponse();
            if (indexResponse.getResult() == DocWriteResponse.Result.CREATED) {
                this.updateModelSnapshotOnJob(modelSnapshot);
            }
            this.bulkAnnotationsPersister.persistAnnotation(ModelSnapshot.annotationDocumentId((ModelSnapshot)modelSnapshot), this.createModelSnapshotAnnotation(modelSnapshot));
        }
        if ((quantiles = result.getQuantiles()) != null) {
            LOGGER.debug("[{}] Parsed Quantiles with timestamp {}", (Object)this.jobId, (Object)quantiles.getTimestamp());
            this.persister.persistQuantiles(quantiles, this::isAlive);
            this.bulkResultsPersister.executeRequest();
            if (!this.processKilled && this.renormalizer.isEnabled()) {
                this.persister.commitResultWrites(this.jobId);
                LOGGER.debug("[{}] Quantiles queued for renormalization", (Object)this.jobId);
                this.renormalizer.renormalize(quantiles);
            }
        }
        if ((flushAcknowledgement = result.getFlushAcknowledgement()) != null) {
            LOGGER.debug("[{}] Flush acknowledgement parsed from output for ID {}", (Object)this.jobId, (Object)flushAcknowledgement.getId());
            Exception exception = null;
            try {
                this.bulkResultsPersister.executeRequest();
                this.bulkAnnotationsPersister.executeRequest();
                this.persister.commitResultWrites(this.jobId);
                this.persister.commitAnnotationWrites();
                LOGGER.debug("[{}] Flush acknowledgement sent to listener for ID {}", (Object)this.jobId, (Object)flushAcknowledgement.getId());
                this.flushListener.acknowledgeFlush(flushAcknowledgement, exception);
            }
            catch (Exception e) {
                try {
                    LOGGER.error("[" + this.jobId + "] failed to bulk persist results and commit writes during flush acknowledgement for ID " + flushAcknowledgement.getId(), (Throwable)e);
                    exception = e;
                    throw e;
                }
                catch (Throwable throwable) {
                    this.flushListener.acknowledgeFlush(flushAcknowledgement, exception);
                    throw throwable;
                }
            }
            this.deleteInterimRequired = true;
        }
    }

    private Annotation createModelSnapshotAnnotation(ModelSnapshot modelSnapshot) {
        assert (modelSnapshot != null);
        Date currentTime = new Date(this.clock.millis());
        return new Annotation.Builder().setAnnotation(Messages.getMessage((String)"Job model snapshot with id [{0}] stored", (Object[])new Object[]{modelSnapshot.getSnapshotId()})).setCreateTime(currentTime).setCreateUsername("_xpack").setTimestamp(modelSnapshot.getLatestResultTimeStamp()).setEndTimestamp(modelSnapshot.getLatestResultTimeStamp()).setJobId(this.jobId).setModifiedTime(currentTime).setModifiedUsername("_xpack").setType(Annotation.Type.ANNOTATION).setEvent(Annotation.Event.MODEL_SNAPSHOT_STORED).build();
    }

    private void processModelSizeStats(ModelSizeStats modelSizeStats) {
        LOGGER.trace("[{}] Parsed ModelSizeStats: {} / {} / {} / {} / {} / {}", (Object)this.jobId, (Object)modelSizeStats.getModelBytes(), (Object)modelSizeStats.getTotalByFieldCount(), (Object)modelSizeStats.getTotalOverFieldCount(), (Object)modelSizeStats.getTotalPartitionFieldCount(), (Object)modelSizeStats.getBucketAllocationFailuresCount(), (Object)modelSizeStats.getMemoryStatus());
        this.persister.persistModelSizeStats(modelSizeStats, this::isAlive);
        this.notifyModelMemoryStatusChange(modelSizeStats);
        this.latestModelSizeStats = modelSizeStats;
    }

    private void notifyModelMemoryStatusChange(ModelSizeStats modelSizeStats) {
        ModelSizeStats.MemoryStatus memoryStatus = modelSizeStats.getMemoryStatus();
        if (memoryStatus != this.latestModelSizeStats.getMemoryStatus()) {
            if (memoryStatus == ModelSizeStats.MemoryStatus.SOFT_LIMIT) {
                this.auditor.warning(this.jobId, Messages.getMessage((String)"Job memory status changed to soft_limit; memory pruning will now be more aggressive"));
            } else if (memoryStatus == ModelSizeStats.MemoryStatus.HARD_LIMIT) {
                if (modelSizeStats.getModelBytesMemoryLimit() == null || modelSizeStats.getModelBytesExceeded() == null) {
                    this.auditor.error(this.jobId, Messages.getMessage((String)"Job memory status changed to hard_limit at {0}; adjust the analysis_limits.model_memory_limit setting to ensure all data is analyzed", (Object[])new Object[]{ByteSizeValue.ofBytes((long)modelSizeStats.getModelBytes()).toString()}));
                } else {
                    this.auditor.error(this.jobId, Messages.getMessage((String)"Job memory status changed to hard_limit; job exceeded model memory limit {0} by {1}. Adjust the analysis_limits.model_memory_limit setting to ensure all data is analyzed", (Object[])new Object[]{ByteSizeValue.ofBytes((long)modelSizeStats.getModelBytesMemoryLimit()).toString(), ByteSizeValue.ofBytes((long)modelSizeStats.getModelBytesExceeded()).toString()}));
                }
            }
        }
    }

    private void notifyCategorizationStatusChange(Annotation annotation) {
        if (annotation.getEvent() == Annotation.Event.CATEGORIZATION_STATUS_CHANGE) {
            long bucketCount = this.priorRunsBucketCount + this.currentRunBucketCount;
            this.auditor.warning(this.jobId, annotation.getAnnotation() + " after " + bucketCount + (bucketCount == 1L ? " bucket" : " buckets"));
        }
    }

    protected void updateModelSnapshotOnJob(final ModelSnapshot modelSnapshot) {
        JobUpdate update = new JobUpdate.Builder(this.jobId).setModelSnapshotId(modelSnapshot.getSnapshotId()).build();
        UpdateJobAction.Request updateRequest = UpdateJobAction.Request.internal((String)this.jobId, (JobUpdate)update);
        try {
            this.updateModelSnapshotSemaphore.acquire();
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.info("[{}] Interrupted acquiring update model snapshot semaphore", (Object)this.jobId);
            return;
        }
        ClientHelper.executeAsyncWithOrigin((Client)this.client, (String)"ml", (ActionType)UpdateJobAction.INSTANCE, (ActionRequest)updateRequest, (ActionListener)new ActionListener<PutJobAction.Response>(){

            public void onResponse(PutJobAction.Response response) {
                AutodetectResultProcessor.this.updateModelSnapshotSemaphore.release();
                LOGGER.debug("[{}] Updated job with model snapshot id [{}]", (Object)AutodetectResultProcessor.this.jobId, (Object)modelSnapshot.getSnapshotId());
            }

            public void onFailure(Exception e) {
                AutodetectResultProcessor.this.updateModelSnapshotSemaphore.release();
                LOGGER.error("[" + AutodetectResultProcessor.this.jobId + "] Failed to update job with new model snapshot id [" + modelSnapshot.getSnapshotId() + "]", (Throwable)e);
            }
        });
    }

    public void awaitCompletion() throws TimeoutException {
        try {
            if (!this.completionLatch.await(MachineLearningField.STATE_PERSIST_RESTORE_TIMEOUT.getMinutes(), TimeUnit.MINUTES)) {
                throw new TimeoutException("Timed out waiting for results processor to complete for job " + this.jobId);
            }
            this.updateModelSnapshotSemaphore.acquire();
            this.updateModelSnapshotSemaphore.release();
            this.waitUntilRenormalizerIsIdle();
            this.persister.commitResultWrites(this.jobId);
            this.persister.commitAnnotationWrites();
            this.persister.commitStateWrites(this.jobId);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            LOGGER.info("[{}] Interrupted waiting for results processor to complete", (Object)this.jobId);
        }
    }

    @Nullable
    public FlushAcknowledgement waitForFlushAcknowledgement(String flushId, Duration timeout) throws Exception {
        return this.failed ? null : this.flushListener.waitForFlush(flushId, timeout);
    }

    public void clearAwaitingFlush(String flushId) {
        this.flushListener.clear(flushId);
    }

    public void waitUntilRenormalizerIsIdle() throws InterruptedException {
        this.renormalizer.waitUntilIdle();
    }

    public boolean isFailed() {
        return this.failed;
    }

    public ModelSizeStats modelSizeStats() {
        return this.latestModelSizeStats;
    }

    public TimingStats timingStats() {
        return this.timingStatsReporter.getCurrentTimingStats();
    }

    boolean isDeleteInterimRequired() {
        return this.deleteInterimRequired;
    }

    private boolean isAlive() {
        if (this.processKilled) {
            return false;
        }
        return this.process.isProcessAliveAfterWaiting();
    }

    void setDeleteInterimRequired(boolean deleteInterimRequired) {
        this.deleteInterimRequired = deleteInterimRequired;
    }
}

