/******************************************************************************
 *{@C
 *      Copyright:      2005-2022 Paul Obermeier (obermeier@tcl3d.org)
 *
 *                      See the file "Tcl3D_License.txt" for information on
 *                      usage and redistribution of this file, and for a
 *                      DISCLAIMER OF ALL WARRANTIES.
 *
 *      Module:         Tcl3D -> tcl3dOgl
 *      Filename:       glu.i
 *
 *      Author:         Paul Obermeier
 *
 *      Description:    SWIG file for wrapping the OpenGL utility library GLU
 *                      contained in file glu.h.
 *
 *****************************************************************************/

%{
#if defined (WIN32)
#include <windows.h>
#include <wingdi.h>
#endif
#include "GL/glu.h"
%}

%typemap(in) double * (double *tmp) {
    if ((tmp = GetDoubleList (interp, $input)) == NULL) {
        return TCL_ERROR;
    } else {
        $1 = tmp;
    }
}

%typemap(in) float * (float *tmp) {
    if ((tmp = GetFloatList (interp, $input)) == NULL) {
        return TCL_ERROR;
    } else {
        $1 = tmp;
    }
}

// Do not generate wrappper code for the GLU callback functions.
// This is work to be done.

%ignore gluQuadricCallback;
%ignore gluTessCallback;
%ignore gluNurbsCallback;

%include tcl3dDefines.i

#define WINGDIAPI
#define APIENTRY


// Redefine the following functions with a return value, so that the SWIG
// generated wrapper function returns a Tcl string.
// Maybe this could be made more elegant with typemaps ??

const char *gluGetString (GLenum name);
const char *gluErrorString (GLenum errorCode);

#if defined (WIN32)

int gluProject (
    GLdouble        objx, 
    GLdouble        objy, 
    GLdouble        objz,  
    const GLdouble  modelMatrix[16], 
    const GLdouble  projMatrix[16], 
    const GLint     viewport[4], 
    GLdouble        *OUTPUT, 
    GLdouble        *OUTPUT, 
    GLdouble        *OUTPUT);

int gluUnProject (
    GLdouble       winx, 
    GLdouble       winy, 
    GLdouble       winz, 
    const GLdouble modelMatrix[16], 
    const GLdouble projMatrix[16], 
    const GLint    viewport[4], 
    GLdouble       *OUTPUT, 
    GLdouble       *OUTPUT, 
    GLdouble       *OUTPUT);
#else

GLint gluProject (
    GLdouble objX, GLdouble objY, GLdouble objZ,
    const GLdouble *model, const GLdouble *proj, const GLint *view,
    GLdouble* OUTPUT, GLdouble* OUTPUT, GLdouble* OUTPUT);

GLint gluUnProject (
    GLdouble winX, GLdouble winY, GLdouble winZ,
    const GLdouble *model, const GLdouble *proj, const GLint *view,
    GLdouble* OUTPUT, GLdouble * OUTPUT, GLdouble* OUTPUT);

#endif

%{
static GLint tcl3dOglProject (
    GLdouble objX, GLdouble objY, GLdouble objZ,
    const GLvoid *model, const GLvoid *proj, const GLvoid *view,
    GLdouble *winX, GLdouble *winY, GLdouble *winZ)
{
    return gluProject (
    objX, objY, objZ,
    (const GLdouble *)model, (const GLdouble *)proj, (const GLint *)view,
    winX, winY, winZ);
}

static GLint tcl3dOglUnProject (
    GLdouble winX, GLdouble winY, GLdouble winZ,
    const GLvoid *model, const GLvoid *proj, const GLvoid *view,
    GLdouble *objX, GLdouble *objY, GLdouble *objZ)
{
    return gluUnProject (
    winX, winY, winZ,
    (const GLdouble *)model, (const GLdouble *)proj, (const GLint *)view,
    objX, objY, objZ);
}
%}

/***************************************************************************
 *[@e
 *      Name:           tcl3dOglProject - Utility function for gluProject.
 *
 *      Synopsis:       GLint tcl3dOglProject {
 *                              objX objY objZ model proj view }
 *
 *      Description:    objX    : double
 *                      objY    : double
 *                      objZ    : double
 *                      model   : tcl3dVector of type GLdouble (size 16)
 *                      proj    : tcl3dVector of type GLdouble (size 16)
 *                      view    : tcl3dVector of type GLint    (size 4)
 *
 *                      Utility function for using the OpenGL GLU function
 *                      gluProject. The signature of the gluProject function
 *                      expects the ModelView and Projection matrix as well as
 *                      the viewport data to be given as Tcl lists.
 *                      This utility function allows these 3 parameters to be
 *                      specified as tcl3dVectors as returned by calls to
 *                      glGetGoublev and glGetIntv.
 *
 *                      A 3-element Tcl list with integer values corresponding
 *                      to the winX, winY and winZ parameters of the OpenGL
 *                      gluProject function is returned.
 *
 *      See also:       tcl3dOglUnProject
 *
 ***************************************************************************/

GLint tcl3dOglProject (
    GLdouble objX, GLdouble objY, GLdouble objZ,
    const GLvoid *model, const GLvoid *proj, const GLvoid *view,
    GLdouble* OUTPUT, GLdouble * OUTPUT, GLdouble* OUTPUT);

/***************************************************************************
 *[@e
 *      Name:           tcl3dOglUnProject - Utility function for gluUnProject.
 *
 *      Synopsis:       GLint tcl3dOglUnProject {
 *                              winX winY winZ model proj view }
 *
 *      Description:    winX    : double
 *                      winY    : double
 *                      winZ    : double
 *                      model   : tcl3dVector of type GLdouble (size 16)
 *                      proj    : tcl3dVector of type GLdouble (size 16)
 *                      view    : tcl3dVector of type GLint    (size 4)
 *
 *                      Utility function for using the OpenGL GLU function
 *                      gluUnProject. The signature of the gluUnProject function
 *                      expects the ModelView and Projection matrix as well as
 *                      the viewport data to be given as Tcl lists.
 *                      This utility function allows these 3 parameters to be
 *                      specified as tcl3dVectors as returned by calls to
 *                      glGetGoublev and glGetIntv.
 *
 *                      A 3-element Tcl list with integer values corresponding
 *                      to the objX, objY and objZ parameters of the OpenGL
 *                      gluUnProject function is returned.
 *
 *      See also:       tcl3dOglProject
 *
 ***************************************************************************/

GLint tcl3dOglUnProject (
    GLdouble winX, GLdouble winY, GLdouble winZ,
    const GLvoid *model, const GLvoid *proj, const GLvoid *view,
    GLdouble* OUTPUT, GLdouble * OUTPUT, GLdouble* OUTPUT);

%include "GL/glu.h"
