/*  $Id: CosmoSmashEngine.h,v 1.50 2024/04/27 05:24:34 sarrazip Exp $

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2000-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#ifndef _H_CosmoSmashEngine
#define _H_CosmoSmashEngine

#include "Command.h"

class Controller;

#include <flatzebra/GameEngine.h>
#include <flatzebra/Sprite.h>
#include <flatzebra/KeyState.h>

// Sound engine:
#include <flatzebra/roundbeetle/SoundEngine.h>
#include <flatzebra/roundbeetle/WaveFileBuffer.h>

#include <string>
#include <vector>
#include <iostream>

enum
{
    WINDOW_WIDTH = 675,
    WINDOW_HEIGHT = 520,
    NUM_LEVELS = 6
};


class CosmoSmashEngine : public flatzebra::GameEngine
/*  A space rock shooting video game engine.
*/
{
public:
    CosmoSmashEngine(long initialScore,
                        bool mirrorHyperspace, bool useSound,
                        bool fullScreen, Controller *controller,
                        std::ostream *gameScriptStream,
                        bool processActiveEvent,
                        bool useAcceleratedRendering);
    /*  See base class.

        Throws a string or an integer code if an error occurs.

        If 'mirrorHyperspace' is true, then hyperspace pivots around
        the center of the ground.  If 'mirrorHyperspace' is false,
        then hyperspace is random.

        'controller' must be a non-null pointer to an object constructed
        with operator new.
        The destructor of this class will call delete on 'controller'.

        'processActiveEvent' determines if the game automatically pauses
        when the game window loses focus.
    */

    virtual ~CosmoSmashEngine();
    /*  Destroys the Controller object given to the constructor,
        by calling operator delete.
        That object must have been constructed with operator new.
    */

    virtual void processKey(SDL_Keycode keysym, bool pressed) override;

    virtual void processActivation(bool appActive) override;

    virtual bool tick() override;

protected:

    // Inherited:
    bool animatePlayer();
    void animateAutomaticCharacters();
    void detectCollisions();
    void restoreBackground();
    void drawSprites();

private:

    typedef std::vector<flatzebra::Couple> CoupleList;

    enum PlayerBoostType
    {
        NO_BOOST, TRIPLE_BULLETS, CLOSER_BULLETS
    };

    class SpinnerSprite : public flatzebra::Sprite
    {
    public:

        // startFreq, endFreq: range of frequencies in Hz of spinner's
        //                     square wave sound.
        // durationInSeconds: life of spinner, assuming it reaches ground.
        //
        SpinnerSprite(const flatzebra::PixmapArray &pixmapArray,
                      const flatzebra::Couple &pos,
                      const flatzebra::Couple &speed,
                      const flatzebra::Couple &collBoxPos,
                      const flatzebra::Couple &collBoxSize,
                      roundbeetle::SoundEngine *mixer,
                      float startFreq,
                      float endFreq,
                      float durationInSeconds)
        :   flatzebra::Sprite(pixmapArray, pos, speed, flatzebra::Couple(0, 0),
                              collBoxPos, collBoxSize),
            soundEngine(mixer),
            soundReq(-1),  // -1 means invalid
            linMovFreq(startFreq, endFreq, durationInSeconds)
        {
        }

        virtual ~SpinnerSprite()
        {
            if (soundEngine != NULL && soundReq != -1)
                soundEngine->stopRequest(soundReq);
        }

        void setSoundReq(int _soundReq) { soundReq = _soundReq; }

        int getSoundReq() const { return soundReq; }

        roundbeetle::LinearMovingFreq &getLinearMovingFreq() { return linMovFreq; }

    private:

        SpinnerSprite(const SpinnerSprite &);
        SpinnerSprite &operator = (const SpinnerSprite &);

    private:

        roundbeetle::SoundEngine *soundEngine;  // may be null
        int soundReq;  // handle of the RoundBeetle sound request
        roundbeetle::LinearMovingFreq linMovFreq;  // roundbeetle thread refers to this

    };



    enum PauseState { PLAYING, LOST_FOCUS, AWAITING_RESUME } pauseState;
    unsigned long tickCount;
    bool useSound;

    // SDL color values (see SDL_MapRGB()):
    SDL_Color levelColors[NUM_LEVELS + 1];  // valid indexes: 1..NUM_LEVELS
    SDL_Color whiteColor;
    SDL_Color greenColor;
    SDL_Color explosionBGColor;

    bool mirrorHyperspace;
    bool useGameExtensions;
    unsigned long playerBoostTicks;
    PlayerBoostType playerBoostType;

    flatzebra::PixmapArray playerPA;
    flatzebra::Sprite *playerSprite;

    flatzebra::PixmapArray baseBulletPA;
    flatzebra::SpriteList  baseBulletSprites;

    flatzebra::PixmapArray bigRockPA;
    flatzebra::PixmapArray smallRockPA;
    flatzebra::SpriteList rockSprites;  // contains big and small rock sprites

    flatzebra::PixmapArray bigSpinnerPA;
    flatzebra::PixmapArray smallSpinnerPA;
    flatzebra::SpriteList spinnerSprites;

    flatzebra::PixmapArray pulsarPA;
    flatzebra::SpriteList pulsarSprites;

    flatzebra::PixmapArray saucerPA;
    flatzebra::SpriteList saucerSprites;

    flatzebra::PixmapArray saucerBulletPA;
    flatzebra::SpriteList saucerBulletSprites;

    flatzebra::PixmapArray explosionPA;
    flatzebra::SpriteList  explosionSprites;

    flatzebra::PixmapArray questionPA;
    flatzebra::SpriteList questionSprites;

    int groundPos;     // y position of highest pixel row of ground
    int groundHeight;  // height of ground in pixels
    int mountainTopPos;  // y position of highest possible pixel for mountains

    Controller *controller;
    flatzebra::KeyState quitKS;

    long   initScore;     // score at which a game starts (determines level)

    long   theScore;      // player's score in points
    long   thePeakScore;  // not displayed
    int    theLevel;      // level number (1--6) (depends on theScore)
    flatzebra::Couple scoreAreaPos;
    flatzebra::Couple peakScoreAreaPos;
    flatzebra::Couple levelNumAreaPos;

    int    numLives;  // number of player lives left
    bool   updateNumLives;
    flatzebra::Couple numLivesAreaPos;

    unsigned long timeBeforeNextPulsar;  // in ticks
    unsigned long timeBeforeNextSaucer;
    unsigned long timeBeforeNextQuestion;

    CoupleList starPositions;
    CoupleList mountainRangePositions;


    /*  SOUND EFFECTS:
    */
    enum { NUM_ROCK_HIT_SOUNDS = 2 };

    roundbeetle::SoundEngine *soundEngine;  // may be null
    roundbeetle::WaveFileBuffer rockHitSounds[NUM_ROCK_HIT_SOUNDS];
    roundbeetle::WaveFileBuffer playerHitSound;
    roundbeetle::WaveFileBuffer pulsarBeepSound;
    roundbeetle::WaveFileBuffer saucerShootingSound;
    roundbeetle::WaveFileBuffer hyperspaceSound;

    unsigned long timeBeforeNextCadenceSound;


    // Game script writing:
    std::ostream *gameScriptStream;  // null means not saving


    /*  Implementation functions:
    */
    void fireBullet(flatzebra::Couple pos);
    void loadPixmap(const char *filePath,
                    flatzebra::PixmapArray &pa,
                    size_t index);
    void initializeSprites();
    void initializeMisc(bool useSound);
    void setTimeBeforeNextPulsar();
    void setTimeBeforeNextSaucer();
    void setTimeBeforeNextQuestion();
    int getSpeedUp() const;
    void readKeyboardCommands();
    flatzebra::Sprite *createExplosionSprite(flatzebra::Couple posOfCenterOfExplosion,
                                    int timeToLive);
    std::pair<flatzebra::Sprite *, flatzebra::Sprite *> splitBigRock(const flatzebra::Sprite &bigRock) const;
    void makeSpriteExplode(const flatzebra::Sprite *target,
                            const flatzebra::Sprite *bullet,
                            long points);
    void addToScore(long n);
    void addToNumLives(int n);
    void killSpritesInList(flatzebra::SpriteList &sl);
    void centerPlayerSprite();
    void repositionPlayerSprite(flatzebra::Couple newpos);
    void killAllAutomaticCharacters();
    void killPlayerBase();
    bool playerIsActive() const;
    std::pair<bool, size_t> animateFallingObjects(flatzebra::SpriteList &sprites);
    void animateSaucers();
    void pause();
    void resume();
    void displayPauseMessage();
    void displayStartMessage(bool display);
    void displayMessage(int row, const char *msg);
    bool detectCollisionsWithBullet(const flatzebra::Sprite &bullet,
                                    flatzebra::SpriteList &slist,
                                    int scoreOnCollision);
    bool detectCollisionsWithExplosions();
    void detectCollisionsBetweenFallingObjects(flatzebra::SpriteList &slist);
    void boostPlayer();
    flatzebra::Couple accelerate(flatzebra::Couple speed) const;
    void playShortExplosionSound();
    void playSoundEffect(const roundbeetle::WaveFileBuffer &wfb);

    void registerCommand(Command command);
    void registerTickEnd();
    void playCadenceSound();
    void startSpinnerSound(SpinnerSprite &s);
    void moveStars();
    bool isBaseExploding() const;


    /*        Forbidden operations:
    */
    CosmoSmashEngine(const CosmoSmashEngine &);
    CosmoSmashEngine &operator = (const CosmoSmashEngine &);
};


#endif  /* _H_CosmoSmashEngine */
