#!/usr/bin/env bash
# =============================================================================
# Title:                    install-gnupg-deps.sh
# Description:              POSIX shell script to build and install
#                           GnuPG dependencies from source
# Author:                   Peter J. Mello, Hooty McOwlface
# Version:                  2.0.0
# Usage:                    [sudo] bash install-gnupg-deps.sh
# Notes:                    Developed and tested on Kubuntu 19.04, intended for
#                           use on Ubuntu and its derivatives > version 18.04
# Bash Version:             5.0.3
# SPDX-License-Identifier:  Apache-2.0
# =============================================================================
# Copyright 2019 Peter J. Mello
# Copyright 2024 FreePG Project
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
# http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
# =============================================================================

# Basic shell built-ins to promote safe operation
set -eu -o pipefail
shopt -qs failglob

# Export environment variable to keep all build processes using bash shell
typeset -x CONFIG_SHELL=$(command -v bash)

# Stash component versions and script shortlink in variables for easy updating
GPGRT_VER=1.56
LIBGCRYPT_VER=1.11.0
KSBA_VER=1.6.7
LIBASSUAN_VER=3.0.2
NTBTLS_VER=0.3.2
NPTH_VER=1.7

# If configure.ac wants more recent minor versions, let's use them
for lib in GPGRT LIBGCRYPT LIBASSUAN KSBA NTBTLS NPTH; do
  j=$(awk -F= /^NEED_${lib}_VERSION=/'{print $2}' < configure.ac)
  eval l=\$${lib}_VER
  j1=${j%%.*}; rest=${j#*.}; j2=${rest%%.*}; rest=${rest#*.}; j3=${rest%%.*}
  k1=${l%%.*}; rest=${l#*.}; k2=${rest%%.*}; rest=${rest#*.}; k3=${rest%%.*}
  [[ $j1 -lt $k1 ]] || [[ $j2 -lt $k2 ]] || [[ $j3 -lt $k3 ]] || eval ${lib}_VER=$j
done

PINENTRY_VER=1.3.1
GPGME_VER=1.23.2
GPA_VER=0.10.0
SIGNING_KEYS=(249B39D24F25E3B6 04376F3EE0856959 2071B08A33BD3F06 8A861B1C7EFD60D9 04376F3EE0856959 2071B08A33BD3F06 8A861B1C7EFD60D9 528897B826403ADA A8DC7067E25FBABB E98E9B2D19C6C8BD)

# Location of build directory - we use this because upstream already gitignores it
BUILD=.deps

# Ensure all necessary dependencies are present on the system
echo "Updating package lists and installing any missing dependencies..."
apt update
apt install -y bzip2 make gettext texinfo gnutls-bin libgnutls28-dev build-essential \
libbz2-dev zlib1g-dev libncurses5-dev libsqlite3-dev libldap2-dev libsecret-1-dev \
wget tar dh-autoreconf libgtk-3-dev imagemagick fig2dev git findutils gnupg

# Determine GnuPG version from the latest git tag
GNUPG_VER=$(git describe || true)
if [[ $GNUPG_VER ]]; then
  GNUPG_VER=${GNUPG_VER#*-}
  GNUPG_VER=${GNUPG_VER%%-*}
else
  GNUPG_VER=unknown
fi

# Create gnupg working dir
mkdir -p /root/.gnupg || echo "Unable to create directory in /root"
chmod og= /root/.gnupg

# Create folder for build files
echo "Creating working directory at '$BUILD' for build processes..."
mkdir -p $BUILD || echo "Unable to create build directory"
cd $BUILD || exit 1

# Retrieve GnuPG release signing keys from Ubuntu keyserver
echo "Retrieving GnuPG release signing keys from Ubuntu keyserver..."
rm /var/tmp/gpg-dist-keyring || true
gpg --no-default-keyring --keyring /var/tmp/gpg-dist-keyring --keyserver keyserver.ubuntu.com --recv-keys ${SIGNING_KEYS[@]}

#Build and install libgpg-error
echo "Building and installing libgpg-error..."
wget -c https://gnupg.org/ftp/gcrypt/libgpg-error/libgpg-error-${GPGRT_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/libgpg-error/libgpg-error-${GPGRT_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring libgpg-error-${GPGRT_VER}.tar.bz2{.sig,}
tar xjof libgpg-error-${GPGRT_VER}.tar.bz2
cd libgpg-error-${GPGRT_VER} || exit 1
autoreconf -fi
./configure
make -s
make check
make install
cd ..

# Build and install libgcrypt
echo "Building and installing libgcrypt..."
wget -c https://gnupg.org/ftp/gcrypt/libgcrypt/libgcrypt-${LIBGCRYPT_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/libgcrypt/libgcrypt-${LIBGCRYPT_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring libgcrypt-${LIBGCRYPT_VER}.tar.bz2{.sig,}
tar xjof libgcrypt-${LIBGCRYPT_VER}.tar.bz2
cd libgcrypt-${LIBGCRYPT_VER} || exit 1
autoreconf -fi
./configure --enable-m-guard --with-capabilities
make -s
make check
make install
cd ..

# Build and install libksba
echo "Building and installing libksba..."
wget -c https://gnupg.org/ftp/gcrypt/libksba/libksba-${KSBA_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/libksba/libksba-${KSBA_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring libksba-${KSBA_VER}.tar.bz2{.sig,}
tar xjof libksba-${KSBA_VER}.tar.bz2
cd libksba-${KSBA_VER} || exit 1
autoreconf -fi
./configure
make -s
make check
make install
cd ..

# Build and install libassuan
echo "Building and installing libassuan..."
wget -c https://gnupg.org/ftp/gcrypt/libassuan/libassuan-${LIBASSUAN_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/libassuan/libassuan-${LIBASSUAN_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring libassuan-${LIBASSUAN_VER}.tar.bz2{.sig,}
tar xjof libassuan-${LIBASSUAN_VER}.tar.bz2
cd libassuan-${LIBASSUAN_VER} || exit 1
autoreconf -fi
./configure
make -s
make check
make install
cd ..

# Build and install ntbTLS
echo "Building and installing ntbtls..."
wget -c https://gnupg.org/ftp/gcrypt/ntbtls/ntbtls-${NTBTLS_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/ntbtls/ntbtls-${NTBTLS_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring ntbtls-${NTBTLS_VER}.tar.bz2{.sig,}
tar xjof ntbtls-${NTBTLS_VER}.tar.bz2
cd ntbtls-${NTBTLS_VER} || exit 1
autoreconf -fi
./configure
make -s
make install
cd ..

# Build and install nPth
echo "Building and installing npth..."
wget -c https://gnupg.org/ftp/gcrypt/npth/npth-${NPTH_VER}.tar.bz2
wget -c https://gnupg.org/ftp/gcrypt/npth/npth-${NPTH_VER}.tar.bz2.sig
gpgv --keyring /var/tmp/gpg-dist-keyring npth-${NPTH_VER}.tar.bz2{.sig,}
tar xjof npth-${NPTH_VER}.tar.bz2
cd npth-${NPTH_VER} || exit 1
autoreconf -fi
./configure
make -s
make check
make install
cd ..

# The below are not build dependencies of GPG, but runtime

## Build and install Pinentry
#echo "Building and installing pinentry..."
#wget -c https://gnupg.org/ftp/gcrypt/pinentry/pinentry-${PINENTRY_VER}.tar.bz2
#wget -c https://gnupg.org/ftp/gcrypt/pinentry/pinentry-${PINENTRY_VER}.tar.bz2.sig
#gpgv --keyring /var/tmp/gpg-dist-keyring pinentry-${PINENTRY_VER}.tar.bz2{.sig,}
#tar xjof pinentry-${PINENTRY_VER}.tar.bz2
#cd pinentry-${PINENTRY_VER} || exit 1
#autoreconf -fi 2>/dev/null
#./configure --enable-libsecret --enable-pinentry-tty --enable-pinentry-qt --enable-pinentry-gnome3
#make -s
#make install
#cd ..
#
## Build and install GPGME
#echo "Building and installing gpgme..."
#wget -c https://gnupg.org/ftp/gcrypt/gpgme/gpgme-${GPGME_VER}.tar.bz2
#wget -c https://gnupg.org/ftp/gcrypt/gpgme/gpgme-${GPGME_VER}.tar.bz2.sig
#gpgv --keyring /var/tmp/gpg-dist-keyring gpgme-${GPGME_VER}.tar.bz2{.sig,}
#tar xjof gpgme-${GPGME_VER}.tar.bz2
#cd gpgme-${GPGME_VER} || exit 1
#autoreconf -fi
#./configure
#make -s
#make check
#make install
#cd ..
#
## Build and install GPA
#echo "Building and installing gpa..."
#wget -c https://gnupg.org/ftp/gcrypt/gpa/gpa-${GPA_VER}.tar.bz2
#wget -c https://gnupg.org/ftp/gcrypt/gpa/gpa-${GPA_VER}.tar.bz2.sig
#gpgv --keyring /var/tmp/gpg-dist-keyring gpa-${GPA_VER}.tar.bz2{.sig,}
#tar xjof gpa-${GPA_VER}.tar.bz2
#cd gpa-${GPA_VER} || exit 1
#autoreconf -fi
#./configure
#make -s
#make install
#cd ..

# Call library linker to scan library directories so newly installed files are available
echo "Calling ldconfig to scan library install locations & update system shared library linker."
ldconfig

# Inform user of the success of all processes
echo "Successfully built and installed GnuPG version ${GNUPG_VER} dependencies to /usr/local"
echo "Now you can run make-gnupg.sh to build GnuPG itself"
exit 0
