/*
 *   drv3780.h -- common data structions for DSP3780 device drivers
 *
 *  Written By: Mike Sullivan IBM Corporation
 *
 *  Copyright (C) 1999 IBM Corporation
 *
 * This program is free software; you can redistribute it and/or modify      
 * it under the terms of the GNU General Public License as published by      
 * the Free Software Foundation; either version 2 of the License, or         
 * (at your option) any later version.                                       
 *                                                                           
 * This program is distributed in the hope that it will be useful,           
 * but WITHOUT ANY WARRANTY; without even the implied warranty of            
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the             
 * GNU General Public License for more details.                              
 *                                                                           
 * NO WARRANTY                                                               
 * THE PROGRAM IS PROVIDED ON AN "AS IS" BASIS, WITHOUT WARRANTIES OR        
 * CONDITIONS OF ANY KIND, EITHER EXPRESS OR IMPLIED INCLUDING, WITHOUT      
 * LIMITATION, ANY WARRANTIES OR CONDITIONS OF TITLE, NON-INFRINGEMENT,      
 * MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. Each Recipient is    
 * solely responsible for determining the appropriateness of using and       
 * distributing the Program and assumes all risks associated with its        
 * exercise of rights under this Agreement, including but not limited to     
 * the risks and costs of program errors, damage to or loss of data,         
 * programs or equipment, and unavailability or interruption of operations.  
 *                                                                           
 * DISCLAIMER OF LIABILITY                                                   
 * NEITHER RECIPIENT NOR ANY CONTRIBUTORS SHALL HAVE ANY LIABILITY FOR ANY   
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL        
 * DAMAGES (INCLUDING WITHOUT LIMITATION LOST PROFITS), HOWEVER CAUSED AND   
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR     
 * TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE    
 * USE OR DISTRIBUTION OF THE PROGRAM OR THE EXERCISE OF ANY RIGHTS GRANTED  
 * HEREUNDER, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGES             
 *                                                                           
 * You should have received a copy of the GNU General Public License         
 * along with this program; if not, write to the Free Software               
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 *                                                                           
 * 
 *  10/23/2000 - Alpha Release 0.1.0
 *            First release to the public
 *
 */
/* define two very handy macros */
#define MKUSHORT(v)   (*(unsigned short *)&v)
#define MKUCHAR(v)    (*(unsigned char *)&v)

/****************************************************************************
 * Overlay Interface ID
 *****************************************************************************/
#define HWIID_DOS_OVERLAY  0x0003


/*****************************************************************************
 * Error codes
 *****************************************************************************/
#define DSP_NOERROR        0x0000
#define DSP_RESET_FAULT    0x0030    /* DSP reset failure */
#define DSP_HALT_FAULT     0x0031    /* DSP halt failure */
#define DSP_WRITE_FAULT    0x0032    /* DSP write failure */
#define DSP_READ_FAULT     0x0033    /* DSP read failure */

/*****************************************************************************
 * Table of offsets for 3780i ISA HBridge registers (base + offset)
 *****************************************************************************/
#define SYS_CTL     0x00          /* System Control register */
#define SYS_CTL2    0x02          /* System Control reg 2 (HBridge CNTL) */
#define SYS_ADRLO   0x04          /* System address lo */
#define SYS_ADRHI   0x06          /* System address high */
#define SYS_DATLO   0x08          /* System data low */
#define SYS_DATHI   0x0A          /* System data high (instruction access) */
#define SYS_DATRC   0x0C          /* System data read/clear operation */
#define SYS_INTR    0x0E          /* System interrupt register */

/*****************************************************************************
 * SYS_CTL register bit definitions (aka ISA Slave Control Register)
****************************************************************************/
#define SYS_CTL_SUSPEND  0x01     /* Suspend chip clocks */
#define SYS_CTL_RESET    0x02     /* 3780i soft reset */
#define SYS_CTL_CFG      0x04     /* Enable configuration mode */
#define SYS_CTL_NORMAL   0x00     /* Normal operating mode */

/*****************************************************************************
 * SYS_CTL2 register bit definitions (aka HBridge Control Register)
 ****************************************************************************/
#define SYS_CTL2_INIT    0x03     /* Memory address auto inc, x86 IRQ enabled */

/****************************************************************************
 * 3780i Configuration Space offsets
 *****************************************************************************/
#define CFG_ADDRESS 0x02          /* Config Address register (only in CFG mode) */
#define CFG_DATA    0x03          /* Config data register (only in CFG mode) */

#define SNDBLSTCR1  0x00          /* Soundblaster Configuration register 1 */
#define SNDBLSTCR2  0x01          /* Soundblaster Configuration register 2 */
#define SNDBLSTCR3  0x02          /* Soundblaster Configuration register 3 */
#define UARTCR1     0x03          /* UART Configuration register 1 */
#define UARTCR2     0x04          /* UART Configuration register 2 */
#define MIDICR1     0x05          /* MIDI Configuration register 1 */
#define MIDICR2     0x06          /* MIDI Configuration register 2 */
#define ISAHBCR1    0x07          /* ISA H-Bridge configuration register 1 */
#define ISAHBCR2    0x08          /* ISA H-Bridge configuration register 2 */
#define ISABMCR1    0x09          /* ISA BusMaster configuration register 1 */
#define ISABMCR2    0x0A          /* ISA BusMaster configuration register 2 */
#define ISAEXCR1    0x0B          /* ISA External device config register 1 */
#define ISAEXCR2    0x0C          /* ISA External device config register 2 */
#define ISAEXCR3    0x0D          /* ISA External device config register 3 */
#define ISAJOYEN    0x0E          /* ISA Joystick Enable Register */
#define ISAHBTMR    0x011         /* ISA H-Bridge timer load value */

/****************************************************************************
 * 3780i Miscellaneous Control Registers definitions
 ****************************************************************************/
#define MSAIOHI16   0x8000        /* Mwave System Address I/O space Hi 16 bits */
#define CIDRLO16    0x0000        /* Chip ID Register */
#define MBDRLO16    0x0580        /* MSP Boot Domain Register */
#define CRRLO16     0x0588        /* Chip Reset Register */
#define CCRLO16     0x058A        /* Clock Control Register */
#define UART_FCR    0x010C        /* UART FIFO Control register */

#define CCR_INIT    0x2520        /* 39.1608 Mhz M=37, N=32 XTAL=33.8688 Mhz */
#define CRR_INIT    0x0048        /* Chip Reset Register */

/*---------------------------------------------------------------------
 * define offsets from ComPort base address to various UART registers
 ---------------------------------------------------------------------*/
#define   REG_IER          1         /* Interrupt Enable Reg (DLAB=0) */
#define   REG_FCR          2         /* FIFO Control Register */
#define   REG_LCR          3         /* Line Control Reg */
#define   REG_MCR          4         /* Modem Control Reg */
#define   REG_SCR          7         /* Scratch Reg */
#define   REG_DLL          0         /* Divisor Latch LS Byte (DLAB=1) */
#define   REG_DLM          1         /* Divisor Latch MS Byte (DLAB=1) */

/*---------------------------------------------------------------------
 * define constants for Line control reg divisor latch access bit (DLAB)
 ---------------------------------------------------------------------*/
#define   LCR_DLAB         0x80      /* divisor latch access bit set mask */
#define   NOT_LCR_DLAB     0x7F      /* divisor latch access bit reset mask */

/***********************************************
 * MSP 3780i family chip IDs
 ************************************************/
#define CHIPID_3780i       0x0112    /* MDSP3780 chip ID */

/****************************************************************************
 *
 *  DSP Hardware Interface service packet
 *  (Hardware Interface ID = HWIID_DSP)
 *
 *****************************************************************************/
struct IDSPHW_Struc {
   unsigned short IDSPHW_SlotNum  ;
   unsigned short IDSPHW_AdapterID;
   unsigned short IDSPHW_CommPort ;
   unsigned short IDSPHW_IRQNumber;
};
#define IDSPHW_Struc_INIT  {0,0,0,0 }

/****************************************************************************
 *
 *  SoundBlaster Hardware Interface service packet
 *  (Hardware Interface ID = HWIID_SBLAST)
 *
 ****************************************************************************/
struct ISBLASTHW_Struc {
   unsigned short ISBLASTHW_Enable    ;
   unsigned short ISBLASTHW_Address   ;
   unsigned short ISBLASTHW_IRQNumber ;
   unsigned short ISBLASTHW_DMAChannel;
};
#define ISBLASTHW_Struc_INIT    {0,0,0,0}

/****************************************************************************
 * A data structure of the following type is maintained for each DSP
 ****************************************************************************/
struct DSPD_Struc {
   unsigned long  DSPD_DSPDescr    ;
   struct IDSPHW_Struc DSPD_pIDSPHW;
   unsigned short DSPD_BaseIo      ;
   unsigned long  DSPD_IoDelay     ;
   unsigned short DSPD_ChipId      ;
   unsigned short DSPD_Kips100     ;
   unsigned long  DSPD_IRQHandle   ;
   unsigned short DSPD_DMAChannel  ;
   unsigned long  DSPD_DMAHandle   ;
   unsigned short DSPD_CommIRQ     ;
   unsigned short DSPD_MIDIenable  ;
   unsigned short DSPD_MIDIaddr    ;
   unsigned short DSPD_MIDIirq     ;
   struct ISBLASTHW_Struc DSPD_pISBLASTHW;
   unsigned long  DSPD_Status       ;
};
#define DSPD_Struc_INIT \
      {0,IDSPHW_Struc_INIT,0,40,0,0,0,0,0,0,0,0,0 \
      ,ISBLASTHW_Struc_INIT,DSP_NOERROR}


/****************************************************************************
 *									     
 *  structure definitions for MSP 3780i configuration registers
 *
 *****************************************************************************/
typedef struct sndblstcfg1
{
  unsigned int IRQ_Type    :2;       /* active hi/lo, pulse/level */
  unsigned int SB_IRQ      :3;       /* 3,4,5,7,9,10,11 */
  unsigned int DMA_Channel :3;       /* 0,1,3,6,7 */
} SNDBLSTCFG1TYPE;
#define SNDBLSTCFG1DEFAULT {3,0,0}

typedef struct sndblstcfg2
{
  unsigned int SB_Addr     :2;       /* 220, 240, 260,  280 */
   unsigned int reserved    :6;
} SNDBLSTCFG2TYPE;
#define SNDBLSTCFG2DEFAULT {0,0}

typedef struct sndblstcfg3
{
  unsigned int SB_Enable   :1;       /* 0=disable I/O decode; hi-Z IRQ & DMA pins */
   unsigned int reserved    :7;
} SNDBLSTCFG3TYPE;
#define SNDBLSTCFG3DEFAULT {1,0}

typedef struct uartcfg1
{
  unsigned int IRQ_Type    :2;       /* active hi/lo, pulse/level*/
  unsigned int UART_IRQ    :3;       /* 3,4,5,7,9,10 */
  unsigned int ComPort     :2;       /* 3F8, 2F8, 3E8, 2E8 */
  unsigned int reserved    :1;
} UARTCFG1TYPE;
#define UARTCFG1DEFAULT {3,0,0,0}

typedef struct uartcfg2
{
  unsigned int UART_Enable :1;       /* 0=disable I/O address decode, hi-Z IRQ pin */
   unsigned int reserved    :7;
} UARTCFG2TYPE;
#define UARTCFG2DEFAULT {1,0}

typedef struct midicfg1
{
  unsigned int IRQ_Type    :2;       /* active hi/lo, pulse/level */
  unsigned int MIDI_IRQ    :3;       /* 3,4,5,7,9,10,11 */
  unsigned int MIDI_Addr   :1;       /* 0=330-332, 1=300-302 */
   unsigned int reserved    :2;
} MIDICFG1TYPE;
#define MIDICFG1DEFAULT {0,0,0,0}

typedef struct midicfg2
{
  unsigned int MIDI_Enable :1;       /* 0=disable I/O address decode, hi-Z IRQ pin */
   unsigned int reserved    :7;
} MIDICFG2TYPE;
#define MIDICFG2DEFAULT {0,0}

typedef struct hbridgecfg1
{
  unsigned int IRQ_Type    :2;       /* active hi/lo, pulse/level */
  unsigned int MSP_IRQ     :3;       /* 3,4,5,7,9,10,11 */
  unsigned int AccessMode  :1;       /* 0=byte mode, 1=word mode */
   unsigned int reserved    :2;
} HBRIDGECFG1TYPE;
#define HBRIDGECFG1DEFAULT {3,0,1,0}

typedef struct hbridgecfg2
{
  unsigned int HB_Enable   :1;       /* 0=disable I/O address decode, hi-Z IRQ pin */
   unsigned int reserved    :7;
} HBRIDGECFG2TYPE;
#define HBRIDGECFG2DEFAULT {1,0}

typedef struct busmastercfg1
{
  unsigned int DMA_Channel :3;       /* 0,1,3,6,7 */
  unsigned int NumTrans    :2;       /* number of transfers per grant */
  unsigned int ReRequest   :2;       /* bus master re-request time */
  unsigned int MemCS16     :1;       /* memory interface mode */
} BUSMASTERCFG1TYPE;
#define BUSMASTERCFG1DEFAULT {0,3,1,0}

typedef struct busmastercfg2
{
  unsigned int Cmd_Width   :2;       /* (472 | 413 | 354 | 295) nanoseconds */
   unsigned int reserved    :6;
} BUSMASTERCFG2TYPE;
#define BUSMASTERCFG2DEFAULT {3,0}


/****************************************************************************
 *
 * The next data structure is for communication of configuration
 * information between the VXD and PDD. There are two structures,
 * one is common to all 3780s, and another is card specific.
 *
 ****************************************************************************/

#define MSP3780CONFIGCheckSum 0x55
#define MSP3780CONFIGAddress  0xFFE0

typedef struct
   {
   unsigned short BaseAddress;     /* MSP Base address  */
   unsigned short Kips100;         /* IPS/100,000 */
   unsigned char  MSPenable;       /* Enable MSP */
   unsigned char  MSPIrq;          /* MSP IRQ */
   unsigned char  MSPdma;          /* MSP Dma channel */
   unsigned char  ComEnable;       /* COM Port Enable */
   unsigned short ComAddr;         /* COM Port Base address */
   unsigned char  ComIrq;          /* COM Port IRQ */
   unsigned char  MIDIenable;      /* MPU Port Enable */
   unsigned short MIDIaddr;        /* MIDI Base address */
   unsigned char  MIDIIrq;         /* MIDI IRQ */
   unsigned char  SBenable;        /* Sound Blaster Enable */
   unsigned short SBAddr;          /* Sound Blaster Base Address */
   unsigned char  SBIrq;           /* Sound Blaster IRQ level */
   unsigned char  SBdma;           /* Sound Blaster DMA Channel */
   unsigned char  ATmode;          /* 0:Original, 1:Enhanced, ... */
   unsigned char  ReRequest;       /* 5, 10, 20, or 40 (1=100ns) */
   unsigned char  NumTrans;        /* 2, 4, 8, or 16 */
   unsigned char  MemCS16;         /* 0: Ignore  1:Use */
   unsigned char  AudioClkInvert;  /* 0: Don't invert, 1: invert clock */
   unsigned short CardId;          /* Card type identifier */
   unsigned char  reserved[4];
   unsigned char  CheckSum;
   } MSP3780CONFIG;


/* ---------------------------------------------------------------------- */
/*   MemTransfer Types                                                    */
/* ---------------------------------------------------------------------- */
#define MEMXFER_Data_Read             0             /* Read data from DSP */
#define MEMXFER_Data_Write            1            /* Write data to DSP   */
/* ---------------------------------------------------------------------- */

/* Common Function Prototypes ------------------------------------------- */
extern void TransferDSPData(unsigned int BaseAddress, unsigned int TransferType,
			    ULONG DSPAddress, void *PCAddress, int ByteCount);

extern void ResetDSP (unsigned short BaseAddress, int Halt);

extern void SaveUARTRegs(USHORT BaseAddress);

extern void RestoreUARTRegs(void);

