/*
 * Decompiled with CFR 0.152.
 */
package apollo.dataadapter.genbank;

import apollo.dataadapter.genbank.GenbankAdapter;
import apollo.dataadapter.genbank.GenbankFormatter;
import apollo.datamodel.AnnotatedFeature;
import apollo.datamodel.AnnotatedFeatureI;
import apollo.datamodel.Comment;
import apollo.datamodel.CurationSet;
import apollo.datamodel.DbXref;
import apollo.datamodel.Exon;
import apollo.datamodel.RangeI;
import apollo.datamodel.SeqFeatureI;
import apollo.datamodel.SequenceI;
import apollo.datamodel.StrandedFeatureSetI;
import apollo.datamodel.Transcript;
import apollo.datamodel.seq.GenbankSequence;
import apollo.util.DateUtil;
import apollo.util.IOUtil;
import java.io.File;
import java.io.FileOutputStream;
import java.io.PrintStream;
import java.util.Date;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Vector;

public class GenbankReport {
    private static final String LOCUS = "LOCUS";
    private static final String DEFINITION = "DEFINITION";
    private static final String ACCESSION = "ACCESSION";
    private static final String VERSION = "VERSION";
    private static final String KEYWORDS = "KEYWORDS";
    private static final String SOURCE = "SOURCE";
    private static final String ORGANISM = "ORGANISM";
    private static final String REFERENCE = "REFERENCE";
    private static final String AUTHORS = "AUTHORS";
    private static final String TITLE = "TITLE";
    private static final String JOURNAL = "JOURNAL";
    private static final String MEDLINE = "MEDLINE";
    private static final String COMMENT = "COMMENT";
    private static final String FEATURES = "FEATURES";
    private static final String ORIGIN = "ORIGIN";
    private static final String DEFAULT_ORGANISM = "Homo sapiens";
    private PrintStream out = null;
    private CurationSet cs = null;
    private String organism = "Homo sapiens";
    SequenceI refseq;

    public GenbankReport(CurationSet cs) {
        this.setCurationSet(cs);
        this.organism = cs.getOrganism();
        if (this.organism == null) {
            this.organism = "";
        }
        this.refseq = cs.getRefSequence();
    }

    public CurationSet getCurationSet() {
        return this.cs;
    }

    public void setCurationSet(CurationSet cs) {
        this.cs = cs;
    }

    public PrintStream getPrintStream() {
        return this.out;
    }

    public void setPrintStream(PrintStream out) {
        this.out = out;
    }

    public String getLocus() {
        if (this.getProp(LOCUS) != null) {
            return this.getProp(LOCUS, "");
        }
        String acc = this.refseq.getAccessionNo();
        if (acc == null) {
            acc = this.refseq.getName();
        }
        String res_type = this.refseq.getResidueType() == "RNA" ? "mRNA" : "DNA ";
        return GenbankFormatter.getHeading(LOCUS) + GenbankFormatter.padBetweenWithSpaces(acc, this.refseq.getLength() + "", 28) + " bp    " + res_type + "    linear   " + GenbankReport.division(this.organism) + " " + DateUtil.dateNoTime(new Date()) + "\n";
    }

    public String getDefinition() {
        if (this.getProp(DEFINITION) != null) {
            return this.getProp(DEFINITION, "");
        }
        String def = this.refseq.getDescription();
        if (def == null || def.equals("")) {
            def = this.refseq.getAccessionNo();
        }
        if (def == null || def.equals("")) {
            def = this.cs.getName();
        }
        return GenbankFormatter.breakSingle(GenbankFormatter.getHeading(DEFINITION) + def + "\n");
    }

    public String getAccession() {
        if (this.getProp(ACCESSION) != null) {
            return this.getProp(ACCESSION, "");
        }
        String acc = this.refseq.getAccessionNo();
        if (acc == null) {
            acc = ".";
        }
        return GenbankFormatter.getHeading(ACCESSION) + acc + "\n";
    }

    public String getVersion() {
        if (this.getProp(VERSION) != null) {
            return this.getProp(VERSION, "");
        }
        return GenbankFormatter.getHeading(VERSION) + "." + "\n";
    }

    public String getKeywords() {
        if (this.getProp(KEYWORDS) != null) {
            return this.getProp(KEYWORDS, "");
        }
        return GenbankFormatter.getHeading(KEYWORDS) + "." + "\n";
    }

    public String getSource() {
        if (this.getProp(SOURCE) != null) {
            return this.getProp(SOURCE, "");
        }
        String org = this.organism.equals("") ? "." : this.organism;
        return GenbankFormatter.getHeading(SOURCE) + org + GenbankReport.commonName(this.organism) + "\n";
    }

    public String getOrganism() {
        if (this.getProp(SOURCE) != null && this.getProp(SOURCE).indexOf(ORGANISM) > 0) {
            return "";
        }
        if (this.getProp(ORGANISM) != null) {
            return this.getProp(ORGANISM, "");
        }
        String org = this.organism.equals("") ? "." : this.organism;
        return GenbankFormatter.getSubHeading(ORGANISM) + org + GenbankReport.taxonomy(this.organism) + "\n";
    }

    public String getReference() {
        if (this.getProp(REFERENCE) != null) {
            return this.getProp(REFERENCE, ".");
        }
        int start = this.cs.getStart();
        int end = this.cs.getEnd();
        return GenbankFormatter.getHeading(REFERENCE) + "1  (bases " + start + " to " + end + ")\n";
    }

    public String getComment() {
        if (this.getProp(COMMENT) != null) {
            return this.getProp(COMMENT, "");
        }
        return "";
    }

    public void printHeader() {
        this.out.print(this.getLocus());
        this.out.print(this.getDefinition());
        this.out.print(this.getAccession());
        this.out.print(this.getVersion());
        this.out.print(this.getKeywords());
        this.out.print(this.getSource());
        this.out.print(this.getOrganism());
        this.out.print(this.getReference());
        this.out.print(this.getComment());
    }

    private String getRange(RangeI r) {
        int start = r.getStart();
        int end = r.getEnd();
        return this.localCoords(start, end);
    }

    private String localCoords(int start, int end) {
        start = start - this.cs.getStart() + 1;
        end = end - this.cs.getStart() + 1;
        return GenbankReport.printRange(start, end);
    }

    private static String printRange(int s, int e) {
        if (s > e) {
            return "complement(" + e + ".." + s + ")";
        }
        return s + ".." + e;
    }

    private static String printRange(String s, String e, String before_s, String before_e) {
        try {
            String s_int = s;
            if (before_s != null && !before_s.equals("")) {
                s_int = s_int.substring(1);
            }
            String e_int = e;
            if (before_e != null && !before_e.equals("") && (e_int = e_int.substring(1)).indexOf(">") > 0) {
                e_int = e_int.substring(0, e_int.indexOf(">"));
            }
            int start = Integer.parseInt(s_int);
            int end = Integer.parseInt(e_int);
            if (before_s == null) {
                before_s = "";
            }
            if (before_e == null) {
                before_e = "";
            }
            if (start > end) {
                return "complement(" + before_e + e + ".." + before_s + s + ")";
            }
            return before_s + s + ".." + before_e + e;
        }
        catch (Exception err) {
            System.out.println("printRange: couldn't parse range " + s + "-" + e);
            return "";
        }
    }

    public void printTranscript(Transcript transcript) {
        AnnotatedFeatureI gene;
        String id;
        if (transcript.getTranslationStart() == 0) {
            try {
                transcript.calcTranslationStartForLongestPeptide();
            }
            catch (NullPointerException e) {
                // empty catch block
            }
        }
        if ((id = (gene = transcript.getGene()).getId()).equals("") || id.equals("no_name")) {
            id = gene.getName();
        }
        this.printmRNA(transcript);
        this.out.println(GenbankFormatter.getFeatureItem("locus_tag", id));
        this.printDbXref(transcript);
        this.printOtherFields(transcript, "mRNA");
        this.printCDS(transcript);
        this.out.println(GenbankFormatter.getFeatureItem("locus_tag", id));
        this.out.println(GenbankFormatter.getFeatureItem("product", GenbankAdapter.getSequenceName(transcript.getGene(), transcript.getPeptideSequence(), transcript)));
        this.out.println(GenbankFormatter.getFeatureItem("prot_desc", GenbankAdapter.getProteinDesc(transcript.getGene(), transcript)));
        this.out.println(GenbankFormatter.getFeatureItem("protein_id", transcript.getProperty("protein_id")));
        this.printFunction(transcript);
        this.printNotes(transcript);
        String codon_start = transcript.getProperty("codon_start");
        if (codon_start == null || codon_start.equals("")) {
            codon_start = "1";
        }
        this.out.println(GenbankFormatter.getFeatureItem("codon_start", codon_start));
        this.printDbXref(transcript);
        try {
            this.out.println(GenbankFormatter.getFeatureItem("translation", transcript.getPeptideSequence().getResidues()));
        }
        catch (NullPointerException e) {
            // empty catch block
        }
        this.printOtherFields(transcript, "CDS");
    }

    public void printGene(AnnotatedFeature gene) {
        String type;
        String type_for_name = type = gene.getFeatureType();
        if (type.equals("transposable_element")) {
            type = "repeat_region";
            type_for_name = "transposon";
        } else if (type.length() > 15) {
            type_for_name = type = type.substring(0, 15);
        }
        this.out.println(GenbankFormatter.getFeatureHeading(type, this.getRange(gene)));
        this.out.println(GenbankFormatter.getFeatureItem(type_for_name, gene.getName()));
        String id = gene.getId();
        if (id.equals("") || id.equals("no_name")) {
            id = gene.getName();
        }
        this.out.println(GenbankFormatter.getFeatureItem("locus_tag", id));
        this.printFunction(gene);
        this.printNotes(gene);
        this.printDbXref(gene);
        this.printOtherFields(gene, "gene");
        if (gene.isProteinCodingGene()) {
            Iterator iter = gene.getFeatures().iterator();
            while (iter.hasNext()) {
                this.printTranscript((Transcript)iter.next());
            }
        }
    }

    private void printmRNA(Transcript transcript) {
        String mrna = "";
        int offset = this.cs.getStart() - 1;
        int numExons = transcript.getExons().size();
        if (numExons > 1) {
            mrna = mrna + "join(";
        }
        Vector exons = transcript.getExons();
        String trans_start = transcript.isMissing5prime() ? "<" : "";
        String trans_stop = transcript.isMissing3prime() ? ">" : "";
        for (int i = 0; i < numExons; ++i) {
            Exon exon = (Exon)exons.elementAt(i);
            int start = exon.getStart() - offset;
            int end = exon.getEnd() - offset;
            String exon_start = trans_start.equals("<") && i == 0 ? trans_start : "";
            String exon_stop = trans_stop.equals(">") && i == numExons - 1 ? trans_stop : "";
            mrna = mrna + GenbankReport.printRange(start + "", end + "", exon_start, exon_stop);
            if (i >= numExons - 1) continue;
            mrna = mrna + ", ";
        }
        if (numExons > 1) {
            mrna = mrna + ")";
        }
        String multiLinemRNA = GenbankFormatter.breakSingle(GenbankFormatter.getFeatureHeading("mRNA", mrna));
        this.out.println(multiLinemRNA.replaceAll(", ", ","));
    }

    private void printCDS(Transcript transcript) {
        SeqFeatureI start_span;
        AnnotatedFeatureI gene = transcript.getGene();
        String gene_has_start = GenbankAdapter.get5primeStart(gene);
        String gene_has_stop = GenbankAdapter.get3primeStop(gene);
        String gb_type = GenbankAdapter.getGenbankType(gene.getFeatureType());
        String CDS_start = gene_has_start.equals("") ? GenbankAdapter.get5primeStart(transcript) : gene_has_start;
        String CDS_stop = gene_has_stop.equals("") ? GenbankAdapter.get3primeStop(transcript) : gene_has_stop;
        int tss = transcript.getTranslationStart();
        int phase = 0;
        if (gene.isMissing5prime() || transcript.isMissing5prime()) {
            phase = transcript.getFeaturePosition(tss) - 1;
        }
        int start_index = (start_span = transcript.getFeatureContaining(tss)) != null ? transcript.getFeatureIndex(start_span) : 0;
        int offset = this.cs.getStart() - 1;
        int end_base = transcript.getLastBaseOfStopCodon();
        Vector exons = transcript.getFeatures();
        int numExons = exons.size();
        boolean found_end = false;
        StringBuffer buf = new StringBuffer();
        if (numExons > 1) {
            buf.append("join(");
        }
        for (int i = start_index; i < numExons && !found_end; ++i) {
            String end;
            Exon exon = (Exon)exons.elementAt(i);
            boolean contains_tss = i == start_index;
            found_end = exon.contains(end_base);
            String start = contains_tss && phase == 0 ? CDS_start + (tss - offset) : (contains_tss && phase > 0 ? CDS_start + (exon.getStart() - offset) : "" + (exon.getStart() - offset));
            int na_except_pos = 0;
            if (na_except_pos > 0 && exon.contains(na_except_pos)) {
                end = "" + (na_except_pos - transcript.getStrand());
                end = "" + (na_except_pos + transcript.getStrand());
                contains_tss = false;
            }
            end = found_end ? "" + (end_base - offset) + CDS_stop : "" + (exon.getEnd() - offset);
            buf.append(GenbankReport.printRange(start, end, CDS_start, CDS_stop));
            if (i >= numExons - 1) continue;
            buf.append(", ");
        }
        if (numExons > 1) {
            buf.append(")");
        }
        String multiLineCDS = GenbankFormatter.breakSingle(GenbankFormatter.getFeatureHeading("CDS", buf.toString()));
        this.out.println(multiLineCDS.replaceAll(", ", ","));
    }

    private static String getDbXrefStr(DbXref dx) {
        String xref = dx.getIdValue();
        if (xref.startsWith(dx.getDbName())) {
            return xref;
        }
        return dx.getDbName() + ":" + xref;
    }

    public void printNotes(AnnotatedFeatureI annotation) {
        if (annotation.getProperty("notes") != null && !annotation.getProperty("notes").equals("")) {
            this.out.println(GenbankFormatter.getFeatureItem("notes", annotation.getProperty("notes")));
            return;
        }
        StringBuffer b = new StringBuffer();
        Iterator iter = annotation.getComments().iterator();
        while (iter.hasNext()) {
            Comment comment = (Comment)iter.next();
            b.append(comment.getText());
            if (!iter.hasNext()) continue;
            b.append(";");
        }
        if (b.length() > 0) {
            this.out.println(GenbankFormatter.getFeatureItem("note", b.toString()));
        }
    }

    public void printFunction(AnnotatedFeature annotation) {
        if (annotation.getProperty("function") != null && !annotation.getProperty("function").equals("")) {
            this.out.println(GenbankFormatter.getFeatureItem("function", annotation.getProperty("function")));
            return;
        }
        if (annotation.getDescription() == null || annotation.getDescription().equals("")) {
            return;
        }
        this.out.println(GenbankFormatter.getFeatureItem("function", annotation.getDescription()));
    }

    public void printDbXref(AnnotatedFeatureI annotation) {
        Iterator iter = annotation.getDbXrefs().iterator();
        while (iter.hasNext()) {
            DbXref dx = (DbXref)iter.next();
            this.out.println(GenbankFormatter.getFeatureItem("db_xref", GenbankReport.getDbXrefStr(dx)));
        }
    }

    public void printFeatures() {
        if (this.cs == null) {
            return;
        }
        this.out.println(GenbankFormatter.getFeature(FEATURES) + "Location/Qualifiers");
        StrandedFeatureSetI annotations = this.cs.getAnnots();
        this.printSeqSource(annotations);
        for (int i = 0; i < annotations.size(); ++i) {
            AnnotatedFeature gene = (AnnotatedFeature)annotations.getFeatureAt(i);
            this.printGene(gene);
        }
    }

    private void printSeqSource(RangeI annotations) {
        this.out.println(GenbankFormatter.getFeatureHeading("source", this.getRange(this.refseq.getRange())));
        this.out.println(GenbankFormatter.getFeatureItem("organism", this.organism));
        if (this.getProp("mol_type") == null) {
            this.out.println(GenbankFormatter.getFeatureItem("mol_type", "genomic DNA"));
        } else {
            this.out.println(GenbankFormatter.getFeatureItem("mol_type", this.getProp("mol_type", "")));
        }
        if (this.cs.getChromosome() != null) {
            this.out.println(GenbankFormatter.getFeatureItem("chromosome", this.cs.getChromosome()));
        }
        this.printOtherFields(null, "source");
    }

    private void printSequence() {
        if (this.refseq.getResidues() == null) {
            return;
        }
        this.out.println(GenbankFormatter.getHeading(ORIGIN));
        this.out.println(GenbankFormatter.formatSequence(this.refseq.getResidues()));
    }

    public void printFooter() {
        this.out.println("//");
    }

    public void writeReport(PrintStream out) {
        if (out == null) {
            System.err.println("GenbankReport.writeReport: PrintStream is null");
            return;
        }
        this.setPrintStream(out);
        this.printHeader();
        this.printFeatures();
        this.printSequence();
        this.printFooter();
    }

    public static void save(CurationSet cs, String filename) {
        try {
            filename = IOUtil.findFile(filename, true);
            File file = new File(filename);
            System.out.println("Saving GenBank report (in human-readable format) to file " + filename);
            GenbankReport.save(cs, file);
        }
        catch (Exception e) {
            System.out.println("Error in GenbankReport.save(cs, " + filename + "): " + e);
            return;
        }
        System.out.println("Done saving GenBank report");
    }

    public static void save(CurationSet cs, File file) {
        try {
            if (file == null) {
                System.err.println("GenbankReport.save: file pointer is null");
            }
            System.out.println("Saving human-readable GenBank report in file " + file + "...");
            GenbankReport report = new GenbankReport(cs);
            PrintStream out = new PrintStream(new FileOutputStream(file));
            report.writeReport(out);
            out.close();
        }
        catch (Exception e) {
            System.out.println("Error saving GenBank report in " + file + ": " + e);
        }
    }

    private static String commonName(String org) {
        if (org.equals(DEFAULT_ORGANISM)) {
            return " (human)";
        }
        if (org.equals("Drosophila melanogaster")) {
            return " (fruit fly)";
        }
        return "";
    }

    private static String taxonomy(String org) {
        if (org.equals(DEFAULT_ORGANISM)) {
            return "\n            Eukaryota; Metazoa; Chordata; Craniata; Vertebrata; Euteleostomi;\n            Mammalia; Eutheria; Primates; Catarrhini; Hominidae; Homo.";
        }
        if (org.equals("Drosophila melanogaster")) {
            return "\n            Eukaryota; Metazoa; Arthropoda; Hexapoda; Insecta; Pterygota;\n            Neoptera; Endopterygota; Diptera; Brachycera; Muscomorpha;\n            Ephydroidea; Drosophilidae; Drosophila.";
        }
        return "";
    }

    private static String division(String org) {
        if (org.equals(DEFAULT_ORGANISM)) {
            return "PRI";
        }
        if (org.equals("Drosophila melanogaster")) {
            return "INV";
        }
        return "   ";
    }

    private String getProp(String field) {
        return this.getProp(field, null);
    }

    private String getProp(String field, String defaultValue) {
        if (!(this.refseq instanceof GenbankSequence)) {
            return defaultValue;
        }
        GenbankSequence gs = (GenbankSequence)this.refseq;
        if (gs.getProperty(field) == null) {
            return defaultValue;
        }
        return gs.getProperty(field);
    }

    private void printOtherFields(AnnotatedFeature feature, String whatKind) {
        if (whatKind.equals("mRNA") || whatKind.equals("CDS")) {
            Transcript transcript = (Transcript)feature;
            Hashtable props = transcript.getProperties();
            this.printProps(props, whatKind);
        } else if (whatKind.equals("gene")) {
            AnnotatedFeature gene = feature;
            Hashtable props = gene.getProperties();
            this.printProps(props, whatKind);
        } else if (whatKind.equals("source")) {
            if (!(this.refseq instanceof GenbankSequence)) {
                return;
            }
            GenbankSequence gs = (GenbankSequence)this.refseq;
            Hashtable props = gs.getProperties();
            this.printProps(props, whatKind);
        }
    }

    private void printProps(Hashtable props, String whatKind) {
        if (props != null) {
            Enumeration en = props.keys();
            while (en.hasMoreElements()) {
                String key = (String)en.nextElement();
                if (!this.wantedField(key, whatKind)) continue;
                this.out.println(GenbankFormatter.getFeatureItem(key, (String)props.get(key)));
            }
        }
    }

    private boolean wantedField(String field, String whatKind) {
        if (whatKind.equals("mRNA")) {
            return !field.equals("locus_tag") && !field.equals("product") && !field.equals("codon_start") && !field.equals("translation") && !field.equals("protein_id") && !field.equals("db_xref");
        }
        if (whatKind.equals("CDS")) {
            return !field.equals("locus_tag") && !field.equals("product") && !field.equals("prot_desc") && !field.equals("protein_id") && !field.equals("codon_start") && !field.equals("translation");
        }
        if (whatKind.equals("gene")) {
            return !field.equals("locus_tag") && !field.equals("function") && !field.equals("notes") && !field.equals("db_xref");
        }
        if (whatKind.equals("source")) {
            return !field.equals("organism") && !field.equals("mol_type") && !field.equals("chromosome") && !field.equalsIgnoreCase(DEFINITION) && !field.equalsIgnoreCase(REFERENCE) && !field.equalsIgnoreCase(FEATURES) && !field.equalsIgnoreCase(ORIGIN) && !field.equalsIgnoreCase(SOURCE) && !field.equalsIgnoreCase(VERSION) && !field.equalsIgnoreCase(ACCESSION) && !field.equalsIgnoreCase(KEYWORDS) && !field.equalsIgnoreCase(LOCUS) && !field.equalsIgnoreCase(COMMENT);
        }
        return true;
    }
}

