/*
 * Decompiled with CFR 0.152.
 */
package apollo.analysis.filter;

import apollo.analysis.filter.AnalysisFilterI;
import apollo.analysis.filter.AnalysisInput;
import apollo.analysis.filter.Compress;
import apollo.analysis.filter.Coverage;
import apollo.analysis.filter.TwilightFilter;
import apollo.dataadapter.exception.BopException;
import apollo.datamodel.CurationSet;
import apollo.datamodel.FeaturePairI;
import apollo.datamodel.FeatureSet;
import apollo.datamodel.FeatureSetI;
import apollo.datamodel.SeqFeatureI;
import apollo.datamodel.Sequence;
import apollo.datamodel.SequenceI;
import apollo.util.CigarUtil;
import apollo.util.SeqFeatureUtil;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Vector;
import org.bdgp.util.DNAUtils;

public class AnalysisFilter
implements AnalysisFilterI {
    protected AnalysisInput filter_input;
    protected static int CHIMERA_GAP = 50000;
    protected static String default_3prime = "3prime";
    protected String suffix_3prime = null;
    FeatureSetI forward_analysis = null;
    FeatureSetI reverse_analysis = null;

    public void cleanUp(CurationSet curation, String analysis_type, AnalysisInput filter_input) {
        this.filter_input = filter_input;
        if (!filter_input.runFilter()) {
            return;
        }
        SequenceI seq = curation.getRefSequence();
        String seq_name = seq != null ? seq.getName() : null;
        String type = filter_input.getAnalysisType();
        this.forward_analysis = this.getAnalysis(curation, analysis_type, 1, type);
        this.reverse_analysis = this.getAnalysis(curation, analysis_type, -1, type);
        this.debugAnalysis(this.forward_analysis);
        this.debugAnalysis(this.reverse_analysis);
        if (filter_input.splitFrames()) {
            System.out.println("Splitting hits into separate frames");
            this.splitHitsByFrame(this.forward_analysis);
            this.splitHitsByFrame(this.reverse_analysis);
        }
        if (filter_input.useCoincidence()) {
            int percent_overlap = filter_input.getCoincidence();
            System.out.println("Removing HSPs that overlap by " + percent_overlap + "%");
            this.cleanUpSpanCoincidents(this.forward_analysis, percent_overlap);
            this.cleanUpSpanCoincidents(this.reverse_analysis, percent_overlap);
        }
        if (filter_input.useAutonomousHSPs()) {
            System.out.println("Making each HSP an individual hit");
            this.promoteHSPsToHits(this.forward_analysis);
            this.promoteHSPsToHits(this.reverse_analysis);
        }
        if (filter_input.splitTandems()) {
            System.out.println("Splitting tandem hits");
            this.splitTandemHits(this.forward_analysis, filter_input);
            this.splitTandemHits(this.reverse_analysis, filter_input);
        }
        if (this.forward_analysis != null) {
            this.processHits(this.forward_analysis, filter_input);
        }
        if (this.reverse_analysis != null) {
            this.processHits(this.reverse_analysis, filter_input);
        }
        if (filter_input.trimPolyA()) {
            System.out.println("Removing polyA tails");
            if (this.forward_analysis != null) {
                this.removeTails(this.forward_analysis);
            }
            if (this.reverse_analysis != null) {
                this.removeTails(this.reverse_analysis);
            }
        }
        if (filter_input.revComp3Prime()) {
            this.suffix_3prime = filter_input.get3PrimeSuffix();
            System.out.println("Reverse complementing 3' EST reads (suffix=" + this.suffix_3prime + ")");
            Vector completed = new Vector();
            if (this.forward_analysis != null) {
                this.reverseESTs(this.forward_analysis, completed);
            }
            if (this.reverse_analysis != null) {
                this.reverseESTs(this.reverse_analysis, completed);
            }
        }
        if (filter_input.joinESTends()) {
            System.out.println("Join EST reads from same cDNA clones");
            this.joinESTs(this.forward_analysis, this.reverse_analysis);
        }
        if (filter_input.removeLowContent()) {
            int wordsize = filter_input.getWordSize();
            int max_ratio = filter_input.getMaxRatio();
            System.out.println("Removing hits with compression < " + max_ratio + "%" + " with wordsize=" + wordsize);
            this.removeLowContent(this.forward_analysis, wordsize, max_ratio);
            this.removeLowContent(this.reverse_analysis, wordsize, max_ratio);
        }
        if (filter_input.removeTwilights()) {
            System.out.println("Removing twilight spans");
            this.cleanupTwilights(this.forward_analysis, filter_input);
            this.cleanupTwilights(this.reverse_analysis, filter_input);
        }
        if (!filter_input.keepPolyA()) {
            System.out.println("Removing polyA predictions");
            this.cleanupByType(this.forward_analysis, "PlyA");
            this.cleanupByType(this.reverse_analysis, "PlyA");
        }
        if (!filter_input.keepPolyA()) {
            System.out.println("Removing promoter predictions");
            this.cleanupByType(this.forward_analysis, "Prom");
            this.cleanupByType(this.reverse_analysis, "Prom");
        }
        if (filter_input.limitAlignGap()) {
            int max_missing = filter_input.getMaxAlignGap();
            System.out.println("Removing hits with internal gaps > " + max_missing + "% of total length");
            this.removeGappedAlign(this.forward_analysis, max_missing);
            this.removeGappedAlign(this.reverse_analysis, max_missing);
        }
        if (filter_input.useLength()) {
            int min_length = filter_input.getMinLength();
            boolean is_percentage = filter_input.usePercentage();
            System.out.println("Removing hits with length < " + min_length + (is_percentage ? "%" : ""));
            this.removeShorties(this.forward_analysis, min_length, is_percentage);
            this.removeShorties(this.reverse_analysis, min_length, is_percentage);
        }
        if (filter_input.limitMaxExons()) {
            int max_exons = filter_input.getMaxExons();
            System.out.println("Removing hits with > " + max_exons + " HSPs");
            this.removeFragmented(this.forward_analysis, max_exons);
            this.removeFragmented(this.reverse_analysis, max_exons);
        }
        Vector regions = Coverage.sortRegions(this.forward_analysis, this.reverse_analysis);
        if (filter_input.removeShadows()) {
            System.out.println("Removing shadows");
            this.cleanUpShadows(regions, this.forward_analysis, this.reverse_analysis);
        }
        if (filter_input.limitCoverage()) {
            Coverage.cleanUp(regions, this.forward_analysis, this.reverse_analysis, filter_input.getMaxCover(), this);
        }
    }

    private void processHits(FeatureSetI analysis, AnalysisInput filter_input) {
        if (analysis == null) {
            System.err.println("AnalysisFilter.processHits: analysis is null");
            return;
        }
        boolean use_expect = filter_input.useExpect();
        double max_expect = filter_input.getMaxExpect();
        boolean use_score = filter_input.useScore();
        int min_score = filter_input.getMinScore();
        boolean use_identity = filter_input.useIdentity();
        int min_identity = filter_input.getMinIdentity();
        if (use_expect) {
            System.out.println("Removing hits with expect > " + max_expect);
        }
        if (use_score) {
            System.out.println("Removing hits with score < " + min_score);
        }
        if (use_identity) {
            System.out.println("Removing hits with identity < " + min_identity);
        }
        int hit_index = 0;
        SeqFeatureI sf = analysis.getFeatureAt(hit_index);
        while (hit_index < analysis.size()) {
            FeatureSetI next_hit;
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(hit_index);
            FeatureSetI featureSetI = next_hit = hit_index + 1 < analysis.size() ? (FeatureSetI)analysis.getFeatureAt(hit_index + 1) : null;
            if (use_expect && hit.getScore("expect") > max_expect) {
                analysis.deleteFeature(hit);
                this.debugFeature(hit, "Removed because expectation too high ");
            } else if (use_score && hit.getScore() < (double)min_score) {
                analysis.deleteFeature(hit);
                this.debugFeature(hit, "Removed because score " + hit.getScore() + " < " + min_score);
            } else if (use_identity && this.lowIdentity(hit, min_identity)) {
                analysis.deleteFeature(hit);
            }
            if (next_hit != null) {
                hit_index = analysis.getFeatureIndex(next_hit);
                continue;
            }
            hit_index = analysis.size();
        }
    }

    protected void cleanupByType(FeatureSetI fs, String type) {
        if (fs == null) {
            return;
        }
        for (int i = fs.size() - 1; i >= 0; --i) {
            SeqFeatureI sf = fs.getFeatureAt(i);
            if (sf.getTopLevelType().equals(type)) {
                fs.deleteFeature(sf);
                this.debugFeature(sf, "Removed because type was " + type);
                continue;
            }
            if (!(sf instanceof FeatureSetI)) continue;
            this.cleanupByType((FeatureSetI)sf, type);
        }
    }

    protected void removeShorties(FeatureSetI analysis, int min_length, boolean is_percentage) {
        if (analysis == null) {
            return;
        }
        for (int i = analysis.size() - 1; i >= 0; --i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            if (!this.tooShort(hit, min_length, is_percentage)) continue;
            analysis.deleteFeature(hit);
        }
    }

    private void debugAnalysis(FeatureSetI analysis) {
        if (analysis == null) {
            return;
        }
        for (int i = analysis.size() - 1; i >= 0; --i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            this.debugFeature(hit, "Feature is now ");
        }
    }

    protected boolean tooShort(FeatureSetI hit, int min_length, boolean is_percentage) {
        if (hit == null) {
            return false;
        }
        int span_cnt = hit.size();
        double hit_length = 0.0;
        for (int i = span_cnt - 1; i >= 0; --i) {
            FeaturePairI span = (FeaturePairI)hit.getFeatureAt(i);
            hit_length += (double)span.getHitFeature().length();
        }
        if (is_percentage) {
            SequenceI hit_seq = hit.getHitSequence();
            hit_length = hit_length * 100.0 / (double)hit_seq.getLength();
        }
        if (hit_length < (double)min_length) {
            this.debugFeature(hit, "This hit's length " + hit_length + " < " + min_length);
        }
        return hit_length < (double)min_length;
    }

    protected void removeGappedAlign(FeatureSetI analysis, int max_missing) {
        if (analysis == null) {
            return;
        }
        for (int i = analysis.size() - 1; i >= 0; --i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            int match_length = hit.getHitSequence().getLength();
            int max_gap = max_missing * match_length / 100;
            this.gappedAlignment(hit, max_gap, 0, 1);
            this.gappedAlignment(hit, max_gap, hit.size() - 1, hit.size() - 2);
            if (hit.size() != 0) continue;
            analysis.deleteFeature(hit);
            this.debugFeature(hit, "This hit has no spans left, where did they go?");
        }
    }

    private void gappedAlignment(FeatureSetI hit, int max_gap, int index1, int index2) {
        if (hit == null) {
            return;
        }
        if (hit.size() > 1) {
            FeaturePairI span1 = (FeaturePairI)hit.getFeatureAt(index1);
            FeaturePairI span2 = (FeaturePairI)hit.getFeatureAt(index2);
            int gap_length = this.gapBetween(span1.getHitFeature(), span2.getHitFeature());
            if (gap_length >= max_gap) {
                String msg = hit.getHitSequence().getName() + "\n\tDistance between span1 " + span1.getHitFeature().getLow() + "-" + span1.getHitFeature().getHigh() + " and span2 " + span2.getHitFeature().getLow() + "-" + span2.getHitFeature().getHigh() + "\n\tis " + gap_length + " bases, exceeding max " + max_gap;
                this.debugFeature(hit, msg);
                hit.deleteFeature(span1);
            }
        }
    }

    protected void removeFragmented(FeatureSetI analysis, int max_exons) {
        if (analysis == null) {
            return;
        }
        for (int i = analysis.size() - 1; i >= 0; --i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            if (hit.size() <= max_exons) continue;
            analysis.deleteFeature(hit);
            this.debugFeature(hit, "This hit had " + hit.size() + " HSPs, which is > max of " + max_exons);
        }
    }

    protected void removeLowContent(FeatureSetI analysis, int wordsize, int max_ratio) {
        if (analysis == null) {
            return;
        }
        int i = 0;
        while (i < analysis.size()) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            if (this.lacksContent(hit, wordsize, max_ratio)) {
                analysis.deleteFeature(hit);
                this.debugFeature(hit, "Deleted low content hit at ");
                continue;
            }
            ++i;
        }
    }

    private boolean lacksContent(FeatureSetI hit, int wordsize, int max_ratio) {
        boolean no_info = true;
        boolean debug = this.filter_input.debugFilter(hit.getHitSequence().getName());
        for (int i = 0; no_info && i < hit.size(); ++i) {
            SeqFeatureI span = hit.getFeatureAt(i);
            if (span.getExplicitAlignment().length() == 0) {
                System.err.println("ERROR!!  no sequence in span " + (i + 1));
                no_info = false;
                continue;
            }
            String alignment = span.getExplicitAlignment();
            double shrinkage = Compress.compress(alignment, wordsize, debug);
            no_info &= shrinkage <= (double)max_ratio;
        }
        return no_info;
    }

    private void splitHitsByFrame(FeatureSetI analysis) {
        FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(0);
        FeaturePairI span = (FeaturePairI)hit.getFeatureAt(0);
        SeqFeatureI matching_span = span.getHitFeature();
        String frame = matching_span.getProperty("frame");
        Hashtable<String, FeatureSetI> frames = new Hashtable<String, FeatureSetI>(6);
        if (frame != null && !frame.equals("")) {
            FeatureSetI current_hit;
            int i;
            int hit_cnt = analysis.size();
            Vector<FeatureSetI> new_frames = new Vector<FeatureSetI>();
            for (i = 0; i < hit_cnt; ++i) {
                frames.clear();
                hit = (FeatureSetI)analysis.getFeatureAt(i);
                span = (FeaturePairI)hit.getFeatureAt(0);
                matching_span = span.getHitFeature();
                frame = matching_span.getProperty("frame");
                frames.put(frame, hit);
                int j = 0;
                while (j < hit.size()) {
                    span = (FeaturePairI)hit.getFeatureAt(j);
                    matching_span = span.getHitFeature();
                    frame = matching_span.getProperty("frame");
                    current_hit = (FeatureSetI)frames.get(frame);
                    if (current_hit == null) {
                        hit.deleteFeature(span);
                        current_hit = this.makeNewHit(hit, span);
                        frames.put(frame, current_hit);
                        continue;
                    }
                    if (current_hit != hit) {
                        hit.deleteFeature(span);
                        this.addToHit(span, current_hit);
                        continue;
                    }
                    ++j;
                }
                Enumeration e = frames.elements();
                while (e.hasMoreElements()) {
                    current_hit = (FeatureSetI)e.nextElement();
                    if (current_hit == hit) continue;
                    new_frames.addElement(current_hit);
                }
            }
            for (i = 0; i < new_frames.size(); ++i) {
                current_hit = (FeatureSetI)new_frames.elementAt(i);
                analysis.addFeature(current_hit, true);
                this.debugFeature(current_hit, "Added new frame ");
            }
        }
    }

    private void removeTails(FeatureSetI analysis) {
        int polyA_length = 5;
        for (int hit_index = 0; hit_index < analysis.size(); ++hit_index) {
            int intron_length;
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(hit_index);
            if (hit.size() <= 1 || hit.isFlagSet(FeatureSet.POLYA_REMOVED)) continue;
            int index = hit.size() - 1;
            FeaturePairI span = (FeaturePairI)hit.getFeatureAt(index);
            String name = span.getHitFeature().getRefSequence().getName();
            String dna = span.getResidues();
            if (Compress.isPolyATail(dna, intron_length = Math.abs(span.getStart() - hit.getFeatureAt(index - 1).getEnd()), name + "-span:" + (index + 1), this.filter_input.debugFilter(name))) {
                hit.deleteFeature(span);
                this.debugFeature(span, "deleted polyA tail from end");
                continue;
            }
            span = (FeaturePairI)hit.getFeatureAt(0);
            dna = span.getResidues();
            if (!Compress.isPolyATail(dna, intron_length = hit.size() > 1 ? Math.abs(hit.getFeatureAt(1).getStart() - span.getEnd()) : 0, (name = span.getHitFeature().getRefSequence().getName()) + "-span:1", this.filter_input.debugFilter(name))) continue;
            hit.deleteFeature(span);
            this.debugFeature(span, "deleted polyA tail from beginning");
        }
    }

    protected void splitTandemHits(FeatureSetI analysis, AnalysisInput filter_input) {
        int hit_index = 0;
        boolean do_the_splits = true;
        while (hit_index < analysis.size()) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(hit_index);
            do_the_splits = this.splitTandemHit(analysis, hit, filter_input);
            hit_index = do_the_splits ? 0 : hit_index + 1;
        }
    }

    private boolean splitTandemHit(FeatureSetI analysis, FeatureSetI hit, AnalysisInput filter_input) {
        boolean changed;
        FeatureSetI current_hit = null;
        FeaturePairI prev_span = (FeaturePairI)hit.getFeatureAt(0);
        int span_index = 1;
        Vector<FeatureSetI> added_hits = new Vector<FeatureSetI>();
        while (span_index < hit.size()) {
            FeaturePairI next_span = (FeaturePairI)hit.getFeatureAt(span_index);
            if (!this.isSubjSequential(prev_span, next_span)) {
                hit.deleteFeature(next_span);
                current_hit = this.makeNewHit(hit, next_span);
                added_hits.addElement(current_hit);
            } else if (current_hit != null) {
                hit.deleteFeature(next_span);
                this.addToHit(next_span, current_hit);
            } else {
                ++span_index;
            }
            prev_span = next_span;
        }
        boolean bl = changed = added_hits.size() > 0;
        if (changed) {
            int i;
            double max_score = this.getHitScore(hit);
            FeatureSetI top_hit = hit;
            for (int i2 = 0; i2 < added_hits.size(); ++i2) {
                current_hit = (FeatureSetI)added_hits.elementAt(i2);
                this.debugFeature(current_hit, "Split proposed new hit #" + (i2 + 1));
                if (!(max_score < current_hit.getScore())) continue;
                max_score = current_hit.getScore();
                top_hit = current_hit;
                this.debugFeature(top_hit, "This is the current top score");
            }
            double min_score = Math.min(max_score * 0.25, 100.0);
            String msg = "Min score = " + min_score;
            if (hit.getScore() < min_score) {
                this.debugFeature(hit, "Deleted initial hit after split, because score " + hit.getScore() + " too low " + msg);
                this.debugFeature(top_hit, "Now starting over after split, with score " + top_hit.getScore());
                added_hits.removeElement(top_hit);
                analysis.deleteFeature(hit);
                analysis.addFeature(top_hit, true);
                hit = top_hit;
            }
            boolean start_over = false;
            for (i = added_hits.size() - 1; i >= 0; --i) {
                current_hit = (FeatureSetI)added_hits.elementAt(i);
                if (!(current_hit.getScore() < min_score)) continue;
                added_hits.removeElement(current_hit);
                start_over = true;
            }
            this.debugFeature(hit, " found " + added_hits.size() + " starting over = " + start_over);
            this.cleanUpTwilight(hit, filter_input);
            for (i = 0; i < added_hits.size(); ++i) {
                current_hit = (FeatureSetI)added_hits.elementAt(i);
                if (start_over) {
                    this.cleanUpTwilight(current_hit, filter_input);
                    if (current_hit == hit) {
                        this.debugFeature(hit, "Why is this in added hits list?", true);
                        continue;
                    }
                    while (current_hit.size() > 0) {
                        FeaturePairI span = (FeaturePairI)current_hit.getFeatureAt(0);
                        current_hit.deleteFeature(span);
                        hit.addFeature(span, true);
                    }
                    continue;
                }
                this.debugFeature(current_hit, "Added after split");
                analysis.addFeature(current_hit, true);
            }
            if (start_over) {
                this.debugFeature(hit, "Remerged splits after culling low scores");
            }
        }
        return changed;
    }

    private void promoteHSPsToHits(FeatureSetI analysis) {
        int i;
        Vector<FeatureSetI> new_hits = new Vector<FeatureSetI>();
        int hit_cnt = analysis.size();
        for (i = 0; i < hit_cnt; ++i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            while (hit.size() > 1) {
                FeaturePairI span = (FeaturePairI)hit.getFeatureAt(1);
                hit.deleteFeature(span);
                FeatureSetI new_hit = this.makeNewHit(hit, span);
                new_hits.addElement(new_hit);
            }
        }
        for (i = 0; i < new_hits.size(); ++i) {
            FeatureSetI new_hit = (FeatureSetI)new_hits.elementAt(i);
            analysis.addFeature(new_hit, true);
        }
    }

    private void removeWeakerSpan(FeatureSetI hit, SeqFeatureI span1, SeqFeatureI span2) {
        SeqFeatureI weaker = this.SecondIsWeaker(span1, span2) ? span2 : (this.SecondIsWeaker(span2, span1) ? span1 : span2);
        hit.deleteFeature(weaker);
        SeqFeatureI stronger = weaker == span1 ? span2 : span1;
        this.debugFeature(weaker, "Removing overlapping span ");
        this.debugFeature(stronger, "Kept this span");
    }

    public String debugName(SeqFeatureI sf) {
        String name = null;
        name = sf instanceof FeaturePairI ? ((FeaturePairI)sf).getHitSequence().getName() : (sf instanceof FeatureSetI ? ((FeatureSetI)sf).getHitSequence().getName() : sf.getRefSequence().getName());
        if (name == null) {
            System.out.println("Something seriously wrong with feature " + sf.toString());
        }
        return name;
    }

    public void debugFeature(SeqFeatureI sf, SeqFeatureI sf2, String prefix, boolean force) {
        String name = this.debugName(sf);
        if (this.filter_input.debugFilter(name) || force) {
            this.debugFeature(sf, prefix, force);
            this.debugFeature(sf2, prefix, true);
        }
    }

    public void debugFeature(SeqFeatureI sf, String prefix) {
        this.debugFeature(sf, prefix, false);
    }

    public void debugFeature(SeqFeatureI sf, String prefix, boolean force) {
        String name = this.debugName(sf);
        if (this.filter_input.debugFilter(name) || force) {
            System.out.println(name + ":  " + prefix + "\n\t" + " strand=" + sf.getStrand() + " start=" + sf.getStart() + " end=" + sf.getEnd() + "\n\tlength=" + sf.length() + " expect=" + sf.getScore("expect") + " bits=" + sf.getScore("bits") + " score=" + sf.getScore() + " type=" + sf.getFeatureType());
            if (sf instanceof FeatureSetI) {
                FeatureSetI fs = (FeatureSetI)sf;
                for (int i = 0; i < fs.size(); ++i) {
                    FeaturePairI fp = (FeaturePairI)fs.getFeatureAt(i);
                    System.out.println("\tSpan " + (i + 1) + " genomic range=" + fp.getStart() + "-" + fp.getEnd() + "\tmatch range=" + fp.getHstart() + "-" + fp.getHend());
                }
            }
        }
    }

    private boolean isSubjSequential(FeaturePairI prev_span, FeaturePairI next_span) {
        boolean is_sequential = true;
        SeqFeatureI prev_hit_span = prev_span.getHitFeature();
        SeqFeatureI next_hit_span = next_span.getHitFeature();
        boolean bl = prev_hit_span.getStrand() == 1 ? next_hit_span.getStart() > prev_hit_span.getStart() : (is_sequential = next_hit_span.getStart() < prev_hit_span.getStart());
        if (is_sequential && prev_hit_span.overlaps(next_hit_span)) {
            int spacing;
            int n = spacing = prev_hit_span.getStrand() == 1 ? next_hit_span.getStart() - prev_hit_span.getEnd() : prev_hit_span.getEnd() - next_hit_span.getStart();
            if (spacing < 0) {
                int overlap = -spacing;
                int genomic_overlap = this.getOverlap(prev_span, next_span);
                if (genomic_overlap > 0 && prev_hit_span.getRefSequence().getResidueType() == "AA") {
                    genomic_overlap /= 3;
                }
                if (genomic_overlap > 0) {
                    double ratio = overlap / genomic_overlap;
                    boolean bl2 = is_sequential = ratio > 0.8 && ratio <= 6.0;
                    if (!is_sequential) {
                        String msg = "overlap = " + overlap + " genomic_overlap = " + genomic_overlap + " ratio = " + ratio + " ";
                        this.debugFeature(prev_hit_span, msg);
                    }
                } else {
                    int hit_extent = Math.max(prev_hit_span.getHigh(), next_hit_span.getHigh()) - Math.min(prev_hit_span.getLow(), next_hit_span.getLow());
                    double percent_overlap = Math.abs(spacing) * 100 / hit_extent;
                    boolean bl3 = is_sequential = percent_overlap <= 10.0;
                    if (!is_sequential) {
                        this.debugFeature(prev_hit_span, "Spacing between this and next span is " + spacing + " genomic overlap is " + genomic_overlap + " hit extent " + hit_extent + " overlap " + percent_overlap + "%");
                    }
                }
            }
        }
        return is_sequential;
    }

    private FeatureSetI makeNewHit(FeatureSetI old_hit, FeaturePairI span) {
        FeatureSet new_hit = new FeatureSet();
        new_hit.setStrand(old_hit.getStrand());
        new_hit.setName(old_hit.getName());
        new_hit.setRefSequence(old_hit.getRefSequence());
        new_hit.setFeatureType(old_hit.getFeatureType());
        new_hit.setHitSequence(old_hit.getHitSequence());
        new_hit.setProgramName(old_hit.getProgramName());
        new_hit.setDatabase(old_hit.getDatabase());
        this.addToHit(span, new_hit);
        return new_hit;
    }

    private double getHitScore(FeatureSetI hit) {
        double max_score = -1.0;
        double hit_expect = 1.0;
        double hit_prob = 1.0;
        for (int i = 0; i < hit.size(); ++i) {
            SeqFeatureI hsp = hit.getFeatureAt(i);
            double hsp_expect = hsp.getScore("expect");
            if (hsp_expect < hit_expect) {
                hit_expect = hsp_expect;
                hit_prob = hsp.getScore("probability");
            }
            if (!(hsp.getScore() > max_score)) continue;
            max_score = hsp.getScore();
        }
        hit.setScore(max_score);
        hit.addScore("expect", hit_expect);
        hit.addScore("probability", hit_prob);
        return max_score;
    }

    private void addToHit(SeqFeatureI span, FeatureSetI hit) {
        hit.addFeature(span, true);
        double hit_expect = hit.getScore("expect");
        for (int i = 0; i < hit.size(); ++i) {
            SeqFeatureI hsp = hit.getFeatureAt(i);
            if (!(hit_expect < 0.0) && !(hsp.getScore("expect") < hit_expect)) continue;
            hit.setScore(hsp.getScore());
            hit.addScore("expect", hsp.getScore("expect"));
            hit.addScore("probability", hsp.getScore("probability"));
        }
    }

    private boolean SecondIsWeaker(SeqFeatureI span1, SeqFeatureI span2) {
        boolean span2weaker = false;
        double expect1 = span1.getScore("expect");
        double expect2 = span2.getScore("expect");
        double score1 = span1.getScore();
        double score2 = span2.getScore();
        if (expect1 < expect2) {
            span2weaker = true;
        } else if (expect1 == expect2 && score1 > score2) {
            span2weaker = true;
        } else {
            double identity1 = this.identity(span1);
            double identity2 = this.identity(span2);
            if (score1 == score2 && expect1 == expect2 && identity1 > identity2) {
                span2weaker = true;
            } else if (score1 == score2 && expect1 == expect2 && identity1 == identity2 && span1.length() > span2.length()) {
                span2weaker = true;
            }
        }
        return span2weaker;
    }

    public double identity(SeqFeatureI span) {
        return span.getScore("identity");
    }

    protected boolean lowIdentity(FeatureSetI hit, int min_identity) {
        int span_cnt = hit.size();
        for (int i = span_cnt - 1; i >= 0; --i) {
            FeaturePairI span = (FeaturePairI)hit.getFeatureAt(i);
            if (!(this.identity(span) < (double)min_identity)) continue;
            hit.deleteFeature(span);
            this.debugFeature(span, "Removed because " + this.identity(span) + "% < " + min_identity + "% identity");
        }
        return hit.size() <= 0;
    }

    private void cleanUpSpanCoincidents(FeatureSetI analysis, int coincidence) {
        int hit_cnt = analysis.size();
        for (int i = 0; i < hit_cnt; ++i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            boolean cleanup = true;
            this.debugFeature(hit, "Checking for overlapping spans ");
            while (cleanup) {
                cleanup = false;
                int span_index = 0;
                while (span_index + 1 < hit.size() && !cleanup) {
                    FeaturePairI span2;
                    FeaturePairI span1 = (FeaturePairI)hit.getFeatureAt(span_index);
                    cleanup = this.spansOverlap(span1, span2 = (FeaturePairI)hit.getFeatureAt(span_index + 1), coincidence);
                    if (cleanup) {
                        this.removeWeakerSpan(hit, span1, span2);
                        continue;
                    }
                    ++span_index;
                }
            }
        }
    }

    private void cleanupTwilights(FeatureSetI analysis, AnalysisInput filter_input) {
        for (int i = 0; i < analysis.size(); ++i) {
            FeatureSetI hit = (FeatureSetI)analysis.getFeatureAt(i);
            this.cleanUpTwilight(hit, filter_input);
        }
    }

    private void cleanUpTwilight(FeatureSetI hit, AnalysisInput filter_input) {
        TwilightFilter twilight = new TwilightFilter();
        if (hit.getRefSequence().getResidueType() == "DNA") {
            twilight.setSeqNucleic(true);
        } else {
            twilight.setSeqNucleic(false);
        }
        if (hit.getHitSequence().getResidueType() == "DNA") {
            twilight.setRefNucleic(true);
        } else {
            twilight.setRefNucleic(false);
        }
        twilight.setCertainlyGood(filter_input.getTwilightUpper());
        twilight.setCertainlyBad(filter_input.getTwilightLower());
        boolean testing = filter_input.debugFilter(hit.getHitSequence().getName());
        try {
            boolean consider_it = true;
            while (hit.size() > 1 && consider_it) {
                FeaturePairI span2;
                FeaturePairI span1 = (FeaturePairI)hit.getFeatureAt(0);
                FeaturePairI remove_span = twilight.cleanUpTwilightZone(span1, span2 = (FeaturePairI)hit.getFeatureAt(1), hit.size() == 2, testing);
                if (remove_span != null && (remove_span == span1 || remove_span == span2 && span1.getScore("bits") <= (double)twilight.getCertainlyBad())) {
                    if (hit.size() == 2) {
                        this.debugFeature(remove_span, "Removing weaker twilight span");
                        hit.deleteFeature(remove_span);
                        continue;
                    }
                    this.debugFeature(span1, "Removing starting twilight span");
                    hit.deleteFeature(span1);
                    continue;
                }
                consider_it = false;
            }
            consider_it = true;
            while (hit.size() > 1 && consider_it) {
                FeaturePairI span2;
                int last_span = hit.size() - 1;
                FeaturePairI span1 = (FeaturePairI)hit.getFeatureAt(last_span);
                FeaturePairI remove_span = twilight.cleanUpTwilightZone(span1, span2 = (FeaturePairI)hit.getFeatureAt(last_span - 1), hit.size() == 2, testing);
                if (remove_span != null) {
                    this.debugFeature(remove_span, "May remove twilight span from end, is span1 " + (remove_span == span1));
                }
                if (remove_span != null && (remove_span == span1 || remove_span == span2 && span1.getScore("bits") <= (double)twilight.getCertainlyBad())) {
                    if (hit.size() == 2) {
                        this.debugFeature(remove_span, "Removing weaker twilight span");
                        hit.deleteFeature(remove_span);
                        continue;
                    }
                    this.debugFeature(span1, "Removing terminal twilight span");
                    hit.deleteFeature(span1);
                    continue;
                }
                consider_it = false;
            }
        }
        catch (Exception e) {
            System.out.println("Error removing twilights " + e.getMessage());
            e.printStackTrace();
        }
    }

    private boolean spansOverlap(SeqFeatureI span1, SeqFeatureI span2, int coincidence) {
        boolean overlaps = false;
        double common = 0.0;
        int overlap = this.getOverlap(span1, span2);
        if (overlap > 0) {
            int extent_start = Math.min(span1.getLow(), span2.getLow());
            int extent_end = Math.max(span1.getHigh(), span2.getHigh());
            common = overlap * 100 / (extent_end - extent_start + 1);
            overlaps = common >= (double)coincidence;
            String msg = "overlap = " + overlap + " common = " + common + "%" + " length = " + (extent_end - extent_start + 1) + " ";
            this.debugFeature(span1, msg);
        }
        return overlaps;
    }

    private int getOverlap(SeqFeatureI span1, SeqFeatureI span2) {
        int overlap = 0;
        if (span1.contains(span2) || span2.contains(span1)) {
            overlap = span1.length() > span2.length() ? span1.length() : span2.length();
        } else if (span1.overlaps(span2)) {
            int connect_start = span1.getLow() < span2.getLow() ? span2.getLow() : span1.getLow();
            int connect_end = span1.getHigh() > span2.getHigh() ? span2.getHigh() : span1.getHigh();
            overlap = connect_end - connect_start;
        }
        return overlap;
    }

    private void cleanUpShadows(Vector regions, FeatureSetI forward_analysis, FeatureSetI reverse_analysis) {
        for (int i = 0; i < regions.size(); ++i) {
            Vector region = (Vector)regions.elementAt(i);
            for (int j = 0; j < region.size(); ++j) {
                FeatureSetI hit = (FeatureSetI)region.elementAt(j);
                SequenceI align_seq = hit.getHitSequence();
                int k = j + 1;
                String hit_name = align_seq.getName();
                while (k < region.size()) {
                    FeatureSetI close_hit = (FeatureSetI)region.elementAt(k);
                    SequenceI close_seq = close_hit.getHitSequence();
                    if (hit_name.equals(close_seq.getName()) && hit.getStrand() != close_hit.getStrand()) {
                        region.removeElementAt(k);
                        if (close_hit.getStrand() == 1) {
                            forward_analysis.deleteFeature(close_hit);
                        } else {
                            reverse_analysis.deleteFeature(close_hit);
                        }
                        this.debugFeature(hit, "Removed because it is a shadow ");
                        continue;
                    }
                    ++k;
                }
            }
        }
    }

    private FeatureSetI getAnalysis(CurationSet curation, String analysis_type, int strand, String type) {
        FeatureSetI the_one = null;
        FeatureSetI analyses = strand == 1 ? curation.getResults().getForwardSet() : curation.getResults().getReverseSet();
        for (int i = 0; i < analyses.size() && the_one == null; ++i) {
            FeatureSetI analysis = (FeatureSetI)analyses.getFeatureAt(i);
            String sf_type = analysis.getFeatureType();
            the_one = sf_type != null && sf_type.equals(analysis_type) ? analysis : null;
        }
        return the_one;
    }

    private void joinESTs(FeatureSetI forward_analysis, FeatureSetI reverse_analysis) {
        FeatureSetI hit;
        int i = 0;
        Vector completed = new Vector();
        while (i < forward_analysis.size()) {
            hit = (FeatureSetI)forward_analysis.getFeatureAt(i);
            if (!completed.contains(hit)) {
                i = this.joinEST(hit, forward_analysis, reverse_analysis, completed);
                continue;
            }
            this.debugFeature(hit, "joinESTs: have already dealt with " + hit.getName() + " on forward strand");
            ++i;
        }
        i = 0;
        while (i < reverse_analysis.size()) {
            hit = (FeatureSetI)reverse_analysis.getFeatureAt(i);
            if (!completed.contains(hit)) {
                i = this.joinEST(hit, forward_analysis, reverse_analysis, completed);
                continue;
            }
            this.debugFeature(hit, "joinESTs: have already dealt with " + hit.getName() + " on reverse strand");
            ++i;
        }
    }

    private boolean isFullLength(String name) {
        return name.indexOf("complete") > 0 || name.indexOf("contig") > 0 && name.indexOf("prime") < 0;
    }

    private boolean is5prime(String name) {
        return name.indexOf("5prime") > 0;
    }

    private boolean is3prime(String name) {
        if (this.suffix_3prime != null) {
            return name.indexOf(this.suffix_3prime) > 0;
        }
        return name.indexOf(default_3prime) > 0;
    }

    private FeatureSetI get5primeMostHit(FeatureSetI hit1, FeatureSetI hit2) {
        return hit1.getStrand() != -1 && hit2.getStart() < hit1.getStart() || hit1.getStrand() == -1 && hit2.getStart() > hit1.getStart() ? hit2 : hit1;
    }

    private FeatureSetI get3primeMostHit(FeatureSetI hit1, FeatureSetI hit2) {
        return hit1.isForwardStrand() && hit2.getEnd() > hit1.getEnd() || !hit1.isForwardStrand() && hit2.getEnd() < hit1.getEnd() ? hit2 : hit1;
    }

    private boolean estIsInternal(FeatureSetI leading_hit, FeatureSetI trailing_hit, FeatureSetI est_hit) {
        return this.estFollows(leading_hit, est_hit) && this.estPrecedes3(trailing_hit, est_hit);
    }

    private boolean estFollows(SeqFeatureI leading, SeqFeatureI check) {
        boolean follows;
        boolean bl = follows = leading.isForwardStrand() && check.getStart() >= leading.getStart() || !leading.isForwardStrand() && check.getStart() <= leading.getStart();
        if (!follows) {
            this.debugFeature(check, "estFollows: this (?) should come after " + ((FeatureSetI)leading).getHitSequence().getName());
        }
        return follows;
    }

    private boolean estPrecedes3(FeatureSetI trailing_est, FeatureSetI est) {
        boolean trails;
        boolean bl = trailing_est != null ? this.get3primeMostHit(trailing_est, est) == trailing_est : (trails = true);
        if (!trails) {
            this.debugFeature(est, "estPrecedes3: this (?) should come before " + trailing_est.getHitSequence().getName());
        }
        return trails;
    }

    private int joinEST(FeatureSetI hit, FeatureSetI forward_analysis, FeatureSetI reverse_analysis, Vector completed) {
        FeatureSetI analysis;
        int hit_index = forward_analysis.getFeatureIndex(hit);
        if (hit_index < 0) {
            hit_index = reverse_analysis.getFeatureIndex(hit);
            analysis = reverse_analysis;
        } else {
            analysis = forward_analysis;
        }
        String name = hit.getHitSequence().getName();
        String clone = this.getPrefix(name);
        Vector est_list = new Vector();
        est_list.addElement(hit);
        completed.addElement(hit);
        this.debugFeature(hit, "joinEST: joining ends now ");
        this.collectESTs(clone, forward_analysis, est_list, completed);
        this.collectESTs(clone, reverse_analysis, est_list, completed);
        FeatureSetI leading_est = this.getLeader(est_list, false);
        FeatureSetI trailing_est = this.getTrailer(est_list, leading_est, false);
        if (est_list.size() > 1) {
            Vector<Vector> est_sets = new Vector<Vector>();
            est_sets.addElement(est_list);
            if (leading_est != null && trailing_est != null) {
                boolean chimeric = this.checkForChimericClones(est_list, est_sets, leading_est, trailing_est);
                if (!chimeric) {
                    boolean inverted;
                    if (leading_est != trailing_est && leading_est.getStrand() != trailing_est.getStrand()) {
                        this.pickStrand(leading_est, trailing_est);
                    }
                    if (!(inverted = this.checkForInversions(est_list, leading_est, trailing_est))) {
                        this.forceToStrand(est_list, leading_est.getStrand());
                    } else {
                        this.forceToStrand(est_list, trailing_est.getStrand());
                    }
                }
            } else {
                this.checkForMissingEnds(est_list, leading_est, trailing_est);
            }
            this.checkForMisidentity(est_sets);
            for (int i = 0; i < est_sets.size(); ++i) {
                est_list = (Vector)est_sets.elementAt(i);
                SeqFeatureUtil.sort(est_list, ((SeqFeatureI)est_list.elementAt(0)).getStrand());
                leading_est = this.getLeader(est_list, true);
                trailing_est = this.getTrailer(est_list, leading_est, true);
                String lead_name = leading_est.getHitSequence().getName();
                est_list.remove(leading_est);
                if (est_list.size() <= 0) continue;
                Hashtable span_seqs = new Hashtable();
                String name_qualifier = est_sets.size() > 1 ? ":set" + (i + 1) : "";
                for (int j = 0; j < leading_est.size(); ++j) {
                    FeaturePairI span = (FeaturePairI)leading_est.getFeatureAt(j);
                    Vector<String> seqs = new Vector<String>();
                    seqs.addElement(span.getHitSequence().getName() + " " + span.getHitSequence().getLength() + " bp");
                    span_seqs.put(span, seqs);
                }
                String seq_id = this.getPrefix(lead_name);
                this.debugFeature(trailing_est, "Synthesizing EST hit to " + seq_id);
                Sequence clone_seq = new Sequence(seq_id, "");
                clone_seq.setAccessionNo(seq_id);
                StringBuffer description = new StringBuffer();
                int internal_count = this.createDescription(leading_est, description, true, 0);
                leading_est.setHitSequence(clone_seq);
                for (int j = 0; j < est_list.size(); ++j) {
                    FeatureSetI est = (FeatureSetI)est_list.elementAt(j);
                    internal_count = this.createDescription(est, description, false, internal_count);
                    if (est.getScore() > leading_est.getScore()) {
                        leading_est.setScore(est.getScore());
                    }
                    if (forward_analysis.getFeatureIndex(est) >= 0) {
                        forward_analysis.deleteFeature(est);
                    } else {
                        reverse_analysis.deleteFeature(est);
                    }
                    this.debugFeature(est, "Adding EST to cDNA group ");
                    while (est.size() > 0) {
                        FeaturePairI span = (FeaturePairI)est.getFeatureAt(0);
                        est.deleteFeature(span);
                        this.debugFeature(span, "Adding EST span to cDNA group ");
                        this.insertSpan(leading_est, span, span_seqs);
                    }
                }
                if (internal_count > 0) {
                    if (description.length() > 0) {
                        description.append(", ");
                    }
                    description.append(internal_count + " internal reads");
                }
                clone_seq.setDescription(description.toString());
                this.extendAlignmentSeq(leading_est);
                this.setSpanSequences(leading_est, span_seqs, name_qualifier);
            }
        } else {
            this.debugFeature(hit, "No other ESTS for " + name);
        }
        if (analysis.getFeatureIndex(hit) < 0) {
            return hit_index;
        }
        return hit_index + 1;
    }

    private boolean checkForChimericClones(Vector est_list, Vector est_sets, FeatureSetI leading_est, FeatureSetI trailing_est) {
        if (leading_est != null && trailing_est != null && this.gapBetween(leading_est, trailing_est) >= CHIMERA_GAP) {
            this.debugFeature(leading_est, trailing_est, "Have a gap of " + this.gapBetween(leading_est, trailing_est) + "bp ", true);
            Vector<FeatureSetI> new_list = new Vector<FeatureSetI>();
            est_list.removeElement(trailing_est);
            this.addQualifier("chimeric", trailing_est);
            new_list.addElement(trailing_est);
            for (int i = 0; i < est_list.size(); ++i) {
                FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
                this.addQualifier("chimeric", est);
                if (this.gapBetween(est, leading_est) <= this.gapBetween(est, trailing_est)) continue;
                est_list.removeElement(est);
                new_list.addElement(est);
            }
            est_sets.addElement(new_list);
        }
        return est_sets.size() > 1;
    }

    private boolean checkForInversions(Vector est_list, FeatureSetI leading_est, FeatureSetI trailing_est) {
        boolean inverted = false;
        if (leading_est != null && trailing_est != null && !this.estPrecedes3(trailing_est, leading_est)) {
            this.addQualifier("inverted", leading_est);
            this.addQualifier("inverted", trailing_est);
            for (int i = 0; i < est_list.size(); ++i) {
                FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
                this.addQualifier("inverted", est);
            }
            inverted = true;
            this.debugFeature(leading_est, "is inverted");
        }
        return inverted;
    }

    private void checkForMissingEnds(Vector est_list, FeatureSetI leading_est, FeatureSetI trailing_est) {
        String hit_name;
        FeatureSetI first_hit;
        if (leading_est == null && trailing_est == null) {
            this.cloneIsUnknown(est_list);
            first_hit = (FeatureSetI)est_list.elementAt(0);
            hit_name = first_hit.getHitSequence().getName();
        } else {
            first_hit = leading_est != null ? leading_est : trailing_est;
            hit_name = first_hit.getHitSequence().getName();
            if (first_hit.getProperty("revcomp").equals("true")) {
                this.reverseEST(first_hit, true);
            }
        }
        int strand = this.pickStrand(est_list);
        this.debugFeature(first_hit, "Forcing to est " + hit_name + " on strand " + strand);
        this.forceToStrand(est_list, strand);
    }

    private void checkForMisidentity(Vector est_sets) {
        for (int i = est_sets.size() - 1; i >= 0; --i) {
            Vector est_list = (Vector)est_sets.elementAt(i);
            FeatureSetI leading_est = this.getLeader(est_list, true);
            FeatureSetI trailing_est = this.getTrailer(est_list, leading_est, true);
            String lead_name = leading_est.getHitSequence().getName();
            int attempts = 0;
            boolean done = false;
            while (attempts < 2 && !done) {
                ++attempts;
                done = true;
                for (int j = est_list.size() - 1; j >= 0 && done; --j) {
                    FeatureSetI est = (FeatureSetI)est_list.elementAt(j);
                    String est_name = est.getHitSequence().getName();
                    boolean rev_comped = est.getProperty("revcomp").equals("true");
                    if (this.estIsInternal(leading_est, trailing_est, est)) continue;
                    this.debugFeature(est, est_name + " is not internal to " + lead_name);
                    boolean extended = true;
                    if (!this.estFollows(leading_est, est)) {
                        this.debugFeature(est, est_name + " does not follow 5' of " + lead_name);
                        if (!(extended &= this.estOverlaps(leading_est, est, 0))) {
                            this.debugFeature(est, est_name + " does not overlap " + lead_name);
                        }
                    }
                    if (!this.estPrecedes3(trailing_est, est)) {
                        extended &= this.estOverlaps(trailing_est, est, trailing_est.size() - 1);
                    }
                    if (extended) continue;
                    if (est_name.indexOf("prime") > 0 && !this.isFullLength(lead_name)) {
                        if (rev_comped) {
                            this.reverseEST(est, true);
                        }
                        int strand = this.pickStrand(est_list);
                        this.forceToStrand(est_list, strand);
                        done = false;
                        continue;
                    }
                    this.addUnknownClone(est_sets, est_list, est, leading_est);
                }
            }
            if (attempts < 2) continue;
            try {
                throw new BopException("Not all EST reads from " + lead_name + " at " + leading_est.getStart() + "-" + leading_est.getEnd() + " are within 5' and 3' reads ");
            }
            catch (Exception e) {
                System.out.println(e.getMessage());
                System.out.println("REMOVING " + lead_name);
                est_sets.removeElement(est_list);
            }
        }
    }

    private boolean isQualified(String qualifier, FeatureSetI est) {
        return est.getProperty(qualifier).equals("true");
    }

    private void addQualifier(String qualifier, FeatureSetI est) {
        if (this.isQualified(qualifier, est)) {
            est.removeProperty(qualifier);
        } else {
            est.addProperty(qualifier, "true");
        }
    }

    private void cloneIsUnknown(Vector est_list) {
        if (est_list.size() > 1) {
            for (int i = 0; i < est_list.size(); ++i) {
                FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
                this.addQualifier("unknown", est);
            }
        }
    }

    private int gapBetween(SeqFeatureI est1, SeqFeatureI est2) {
        if (est1.overlaps(est2)) {
            return 0;
        }
        if (est1.getLow() < est2.getLow()) {
            return est2.getLow() - est1.getHigh();
        }
        return est1.getLow() - est2.getHigh();
    }

    private boolean estOverlaps(FeatureSetI hit1, FeatureSetI hit2, int terminus) {
        boolean overlaps = false;
        FeaturePairI span1 = (FeaturePairI)hit1.getFeatureAt(terminus);
        for (int i = 0; i < hit2.size() && !overlaps; ++i) {
            FeaturePairI span2 = (FeaturePairI)hit2.getFeatureAt(i);
            overlaps = span1.overlaps(span2);
        }
        return overlaps;
    }

    private void insertSpan(FeatureSetI hit, FeaturePairI new_span, Hashtable span_seqs) {
        Vector seqs;
        int i = 0;
        boolean inserted = false;
        while (i < hit.size() && !inserted) {
            FeaturePairI hit_span = (FeaturePairI)hit.getFeatureAt(i);
            if (hit_span.overlaps(new_span)) {
                this.mergeSpans(hit_span, new_span);
                hit.adjustEdges(hit_span);
                seqs = (Vector)span_seqs.get(hit_span);
                seqs.addElement(new_span.getHitSequence().getName() + " " + new_span.getHitSequence().getLength() + " bp");
                this.debugFeature(hit_span, "Merged spans " + hit_span.getHitSequence().getName() + " and " + new_span.getHitSequence().getName());
                inserted = true;
                continue;
            }
            ++i;
        }
        if (!inserted) {
            seqs = new Vector();
            seqs.addElement(new_span.getHitSequence().getName() + " " + new_span.getHitSequence().getLength() + " bp");
            span_seqs.put(new_span, seqs);
            if (!this.estFollows(hit, new_span)) {
                System.out.println("How did insert of " + new_span.getHitSequence().getName() + " at " + new_span.getStart() + " get before " + hit.getHitSequence().getName() + " at " + hit.getEnd());
            }
            hit.addFeature(new_span, true);
        }
    }

    private void setSpanSequences(FeatureSetI leading_hit, Hashtable span_seqs, String name_qualifier) {
        String seq_id = leading_hit.getHitSequence().getName();
        Vector prev_seqs = null;
        SequenceI prev_seq = null;
        int contig = 0;
        for (int i = 0; i < leading_hit.size(); ++i) {
            FeaturePairI span = (FeaturePairI)leading_hit.getFeatureAt(i);
            Vector seqs = (Vector)span_seqs.get(span);
            span.addProperty("READS", this.appendToDescription("", seqs));
            SequenceI seq = null;
            if (prev_seqs != null) {
                for (int j = 0; j < prev_seqs.size(); ++j) {
                    String prev_name = (String)prev_seqs.elementAt(j);
                    for (int k = seqs.size() - 1; k >= 0; --k) {
                        String this_name = (String)seqs.elementAt(k);
                        if (!prev_name.equals(this_name)) continue;
                        seq = prev_seq;
                        seqs.removeElementAt(k);
                    }
                }
            }
            String description = "";
            if (seq == null) {
                String span_name = seq_id + name_qualifier + ":contig" + ++contig;
                seq = new Sequence(seq_id, "");
                seq.setAccessionNo(span_name);
                prev_seqs = seqs;
                this.debugFeature(span, "Set span " + (i + 1) + " to " + span_name);
            } else {
                description = seq.getDescription();
                for (int j = 0; j < seqs.size(); ++j) {
                    String this_name = (String)seqs.elementAt(j);
                    prev_seqs.addElement(this_name);
                }
            }
            description = this.appendToDescription(description, seqs);
            seq.setDescription(description);
            span.setHitSequence(seq);
            prev_seq = seq;
        }
    }

    private String appendToDescription(String prefix, Vector seqs) {
        String description = prefix;
        for (int j = 0; j < seqs.size(); ++j) {
            String this_name = (String)seqs.elementAt(j);
            int suffix_index = this_name.indexOf(".");
            String suffix = suffix_index >= 0 ? this_name.substring(suffix_index + 1) : this_name;
            if (description.length() > 0) {
                description = description + ", ";
            }
            description = description + suffix;
        }
        return description;
    }

    private void forceToStrand(Vector est_list, int strand) {
        for (int i = 0; i < est_list.size(); ++i) {
            FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
            if (est.getStrand() == strand) continue;
            this.reverseEST(est, true);
        }
    }

    private int pickStrand(Vector est_list) {
        int strand = 0;
        for (int i = 0; i < est_list.size() && strand == 0; ++i) {
            FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
            String sim4_set = est.getProperty("sim4_set");
            if (sim4_set == null || !sim4_set.equals("true")) continue;
            strand = est.getStrand();
        }
        if (strand == 0) {
            FeatureSetI est = (FeatureSetI)est_list.elementAt(0);
            strand = est.getStrand();
        }
        return strand;
    }

    private void pickStrand(FeatureSetI est1, FeatureSetI est2) {
        if (est1.getStrand() != est2.getStrand()) {
            boolean est2_set;
            String sim4_set = est1.getProperty("sim4_set");
            boolean est1_set = sim4_set != null && sim4_set.equals("true");
            sim4_set = est2.getProperty("sim4_set");
            boolean bl = est2_set = sim4_set != null && sim4_set.equals("true");
            if (est2_set && !est1_set) {
                this.reverseEST(est1, true);
            } else {
                this.reverseEST(est2, true);
            }
        }
    }

    private void reverseESTs(FeatureSetI hits, Vector completed) {
        for (int i = hits.size() - 1; i >= 0; --i) {
            FeatureSetI hit = (FeatureSetI)hits.getFeatureAt(i);
            String name = hit.getHitSequence().getName();
            if (!this.is3prime(name) || completed.contains(hit)) continue;
            this.debugFeature(hit, "Checking for reversal");
            this.reverseEST(hit, false);
            completed.addElement(hit);
        }
    }

    private void reverseEST(FeatureSetI hit, boolean force) {
        SequenceI seq = hit.getHitSequence();
        String seq_name = seq.getName();
        String sim4_set = hit.getProperty("sim4_set");
        if (sim4_set != null && sim4_set.equals("true") && !force) {
            this.debugFeature(hit, "Tried reversing: " + seq_name + ", but it is set");
            return;
        }
        this.debugFeature(hit, "Reversing: " + seq_name);
        hit.flipFlop();
        SequenceI aligned_seq = hit.getHitSequence();
        int seq_length = aligned_seq.getLength();
        for (int i = 0; i < hit.size(); ++i) {
            FeaturePairI span = (FeaturePairI)hit.getFeatureAt(i);
            SeqFeatureI hit_span = span.getHitFeature();
            int pos1 = seq_length - hit_span.getStart() + 1;
            int pos2 = seq_length - hit_span.getEnd() + 1;
            hit_span.setStrand(pos2 <= pos1 ? 1 : -1);
            hit_span.setStart(pos2);
            hit_span.setEnd(pos1);
            String coord_seq = span.getExplicitAlignment();
            String align_seq = hit_span.getExplicitAlignment();
            if (coord_seq == null || coord_seq.equals("") || align_seq == null || align_seq.equals("")) continue;
            coord_seq = DNAUtils.reverseComplement((String)coord_seq);
            align_seq = DNAUtils.reverseComplement((String)align_seq);
            span.setExplicitAlignment(coord_seq);
            hit_span.setExplicitAlignment(align_seq);
            span.setCigar(CigarUtil.roll(coord_seq, align_seq, 1));
        }
        if (hit.getProperty("revcomp").equals("true")) {
            hit.removeProperty("revcomp");
        } else {
            hit.addProperty("revcomp", "true");
        }
        String rev = DNAUtils.reverseComplement((String)seq.getResidues());
        seq.setResidues(rev);
        FeatureSetI analysis = (FeatureSetI)hit.getRefFeature();
        if (analysis.getStrand() != hit.getStrand()) {
            if (analysis == this.forward_analysis) {
                this.forward_analysis.deleteFeature(hit);
                this.reverse_analysis.addFeature(hit, true);
                this.debugFeature(hit, "Moved: " + seq_name + " to minus strand");
            } else if (analysis == this.reverse_analysis) {
                this.reverse_analysis.deleteFeature(hit);
                this.forward_analysis.addFeature(hit, true);
                this.debugFeature(hit, "Moved: " + seq_name + " to plus strand");
            } else {
                System.out.println("What is this analysis " + analysis.getName() + " a " + analysis.getClass().getName() + " for " + hit.getHitSequence().getName());
            }
        } else {
            System.out.println("Why are strands the same for " + analysis.getName() + " a " + analysis.getClass().getName() + " and " + hit.getHitSequence().getName());
        }
    }

    private void extendAlignmentSeq(FeatureSetI hit) {
        FeaturePairI span = (FeaturePairI)hit.getFeatureAt(0);
        if (span == null) {
            System.out.println("extendAlignmentSeq: hit has no span " + hit.getHitSequence().getName());
        }
        int offset = span.getHstart() - 1;
        for (int i = 0; i < hit.size(); ++i) {
            span = (FeaturePairI)hit.getFeatureAt(i);
            int length = span.length();
            span.setHstart(offset + 1);
            span.setHend(offset + length);
            this.debugFeature(span, " from offset " + (offset += length) + " set span to " + span.getHstart() + "-" + span.getHend());
        }
        this.debugFeature(hit, " final total length is " + offset);
        hit.getHitSequence().setLength(offset);
    }

    private String getPrefix(String name) {
        String clone = name.indexOf(".") < 0 ? name : name.substring(0, name.indexOf("."));
        return clone;
    }

    private void mergeSpans(FeaturePairI into, FeaturePairI from) {
        if (into.getStrand() != from.getStrand()) {
            System.err.println("Error: spans do not agree on direction.\n " + into.getStart() + "," + into.getEnd() + " - " + from.getStart() + "," + from.getEnd());
            return;
        }
        if (from.getHigh() > into.getHigh()) {
            into.setHigh(from.getHigh());
        }
        if (from.getLow() < into.getLow()) {
            into.setLow(from.getLow());
        }
    }

    private void addUnknownClone(Vector est_sets, Vector est_list, FeatureSetI est_hit, FeatureSetI leading_hit) {
        System.out.println(est_hit.getHitSequence().getName() + " at " + est_hit.getStart() + "-" + est_hit.getEnd() + " is somewhere different than the other ESTs");
        this.addQualifier("unknown", est_hit);
        est_list.remove(est_hit);
        boolean placed = false;
        for (int i = 0; i < est_sets.size() && !placed; ++i) {
            FeatureSetI check_est;
            Vector check_hits = (Vector)est_sets.elementAt(i);
            if (check_hits == est_list || check_hits.size() <= 0 || this.gapBetween(check_est = (FeatureSetI)check_hits.elementAt(0), est_hit) >= CHIMERA_GAP) continue;
            placed = true;
            check_hits.addElement(est_hit);
        }
        if (!placed) {
            Vector<FeatureSetI> new_list = new Vector<FeatureSetI>();
            new_list.addElement(est_hit);
            est_sets.addElement(new_list);
        }
    }

    private int createDescription(FeatureSetI est, StringBuffer description, boolean add_bp, int internal_count) {
        String est_name = est.getHitSequence().getName();
        if (add_bp) {
            if (description.length() > 0) {
                description.append(", ");
            }
            description.append(est_name + " " + est.getHitSequence().getLength() + "bp");
        } else {
            ++internal_count;
        }
        return internal_count;
    }

    private void collectESTs(String clone, FeatureSetI analysis, Vector est_list, Vector completed) {
        for (int i = 0; i < analysis.size(); ++i) {
            FeatureSetI check_hit = (FeatureSetI)analysis.getFeatureAt(i);
            String check_name = check_hit.getHitSequence().getName();
            String check_clone = this.getPrefix(check_name);
            if (!clone.equals(check_clone) || completed.contains(check_hit)) continue;
            est_list.addElement(check_hit);
            completed.addElement(check_hit);
        }
    }

    private FeatureSetI getLeader(Vector est_list, boolean force) {
        FeatureSetI leading_est = null;
        FeatureSetI best_begin = est_list.size() > 0 ? (FeatureSetI)est_list.elementAt(0) : null;
        for (int i = 0; i < est_list.size(); ++i) {
            FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
            String est_name = est.getHitSequence().getName();
            best_begin = this.get5primeMostHit(best_begin, est);
            if (!this.isFullLength(est_name) && !this.is5prime(est_name) && (!this.is3prime(est_name) || !this.isQualified("inverted", est))) continue;
            leading_est = leading_est != null ? this.get5primeMostHit(leading_est, est) : est;
        }
        if (force && leading_est == null) {
            leading_est = best_begin;
            this.debugFeature(leading_est, "Leading 5' EST is forced to be " + best_begin.getHitSequence().getName());
        }
        return leading_est;
    }

    private FeatureSetI getTrailer(Vector est_list, FeatureSetI leading_est, boolean force) {
        FeatureSetI trailing_est = null;
        FeatureSetI best_end = est_list.size() > 0 ? (FeatureSetI)est_list.elementAt(0) : null;
        for (int i = 0; i < est_list.size(); ++i) {
            FeatureSetI est = (FeatureSetI)est_list.elementAt(i);
            best_end = this.get3primeMostHit(best_end, est);
            String est_name = est.getHitSequence().getName();
            if (!this.isFullLength(est_name) && !this.is3prime(est_name) && (!this.is5prime(est_name) || !this.isQualified("inverted", est))) continue;
            trailing_est = trailing_est != null ? this.get3primeMostHit(trailing_est, est) : est;
        }
        if (force) {
            boolean changed = false;
            if (trailing_est == null) {
                trailing_est = best_end;
                this.debugFeature(trailing_est, "Trailing 3' EST is forced to be " + best_end.getHitSequence().getName());
                changed = true;
            }
            if (trailing_est == leading_est && !this.isFullLength(trailing_est.getHitSequence().getName())) {
                int index = est_list.size() - 1 - est_list.indexOf(leading_est);
                trailing_est = (FeatureSetI)est_list.elementAt(index);
                this.debugFeature(trailing_est, "Trailing 3' EST is forced to end opposite " + leading_est.getHitSequence().getName());
                changed = true;
            }
            if (!changed) {
                this.debugFeature(trailing_est, "Trailing 3' EST is " + trailing_est.getHitSequence().getName());
            }
        }
        return trailing_est;
    }
}

