/*  This file is part of "xprintmon"
 *  Copyright (C) 2006 Bernhard R. Link
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02111-1301  USA
 */
#include <config.h>

#include <assert.h>
#include <stdio.h>
#include <stdbool.h>
#include <stdlib.h>
#include <stdarg.h>
#include <ctype.h>
#include <alloca.h>

#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Cardinals.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Viewport.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Box.h>
#include "widgets/VBox.h"

#include "global.h"

static Pixmap check_bm, uncheck_bm;

static const unsigned char check_bm_bits[16*16/8] = {
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x20,
	0x00, 0x30,
	0x00, 0x18,
	0x00, 0x0c,
	0x0c, 0x06,
	0x18, 0x03,
	0xb0, 0x01,
	0xe0, 0x00,
	0x40, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00
};
static const unsigned char uncheck_bm_bits[16*16/8] = {
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00,
	0x00, 0x00
};

void initPrinterWindowData(Widget app_shell) {
	check_bm = XCreateBitmapFromData(XtDisplay(app_shell),
			DefaultRootWindow(XtDisplay(app_shell)),
			(char*)check_bm_bits, 16, 16);
	uncheck_bm = XCreateBitmapFromData(XtDisplay(app_shell),
			DefaultRootWindow(XtDisplay(app_shell)),
			(char*)uncheck_bm_bits, 16, 16);
	if( uncheck_bm == None || check_bm == None ) {
		fprintf(stderr, "Error creating bitmap!\n");
		exit(EXIT_FAILURE);
	}
}
struct printer_window {
	struct printer_window *next;
	char *printer;
	struct queue_part {
		struct queue_part *next;
		struct printer_window *parent;
		size_t len;
		char *name;
		Widget form,
			checkbox_queue,
			checkbox_print,
			label_dest,
			label_state,
			label_error,
			label_jobs;
		struct job {
			struct queue_part *parent;
			struct job *next;
			char *id;
			Widget label, button;
		} *jobs;
		char *text_dest, *text_state, *text_error;
		Boolean stopped, disabled;
	} *parts;
	Widget popup,
	       	outer_form,
			viewport,
				inner_form,
			/*buttons,*/
				closeBtn,
				rescanBtn,
				rawBtn;
	struct child *action;
	size_t parsed;
	enum parsemode {pm_START=0, pm_HEADER, pm_JOBS, pm_IGNORE} parsemode;
	struct queue_part *parsepart;
	bool hasActive, iconActive;
	XtIntervalId rescanTimer;
	struct printerwindow_resourcedata {
		int rescanTimeOut;
	} resources;
} *printer_window_root;

static XtResource printerwindow_resources[] = {
	{ "rescan", "TimeOut", XtRInt, sizeof(int),
	  XtOffsetOf(struct printerwindow_resourcedata, rescanTimeOut), XtRImmediate,
	  (XtPointer)3000 }
};

static void free_printer_window(struct printer_window *w) {
	struct queue_part *q;
	struct job *j;

	free(w->printer);
	while( (q = w->parts) != NULL ) {
		w->parts = q->next;
		free(q->name);
		free(q->text_dest);
		free(q->text_state);
		free(q->text_error);
		while( (j = q->jobs) != NULL ) {
			q->jobs = j->next;
			free(j->id);
			free(j);
		}
		free(q);
	}
	free(w);
}

static void showActiveIcon(struct printer_window *w) {
	w->hasActive = true;
	if( w->iconActive )
		return;
	XtVaSetValues(w->popup,
			XtNiconPixmap,  bm_activeprinter,
			XtNiconMask,    bm_activeprintermask,
			NULL);
	w->iconActive = true;
}

static void showInactiveIcon(struct printer_window *w) {
	if( !w->iconActive )
		return;
	XtVaSetValues(w->popup,
			XtNiconPixmap,  bm_printer,
			XtNiconMask,    bm_printermask,
			NULL);
	w->iconActive = false;
}
static void lpql_rescan(Widget w UNUSED, XtPointer privdata, XtPointer cad UNUSED);

static void printCheckbox(Widget w UNUSED, XtPointer privdata, XtPointer calldata UNUSED) {
	struct queue_part *data = privdata;

	notified_logged_action(context, data->parent->popup,
			lpql_rescan, data->parent,
			"/usr/sbin/lpc",
			data->stopped?"start":"stop",
			data->name,
			NULL);
}
static void queueCheckbox(Widget w UNUSED, XtPointer privdata, XtPointer calldata UNUSED) {
	struct queue_part *data = privdata;

	notified_logged_action(context, data->parent->popup,
			lpql_rescan, data->parent,
			"/usr/sbin/lpc",
			data->disabled?"enable":"disable",
			data->name,
			NULL);
}

static struct queue_part *queue_part(struct printer_window *w, const char *part, size_t len) {
	struct queue_part *n, *p = w->parts;
	XtPointer insertname, insertafter;
	Widget wid, queue_label;

	while( p != NULL && (p->len != len || memcmp(p->name, part, len) != 0) )
		p = p->next;
	if( p != NULL )
		return p;

	n = calloc(1, sizeof(struct queue_part));
	if( n == NULL )
		return NULL;
	n->parent = w;
	n->name = strndup(part, len);
	n->len = len;
	if( n->name == NULL ) {
		free(n);
		return NULL;
	}
	p = w->parts;
	while( p != NULL && p->next != NULL )
		p = p->next;
	if( p == NULL ) {
		insertname = NULL;
		insertafter = NULL;
	} else {
		insertafter = p->form;
		insertname = XtNfromVert;
	}
//	fprintf(stderr, "Creating new queue part '%s' for '%s'\n", n->name, w->printer);
	n->form = XtVaCreateManagedWidget(n->name,
			formWidgetClass, w->inner_form,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainRight,
			/* must be last */
			insertname,	insertafter,
			NULL);
//	fprintf(stderr, "form is %p\n", n->form);
	queue_label = XtVaCreateManagedWidget("queue_label",
			labelWidgetClass, n->form,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	wid = XtVaCreateManagedWidget(n->name,
			labelWidgetClass, n->form,
			XtNfromHoriz,		(XtPointer)queue_label,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->checkbox_queue = XtVaCreateManagedWidget("enabled",
			commandWidgetClass, n->form,
			XtNfromHoriz,		(XtPointer)wid,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNbitmap,		uncheck_bm,
			XtNsensitive,		(XtPointer)False,
			XtNinternalHeight,	(XtPointer)0,
			XtNinternalWidth,	(XtPointer)0,
			NULL);
	XtAddCallback(n->checkbox_queue, XtNcallback, queueCheckbox, n);
	wid = XtVaCreateManagedWidget("enabled_label",
			labelWidgetClass, n->form,
			XtNfromHoriz,		(XtPointer)n->checkbox_queue,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->checkbox_print = XtVaCreateManagedWidget("started",
			commandWidgetClass, n->form,
			XtNfromHoriz,		(XtPointer)wid,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNbitmap,		uncheck_bm,
			XtNsensitive,		(XtPointer)False,
			XtNinternalHeight,	(XtPointer)0,
			XtNinternalWidth,	(XtPointer)0,
			NULL);
	XtAddCallback(n->checkbox_print, XtNcallback, printCheckbox, n);
	XtVaCreateManagedWidget("started_label",
			labelWidgetClass, n->form,
			XtNfromHoriz,		(XtPointer)n->checkbox_print,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->label_state = XtVaCreateManagedWidget("state",
			labelWidgetClass, n->form,
			XtNfromVert,		(XtPointer)queue_label,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			XtNresize,		(XtPointer)True,
			XtNheight,		(XtPointer)0,
			NULL);
	n->label_error = XtVaCreateManagedWidget("error",
			labelWidgetClass, n->form,
			XtNfromVert,		(XtPointer)queue_label,
			XtNfromHoriz,		(XtPointer)n->label_state,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			XtNresize,		(XtPointer)True,
			XtNheight,		(XtPointer)0,
			NULL);
	wid = XtVaCreateManagedWidget("dest_label",
			labelWidgetClass, n->form,
//			XtNfromVert,		(XtPointer)queue_label,
			XtNfromVert,		(XtPointer)n->label_state,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->label_dest = XtVaCreateManagedWidget("dest",
			labelWidgetClass, n->form,
//			XtNfromVert,		(XtPointer)queue_label,
			XtNfromVert,		(XtPointer)n->label_error,
			XtNfromHoriz,		(XtPointer)wid,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->next = NULL;
	if( p == NULL )
		w->parts = n;
	else
		p->next = n;
	return n;
}

static void preparejobs(struct queue_part *q, const char *title, size_t len) {
	if( q->label_jobs == None ) {
		char *text = alloca(len+1);
		memcpy(text,title,len);
		text[len] = '\0';
		q->label_jobs = XtVaCreateManagedWidget("jobs_label",
				labelWidgetClass, q->form,
				XtNlabel,		text,
				XtNfromVert,		(XtPointer)q->label_dest,
				XtNtop,			(XtPointer)XawChainTop,
				XtNbottom,		(XtPointer)XawChainTop,
				XtNleft,		(XtPointer)XawChainLeft,
				XtNright,		(XtPointer)XawChainLeft,
				XtNresizable,		(XtPointer)True,
				NULL);
	} else {
		// update title
	}
	while( q->jobs != NULL ) {
		struct job *j = q->jobs;
		q->jobs = j->next;
		XtDestroyWidget(j->label);
		XtDestroyWidget(j->button);
		free(j->id);
		free(j);
	}
}

/* to be called at the end of each part */
static void parsequeuepartend(struct printer_window *w) {
	struct queue_part *q;
	if( w->parsemode != pm_HEADER )
		return;
	q = w->parsepart;
	if( q == NULL )
		return;
	/* there was a started queue-part without header line, that means there
	 * are no jobs at all. But there might have been before, so tidy up: */
	while( q->jobs != NULL ) {
		struct job *j = q->jobs;
		q->jobs = j->next;
		XtDestroyWidget(j->label);
		XtDestroyWidget(j->button);
		free(j->id);
		free(j);
	}
	if( q->label_jobs != None ) {
		XtDestroyWidget(q->label_jobs);
		q->label_jobs = None;
	}
}

static void deleteButton(Widget w UNUSED, XtPointer privdata, XtPointer calldata UNUSED) {
	struct job *job = privdata;

	notified_logged_action(context, job->parent->parent->popup,
			lpql_rescan, job->parent->parent,
			"lprm", "-P", job->parent->name, job->id, NULL);
}

static void newjob(struct printer_window *w, struct queue_part *q,
		const char *rank, size_t rank_l, const char *id, size_t id_l,
		const char *line, size_t len) {
	char *text = alloca(len+1);
	const char *labelname;
	struct job *n, *j;
	Widget behind;

	if( rank_l == 0 )
		return;
	memcpy(text,line,len);
	text[len] = '\0';
	if( rank_l == 4 && strncmp(rank,"done",4) == 0 )
		labelname = "donejob";
	else if( rank_l == 6 && strncmp(rank,"active",6) == 0 ) {
		labelname = "activejob";
		showActiveIcon(w);
	} else if( rank_l >= 7 && strncmp(rank,"stalled",7) == 0 ) {
		labelname = "stalledjob";
		showActiveIcon(w);
	} else if ( rank_l > 0 && isdigit(*rank) ) {
		labelname = "normaljob";
		showActiveIcon(w);
	} else if( rank_l == 5 && strncmp(rank,"error",5) == 0 )
		labelname = "errorjob";
	else
		labelname = "strangejob";
	n = calloc(1,sizeof(struct job));
	if( n == NULL )
		return;
	n->id = strndup(id, id_l);
	if( n->id == NULL ) {
		free(n);
		return;
	}
	n->parent = q;
	assert( q->label_jobs != None );
	if( q->jobs == NULL ) {
		j = NULL;
		behind = q->label_jobs;
	} else {
		j = q->jobs;
		while( j->next != NULL )
			j = j->next;
		behind = j->label;
	}
	n->label = XtVaCreateManagedWidget(labelname,
			labelWidgetClass, q->form,
			XtNlabel,		text,
			XtNfromVert,		(XtPointer)behind,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	n->button = XtVaCreateManagedWidget("delete",
			commandWidgetClass, q->form,
			XtNfromVert,		(XtPointer)behind,
			XtNfromHoriz,		(XtPointer)n->label,
			XtNtop,			(XtPointer)XawChainTop,
			XtNbottom,		(XtPointer)XawChainTop,
			XtNleft,		(XtPointer)XawChainLeft,
			XtNright,		(XtPointer)XawChainLeft,
			XtNresizable,		(XtPointer)True,
			NULL);
	XtAddCallback(n->button, XtNcallback, deleteButton, n);
	if( j == NULL )
		q->jobs = n;
	else
		j->next = n;

}

static void parsequeueline(struct printer_window *w, const char *line, size_t len) {
	const char *e;

#define startswith(a) (len > sizeof(a)-1 && memcmp(line, a, sizeof(a)-1) == 0 )?(line +=sizeof(a)-1,len-= sizeof(a)-1,True):(False)
#define runover(cond) while( len > 0 && cond ) { line++;len--; }
#define toendof(cond) {e = line; while( (size_t)(e-line) <= len && cond ) e++; }
#define setlabel(which, p, l) { char *n = strndup(p,l); if( n == NULL ) { XtVaSetValues(w->parsepart->label_ ## which , XtNlabel, "OoM", NULL); } else { XtVaSetValues(w->parsepart->label_ ## which , XtNlabel, n, NULL); free(w->parsepart->text_ ## which); w->parsepart->text_ ## which = n;}}
#define startwithend()	{len -= (e-line); line = e; }

	if( startswith("Printer: ") ) {
		parsequeuepartend(w);
		runover( isspace(*line) );
		toendof( !isspace(*e) );
		w->parsepart = queue_part(w, line, e-line);
		if( w->parsepart == NULL ) {
			w->parsemode = pm_IGNORE;
			return;
		}
		w->parsemode = pm_HEADER;
		startwithend();
		w->parsepart->stopped = False;
		w->parsepart->disabled = False;
		runover( isspace(*line) );
		while( len > 0 ) {
			if( *line != '(' ) {
				return;
			}
			line++;len--;
			while( len > 0 ) {
				runover( isspace(*line) );
				if( startswith("dest ") ) {
					toendof(*e != ',' && *e != ')' );
					setlabel(dest,line,e-line);
					startwithend();
				} else if( startswith("queueing disabled") ) {
					runover(*line != ',' && *line != ')' );
					w->parsepart->disabled = True;
				} else if( startswith("spooling disabled") ) {
					runover(*line != ',' && *line != ')' );
					w->parsepart->disabled = True;
				} else if( startswith("printing disabled") ) {
					runover(*line != ',' && *line != ')' );
					w->parsepart->stopped = True;
				} else {
					toendof(*e != ',' && *e != ')' );
					setlabel(error,line,e-line);
					startwithend();
				}
				runover( isspace(*line) );
				if( len <= 0 )
					return;
				if( *line == ')' )
					break;
				if( *line != ',' )
					return;
				line++;len--;
			}
			if( len <= 0 || *line != ')' )
				return;
			line++;len--;
			runover( isspace(*line) );
		}
		XtVaSetValues(w->parsepart->checkbox_print,
			XtNbitmap, w->parsepart->stopped?uncheck_bm:check_bm,
			XtNsensitive, (XtPointer)True,
			NULL);
		XtVaSetValues(w->parsepart->checkbox_queue,
			XtNbitmap, w->parsepart->disabled?uncheck_bm:check_bm,
			XtNsensitive, (XtPointer)True,
			NULL);
		return;
	}
	if( startswith("Printer '") ) {
		parsequeuepartend(w);
		toendof( *e != '\'' );
		if( (size_t)(e-line) >= len )
			return;
		w->parsepart = queue_part(w, line, e-line);
		w->parsemode = pm_START;
		if( w->parsepart == NULL )
			return;
		e++;
		startwithend();
		if( startswith(" - ") )
			{;}
		runover( isspace(*line) );
		setlabel(error,line,len);
		XtVaSetValues(w->parsepart->checkbox_queue,
			XtNsensitive, (XtPointer)False,
			NULL);
		XtVaSetValues(w->parsepart->checkbox_print,
			XtNsensitive, (XtPointer)False,
			NULL);
	}
	if( w->parsemode == pm_IGNORE )
		return;
	if( w->parsemode == pm_HEADER ) {
		if( startswith(" Queue: ") ) {
			setlabel(state,line,len);
			return;
		}
		if( len >= 6 && strncmp(line, " Rank ",6 ) == 0 ) {
			preparejobs(w->parsepart, line, len);
			w->parsemode = pm_JOBS;
			return;
		}
	}
	if( w->parsemode == pm_JOBS ) {

		if( len > 0 && ( isdigit(*line) || islower(*line) ) ) {
			const char *id;
			size_t rank_l, id_l;

			toendof( !isspace(*e) );
			rank_l = e-line;
			while( (size_t)(e-line) < len && isspace(*e) )
				e++;
			id = e;
			while( (size_t)(e-line) < len && !isspace(*e) )
				e++;
			id_l = e-id;

			newjob(w, w->parsepart,
					line, 	rank_l,
					id,	id_l,
					line,	len);
			return;
		}
		w->parsemode = pm_START;
	}
	return;

#undef startswith
#undef runover
#undef startwithend
}

static void timed_rescan(XtPointer privdata, XtIntervalId *timer UNUSED) {
	struct printer_window *window = privdata;

	window->rescanTimer = (XtIntervalId)-1;
	lpql_rescan(None, window, NULL);
}

static void lpql(void *privdata, const char *data UNUSED, size_t newsize, bool finished) {
	struct printer_window *window = privdata;
	size_t i;

	assert( window->action != NULL );
	assert( newsize >= (size_t)window->parsed );

	for( i = window->parsed ; i < newsize ; i++ ) {
		if( data[i] == '\n' ) {
			size_t j = i;
			while( j > window->parsed && isspace(data[j-1]) )
				j--;
			if( j >= window->parsed ) {
				parsequeueline(window,
						data+window->parsed,
						j-window->parsed);
			}
			window->parsed = i+1;
		}
	}

	if( finished ) {
		if( newsize > window->parsed ) {
			size_t j = newsize;
			while( j > window->parsed && isspace(data[j-1]) )
				j--;
			if( j >= window->parsed ) {
				parsequeueline(window,
						data+window->parsed,
						j-window->parsed);
			}
		}
		parsequeuepartend(window);
		if( !window->hasActive )
			showInactiveIcon(window);
		window->parsed = 0;
		window->parsemode = pm_START;
		abortCollector(window->action);
		window->action = NULL;
		if( window->popup == NULL ) {
			// free(window); //???
			return;
		} else {
			XtVaSetValues(window->rescanBtn,
					XtNsensitive,	(XtPointer)True,
					NULL);
			if( window->rescanTimer == (XtIntervalId)-1
			    && window->resources.rescanTimeOut > 0 ) {
				window->rescanTimer = XtAppAddTimeOut(context,
						window->resources.rescanTimeOut,
						timed_rescan, window);
			}
		}
	}
}

static void lpql_destroy(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct printer_window **pp, *window = privdata;

	if( window->rescanTimer != (XtIntervalId)-1 ) {
		XtRemoveTimeOut(window->rescanTimer);
		window->rescanTimer = (XtIntervalId)-1;
	}

	if( window->action != NULL )
		abortCollector(window->action);
	window->action = NULL;
	window->popup = NULL;
	pp = &printer_window_root;
	while( *pp != NULL && *pp != window)
		pp = &(*pp)->next;
	if( *pp == window )
		*pp = window->next;
	else
		fprintf(stderr, "Lost printer window!!!\n");
	free_printer_window(window);
}

static void lpql_close(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct printer_window *window = privdata;

	if( window->rescanTimer != (XtIntervalId)-1 ) {
		XtRemoveTimeOut(window->rescanTimer);
		window->rescanTimer = (XtIntervalId)-1;
	}
	if( window->action != NULL )
		abortCollector(window->action);
	window->action = NULL;
	XtDestroyWidget(window->popup);
}

static void lpql_rescan(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct printer_window *window = privdata;

	if( window->rescanTimer != (XtIntervalId)-1 ) {
		XtRemoveTimeOut(window->rescanTimer);
		window->rescanTimer = (XtIntervalId)-1;
	}
	if( window->action != NULL )
		abortCollector(window->action);
	window->hasActive = false;
	window->action = newCollectorVa(context, lpql, window,
			"lpq", "-lll", "-P", window->printer, NULL);
	XtVaSetValues(window->rescanBtn,
			XtNsensitive,	(XtPointer)False,
			NULL);
}

static void lpql_raw(Widget w UNUSED, XtPointer privdata,
		XtPointer cad UNUSED) {
	struct printer_window *window = privdata;

	logged_action(context, app_shell,
			"lpq", "-L", "-P", window->printer, NULL);
}

static struct printer_window *createPrinterWindow(Widget parent, const char *printername) {
	struct printer_window *window;
	Widget w;

	window = calloc(1,sizeof(struct printer_window));
	if( window == NULL )
		return NULL;
	window->printer = strdup(printername);
	if( window->printer == NULL ) {
		free(window);
		return NULL;
	}
	window->rescanTimer = (XtIntervalId)-1;

	window->popup = XtVaCreatePopupShell(window->printer,
			topLevelShellWidgetClass,
			parent,
			XtNiconPixmap,		bm_printer,
			XtNiconMask,    	bm_printermask,
			XtNsensitive,		(XtPointer)True,
			NULL);
	window->outer_form = XtVaCreateManagedWidget("lpqPaned",
			panedWidgetClass, window->popup,
			NULL);
	window->viewport = XtVaCreateManagedWidget("viewport",
			viewportWidgetClass, window->outer_form,
			NULL);
	window->inner_form = XtVaCreateManagedWidget("lpq",
			vboxWidgetClass, window->viewport,
			NULL);

	w = XtVaCreateManagedWidget("buttons",
			boxWidgetClass, window->outer_form,
			NULL);
	window->closeBtn = XtVaCreateManagedWidget("close",
			commandWidgetClass, w,
			NULL);
	window->rescanBtn = XtVaCreateManagedWidget("rescan",
			commandWidgetClass, w,
			XtNsensitive,	(XtPointer)False,
			NULL);
	window->rawBtn = XtVaCreateManagedWidget("raw data",
			commandWidgetClass, w,
			NULL);
	XtAddCallback(window->closeBtn, XtNcallback, lpql_close, window);
	XtAddCallback(window->rescanBtn, XtNcallback, lpql_rescan, window);
	XtAddCallback(window->rawBtn, XtNcallback, lpql_raw, window);
	XtAddCallback(window->popup, XtNdestroyCallback, lpql_destroy, window);
	XtInstallAllAccelerators(window->popup, window->popup);
//	XtInstallAllAccelerators(window->text, window->popup);
//	XtSetKeyboardFocus(window->popup, window->text);
	XtRealizeWidget(window->popup);
	setDeleteHandler(window->popup, wmdDestroy);
	XtGetApplicationResources(window->popup, &window->resources,
			printerwindow_resources, XtNumber(printerwindow_resources),
			NULL, ZERO);
	// TODO: start at realize time, only when not iconified
	window->action = newCollectorVa(context, lpql, window,
			"lpq", "-lll", "-P", window->printer, NULL);
	return window;
}

void printerWindow(const char *printername) {
	struct printer_window *p = printer_window_root;

	while( p != NULL && strcmp(p->printer,printername) != 0 )
		p = p->next;

	if( p == NULL ) {
		p = createPrinterWindow(app_shell, printername);
		if( p == NULL )
			return;
		p->next = printer_window_root;
		printer_window_root = p;
	}
	XtPopup(p->popup, XtGrabNone);
}
