/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "geometry.h"

#include <string.h>
#include <GL/gl.h>
#include <GL/glu.h>

#include <opengl.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolConfigFile.h>
#include <openGLFunctions/objectList.h>
#include <openGLFunctions/text.h>
#include <visu_extension.h>
#include <visu_configFile.h>

#define GEOMETRY_DIFF_ID        "geometry_diff"
#define GEOMETRY_DIFF_MINMAX_ID "geometry_diffMinMax"

#define FLAG_RESOURCE_ARROW "nodeDisplacement_arrow"
#define DESC_RESOURCE_ARROW "Describe the arrow to be drawn ; four floats (tail lg. tail rd. head lg. head rd., negative values means auto)"
static gboolean readArrow(gchar **lines, int nbLines, int position,
			  VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_RATIO_MIN "nodeDisplacement_minThreshold"
#define DESC_RESOURCE_RATIO_MIN "Choose the minimum value for drawn arrows in geometry differences ; float (ratio threshold if between -1 and 0)"
#define DEFT_RESOURCE_RATIO_MIN -0.3f
static gboolean readRatioMin(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_RATIO_STR "nodeDisplacement_lblThreshold"
#define DESC_RESOURCE_RATIO_STR "Choose the minimum value for labels in geometry differences ; float (ratio threshold if between -1 and 0)"
#define DEFT_RESOURCE_RATIO_STR -0.9f
static gboolean readRatioStr(gchar **lines, int nbLines, int position,
			     VisuData *dataObj, GError **error);

#define FLAG_RESOURCE_MULT "nodeDisplacement_factor"
#define DESC_RESOURCE_MULT "Choose the factor to draw arrows in geometry differences ; float (negative means auto)"
#define DEFT_RESOURCE_MULT -1.f
static gboolean readMult(gchar **lines, int nbLines, int position,
			 VisuData *dataObj, GError **error);

static void exportResources(GString *data, VisuData *dataObj);

/* Local variables. */
static OpenGLExtension *extGeo = NULL;
static float ratioMin = DEFT_RESOURCE_RATIO_MIN;
static float ratioStr = DEFT_RESOURCE_RATIO_STR;
static float mult = DEFT_RESOURCE_MULT;
static float arrow[4] = {-1.f, -1.f, -1.f, -1.f};

/* Local methods. */
static void geometryCreate_openGLExtension();

static void freeData(gpointer obj, gpointer data _U_)
{
#if GLIB_MINOR_VERSION > 9
  g_slice_free1(sizeof(float) * 6, obj);
#else
  g_free(obj);
#endif
}
static gpointer newOrCopyData(gconstpointer orig, gpointer user_data _U_)
{
  float *data;

#if GLIB_MINOR_VERSION > 9
  data = g_slice_alloc(sizeof(float) * 6);
#else
  data = g_malloc(sizeof(float) * 6);
#endif
  if (orig)
    memcpy(data, orig, sizeof(float) * 6);
  else
    memset(data, 0, sizeof(float) * 6);

  return (gpointer)data;
}

void periodicDistance(float diff[3], VisuData *data, VisuNode *node1, VisuNode *node2)
{
  float red[3];
  OpenGLBoxBoundaries bc;

  diff[0] = node1->xyz[0] + node1->translation[0] -
    node2->xyz[0] - node2->translation[0];
  diff[1] = node1->xyz[1] + node1->translation[1] -
    node2->xyz[1] - node2->translation[1];
  diff[2] = node1->xyz[2] + node1->translation[2] -
    node2->xyz[2] - node2->translation[2];
  
  bc = visuDataGet_boundaryConditions(data);
  if (bc == BOX_FREE)
    return;

  visuDataConvert_XYZtoBoxCoordinates(data, red, diff);
  if (bc != BOX_SURFACE_YZ)
    {
      while (red[0] >= 0.5f) red[0] -= 1.f;
      while (red[0] < -0.5f) red[0] += 1.f;
    }

  if (bc != BOX_SURFACE_ZX)
    {
      while (red[1] >= 0.5f) red[1] -= 1.f;
      while (red[1] < -0.5f) red[1] += 1.f;
    }

  if (bc != BOX_SURFACE_XY)
    {
      while (red[2] >= 0.5f) red[2] -= 1.f;
      while (red[2] < -0.5f) red[2] += 1.f;
    }
  visuDataConvert_boxCoordinatestoXYZ(data, diff, red);
}

gboolean geometryDiff(VisuData *dataRef, VisuData *data)
{
  VisuNodeArray *nodesRef, *nodes;
  guint i;
  VisuNodeProperty *prop;
  VisuDataIter iter, iterRef;
  GValue diffValue = {0, {{0}, {0}}};
  float *diff;
  float *minMax;

  g_return_val_if_fail(IS_VISU_DATA_TYPE(dataRef), FALSE);
  g_return_val_if_fail(IS_VISU_DATA_TYPE(data), FALSE);

  DBG_fprintf(stderr, "Geometry: making a diff between %p and %p.\n",
	      (gpointer)dataRef, (gpointer)data);

  nodesRef = visuDataGet_nodeArray(dataRef);
  nodes    = visuDataGet_nodeArray(data);

  if (nodes->ntype != nodesRef->ntype)
    return FALSE;

  for (i = 0; i < nodes->ntype; i++)
    if (nodes->numberOfStoredNodes[i] != nodesRef->numberOfStoredNodes[i])
      return FALSE;

  /* Ok, here the two files match with respect of number of nodes per
     element. */
  g_value_init(&diffValue, G_TYPE_POINTER);

  /* We add a VisuData property to store the min max values. */
  minMax = g_malloc(sizeof(float) * 2);
  g_object_set_data_full(G_OBJECT(data), GEOMETRY_DIFF_MINMAX_ID,
			 (gpointer)minMax, g_free);
  minMax[0] = G_MAXFLOAT;
  minMax[1] = 0.f;

  /* Add a node property with the number of columns. */
  prop = visuNodeNew_pointerProperty(nodes, GEOMETRY_DIFF_ID,
				     freeData, newOrCopyData, (gpointer)0);

  visuDataIter_new(data, &iter);
  visuDataIter_new(dataRef, &iterRef);
  visuDataIter_start(dataRef, &iterRef);
  for (visuDataIter_start(data, &iter); iter.node;
       visuDataIter_next(data, &iter))
    {
      g_return_val_if_fail(iterRef.node, FALSE);

      /* We read the data from the line, if not void. */
      diff = newOrCopyData((gconstpointer)0, (gpointer)0);
      periodicDistance(diff, data, iter.node, iterRef.node);
      matrix_cartesianToSpherical(diff + 3, diff);
      DBG_fprintf(stderr, "Geometry: diff atom %d, %9g %9g %9g (%9g %5g %5g).\n",
		  iter.node->number, diff[0], diff[1], diff[2],
		  diff[3], diff[4], diff[5]);
      minMax[0] = MIN(minMax[0], diff[3]);
      minMax[1] = MAX(minMax[1], diff[3]);
      
      /* Associates the values to the node. */
      g_value_set_pointer(&diffValue, diff);
      visuNodePropertySet_value(prop, iter.node, &diffValue);

      visuDataIter_next(dataRef, &iterRef);
    }

  return TRUE;
}

void geometryDraw(VisuData *data)
{
  VisuNodeArray *nodes;
  VisuDataIter iter;
  GValue diffValue = {0, {{0}, {0}}};
  float *diff, *minMax, xyz[3], rdT_, rdT, lgT, rdH_, rdH, lgH_, lgH, fact;
  float eleSize, rMult, sMult, r, s;
  VisuNodeProperty *prop;
  GLUquadricObj *obj;
  VisuElement *prevEle;
  char distStr[108];

  obj = gluNewQuadric();

  g_return_if_fail(IS_VISU_DATA_TYPE(data));

  g_value_init(&diffValue, G_TYPE_POINTER);

  minMax = (float*)g_object_get_data(G_OBJECT(data), GEOMETRY_DIFF_MINMAX_ID);
  g_return_if_fail(minMax);

  /* We compute the maximum size of the arrow tail. */
  eleSize = visuDataGet_allElementExtens(data);

  nodes = visuDataGet_nodeArray(data);
  prop = visuNodeGet_property(nodes, GEOMETRY_DIFF_ID);
  g_return_if_fail(prop);

  if (arrow[1] <= 0.f || mult <= 0.f)
    rdT_ = rdT = 0.2f * eleSize;
  else
    rdT_ = rdT = arrow[1];
  if (arrow[2] <= 0.f || mult <= 0.f)
    lgH_ = lgH = 0.5f * eleSize;
  else
    lgH_ = lgH = arrow[2];
  if (arrow[3] <= 0.f || mult <= 0.f)
    rdH_ = rdH = 0.3f * eleSize;
  else
    rdH_ = rdH = arrow[3];
  if (mult <= 0.f)
    fact = eleSize * 4.f / minMax[1];
  else
    fact = mult;
  if (ratioMin <= 0.f)
    {
      rMult = 1.f / minMax[1];
      r = -1.f;
    }
  else
    {
      rMult = 1.f;
      r = 1.f;
    }
  if (ratioStr <= 0.f)
    {
      sMult = 1.f / minMax[1];
      s = -1.f;
    }
  else
    {
      sMult = 1.f;
      s = 1.f;
    }

  prevEle = (VisuElement*)0;
  visuDataIter_new(data, &iter);
  for (visuDataIter_start(data, &iter); iter.node;
       visuDataIter_next(data, &iter))
    {
      if (!iter.node->rendered || !iter.element->rendered)
	continue;

      visuNodePropertyGet_value(prop, iter.node, &diffValue);
      diff = (float*)g_value_get_pointer(&diffValue);

      if (diff[3] * rMult > ratioMin * r)
	{
	  /* The size of the arrow... */
	  lgT = diff[3] * fact;
	  if (arrow[0] > 0.f && mult > 0.f)
	    {
	      rdT = rdT_ * diff[3] * fact;
	      lgH = lgH_ * diff[3] * fact;
	      rdH = rdH_ * diff[3] * fact;
	    }

	  /* We draw the arrow. */
	  visuDataGet_nodePosition(data, iter.node, xyz);

	  glPushMatrix();
	  glTranslated(xyz[0], xyz[1], xyz[2]);
	  glRotated(diff[5], 0, 0, 1);  
	  glRotated(diff[4], 0, 1, 0); 

	  if (prevEle != iter.element)
	    {
	      openGLSet_highlightColor(iter.element->material,
				       iter.element->rgb, 1.f);
	      prevEle = iter.element;
	    }
	  openGLObjectListDraw_smoothArrow(obj, -1, FALSE,
					   MAX(0.f, lgT - lgH), rdT, 10, FALSE,
					   MIN(lgH, lgT), rdH, 10, FALSE);

	  if (diff[3] * sMult > ratioStr * s)
	    {
	      glRasterPos3f(0.f, 0.f, 0.f);
	      sprintf(distStr, "%6.3f", diff[3]);
	      openGLText_drawChars(distStr, TEXT_NORMAL);
	    }

	  glPopMatrix();
	}
    }

  gluDeleteQuadric(obj);
}

void geometryRebuild_list(VisuData *dataObj)
{
  DBG_fprintf(stderr, "Geometry: rebuilding object list for visuData %p.\n",
	      (gpointer)dataObj);

  if (!extGeo)
    geometryCreate_openGLExtension();

  glNewList(extGeo->objectListId, GL_COMPILE);
  geometryDraw(dataObj);
  glEndList();
}

static void geometryCreate_openGLExtension()
{
  int id;

  /* Get an OpenGL identifier to store the arrows and labels. */
  id     = openGLObjectList_new(1);
  extGeo = OpenGLExtension_new("geometry", "geometry",
			       "arrows representing node moves.",
			       id, geometryRebuild_list);
  OpenGLExtensionSet_priority(extGeo, OPENGL_EXTENSION_PRIORITY_FIRST + 1);
  extGeo->used = 0;
  OpenGLExtensionRegister(extGeo);
}

gboolean geometrySet_active(gboolean value)
{
  if (!extGeo)
    geometryCreate_openGLExtension();

  if (value == extGeo->used)
    return FALSE;

  extGeo->used = value;

  return TRUE;
}

void geometryInit()
{
  VisuConfigFileEntry *conf;

  DBG_fprintf(stderr, "Geometry: set the conf entries for this module.\n");
  conf = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_ARROW,
				 DESC_RESOURCE_ARROW,
				 1, readArrow);
  visuConfigFileSet_version(conf, 3.5f);
  conf = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_RATIO_MIN,
				 DESC_RESOURCE_RATIO_MIN,
				 1, readRatioMin);
  visuConfigFileSet_version(conf, 3.5f);
  conf = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_RATIO_STR,
				 DESC_RESOURCE_RATIO_STR,
				 1, readRatioStr);
  visuConfigFileSet_version(conf, 3.5f);
  conf = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				 FLAG_RESOURCE_MULT,
				 DESC_RESOURCE_MULT,
				 1, readMult);
  visuConfigFileSet_version(conf, 3.5f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResources);
}

static gboolean readArrow(gchar **lines, int nbLines, int position,
			  VisuData *dataObj _U_, GError **error)
{
  float vals[4];

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, vals, 4, error))
    return FALSE;
  arrow[0] = vals[0];
  arrow[1] = vals[1];
  arrow[2] = vals[2];
  arrow[3] = vals[3];

  return TRUE;
}
static gboolean readRatioMin(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &val, 1, error))
    return FALSE;
  ratioMin = val;

  return TRUE;
}
static gboolean readRatioStr(gchar **lines, int nbLines, int position,
			     VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &val, 1, error))
    return FALSE;
  ratioStr = val;

  return TRUE;
}
static gboolean readMult(gchar **lines, int nbLines, int position,
			 VisuData *dataObj _U_, GError **error)
{
  float val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &val, 1, error))
    return FALSE;
  mult = val;

  return TRUE;
}
static void exportResources(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_ARROW);
  g_string_append_printf(data, "%s:\n    %f %f %f %f\n", FLAG_RESOURCE_ARROW,
			 arrow[0], arrow[1], arrow[2], arrow[3]);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_MULT);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_MULT, mult);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RATIO_MIN);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_RATIO_MIN, ratioMin);

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_RATIO_STR);
  g_string_append_printf(data, "%s:\n    %f\n", FLAG_RESOURCE_RATIO_STR, ratioStr);
}
