/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-page-channel-group.c:
 * A GtkNotebook page exposing org.freedesktop.Telepathy.Channel.Interface.Group
 * functionality.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "page-channel-group.h"

#include <string.h>

#include <glade/glade.h>
#include <telepathy-glib/enums.h>

#include "constants.h"
#include "dlg-add-members.h"
#include "dlg-get-text.h"
#include "page-priv.h"
#include "preferences.h"
#include "util.h"

enum {
    TI_COLUMN_MEMBERS_LIST_HANDLE = 0,
    TI_COLUMN_MEMBERS_LIST_NAME,
    TI_COLUMN_MEMBERS_LIST_STATUS,
    TI_COLUMN_MEMBERS_LIST_OWNER,
    TI_COLUMN_MEMBERS_LIST_OWNER_NAME,
    TI_COLUMN_MEMBERS_LIST_N_COLUMNS
};

#define TI_GROUP_MEMBER_STATUS_REGULAR        "member"
#define TI_GROUP_MEMBER_STATUS_LOCAL_PENDING  "local pending"
#define TI_GROUP_MEMBER_STATUS_REMOTE_PENDING "remote pending"

struct _TIPageChannelGroupClass {
    TIPageClass parent;
};

G_DEFINE_TYPE (TIPageChannelGroup, ti_page_channel_group, TI_TYPE_PAGE);

/* Function prototypes */
static void _ti_page_channel_group_setup_page (TIPage *page, GladeXML *glade_xml);
static void _ti_page_channel_group_build_members_treeview (TIPageChannelGroup *self, GladeXML *glade_xml);
static void _ti_page_channel_group_fill_members_list (TIPageChannelGroup *self);
static void _ti_page_channel_group_append_members_to_list (TIPageChannelGroup *,
    const GArray *, const gchar *);
static void _ti_page_channel_group_setup_group_flags (TIPageChannelGroup *self, GladeXML *glade_xml);
static void _ti_page_channel_group_remove_selected_members (TIPageChannelGroup *self);
static void _ti_page_channel_group_add_members (TIPageChannelGroup *self);
static void _ti_page_channel_group_log_members_changed (TIPageChannelGroup *,
    const gchar *message, const GArray *added, const GArray *removed,
    const GArray *local_pending, const GArray *remote_pending,
    guint actor, guint reason);
static void _ti_page_channel_group_remove_members (TIPageChannelGroup *,
    const GArray *);
static void _ti_page_channel_group_remove_member (TIPageChannelGroup *self, guint handle);
static void _ti_page_channel_group_update_button_remove_sensitivity (TIPageChannelGroup *self);
static void _ti_page_channel_group_clear_log (TIPageChannelGroup *self);
static void _ti_page_channel_group_log_group_flags_changed (TIPageChannelGroup *self, guint added, guint removed);
static void _ti_page_channel_group_refresh_group_flags_checkboxes (TIPageChannelGroup *self);
static void _ti_page_channel_group_update_selection_permissions (TIPageChannelGroup *self);
static void _group_flags_to_string (GString *text, guint flags);
static void _ti_page_channel_group_handle_display_mode_changed (TIPageChannelGroup *self, guint handle_display_mode);
static gboolean _ti_page_channel_get_contact_row (TIPageChannelGroup *self, guint contact_handle, GtkTreeIter *iter);

/**
 * Instance private data.
 */
struct _TIPageChannelGroupPrivate {
    TpChannel *channel;

    TIPreferences *preferences;
    TIHandleMapper *handle_mapper;

    guint group_flags;

    TIDlgAddMembers *dialog_add_members;
    TIDlgGetText *dialog_get_text;

    GtkListStore *members_list;
    GtkTreeView *members_tree_view;
    GtkTreeSelection *members_selection;

    GtkWidget *checkbutton_can_add;
    GtkWidget *checkbutton_can_remove;
    GtkWidget *checkbutton_can_rescind;
    GtkWidget *checkbutton_message_add;
    GtkWidget *checkbutton_message_remove;
    GtkWidget *checkbutton_message_accept;
    GtkWidget *checkbutton_message_reject;
    GtkWidget *checkbutton_message_rescind;
    GtkWidget *checkbutton_channel_specific_handles;

    GtkButton *button_remove_members;
    GtkButton *button_add_members;

    GtkTextBuffer *textbuffer_event_log;
};
typedef struct _TIPageChannelGroupPrivate TIPageChannelGroupPrivate;

#define TI_PAGE_CHANNEL_GROUP_GET_PRIVATE(object)  (G_TYPE_INSTANCE_GET_PRIVATE ((object), TI_TYPE_PAGE_CHANNEL_GROUP, TIPageChannelGroupPrivate))

/**
 * Drop all references to other objects.
 */
static void
ti_page_channel_group_dispose (GObject *object)
{
  TIPageChannelGroup *self = TI_PAGE_CHANNEL_GROUP (object);
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);

  if (priv->channel != NULL)
    {
      g_object_unref (priv->channel);
      priv->channel = NULL;
    }

  if (priv->preferences != NULL)
    {
      g_signal_handlers_disconnect_by_func (priv->preferences,
          G_CALLBACK (_ti_page_channel_group_handle_display_mode_changed),
          self);

      g_object_unref (priv->preferences);
      priv->preferences = NULL;
  }

  TI_OBJ_UNREF (priv->handle_mapper);
  TI_OBJ_UNREF (priv->dialog_add_members);
  TI_OBJ_UNREF (priv->dialog_get_text);
  TI_OBJ_UNREF (priv->members_list);

  G_OBJECT_CLASS (ti_page_channel_group_parent_class)->dispose (object);
}

static void
ti_page_channel_group_class_init (TIPageChannelGroupClass *klass)
{
  GObjectClass *gobject_class = G_OBJECT_CLASS (klass);
  TIPageClass *page_class = TI_PAGE_CLASS (klass);

  gobject_class->dispose = ti_page_channel_group_dispose;

  page_class->setup_page = _ti_page_channel_group_setup_page;

  g_type_class_add_private (klass, sizeof (TIPageChannelGroupPrivate));
}

static void
ti_page_channel_group_init (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);

  priv->channel = NULL;
  priv->preferences = ti_preferences_new ();
  priv->handle_mapper = NULL;
  priv->dialog_add_members = NULL;
  priv->dialog_get_text = NULL;
  priv->members_list = NULL;
}

static void
_ti_page_channel_group_members_changed (TpChannel *channel,
                                        const gchar *message,
                                        const GArray *added,
                                        const GArray *removed,
                                        const GArray *local_pending,
                                        const GArray *remote_pending,
                                        guint actor,
                                        guint reason,
                                        gpointer unused,
                                        GObject *weak_object)
{
  TIPageChannelGroup *self = TI_PAGE_CHANNEL_GROUP (weak_object);

  g_assert (added != NULL);
  g_assert (local_pending != NULL);
  g_assert (remote_pending != NULL);

  _ti_page_channel_group_log_members_changed (self, message, added, removed,
      local_pending, remote_pending, actor, reason);

  _ti_page_channel_group_append_members_to_list (self, added,
      TI_GROUP_MEMBER_STATUS_REGULAR);
  _ti_page_channel_group_append_members_to_list (self, local_pending,
      TI_GROUP_MEMBER_STATUS_LOCAL_PENDING);
  _ti_page_channel_group_append_members_to_list (self, remote_pending,
      TI_GROUP_MEMBER_STATUS_REMOTE_PENDING);

  _ti_page_channel_group_remove_members (self, removed);
}

static void
_ti_page_channel_group_group_flags_changed (TpChannel *channel,
                                            guint added,
                                            guint removed,
                                            gpointer unused,
                                            GObject *weak_object)
{
  TIPageChannelGroup *self = TI_PAGE_CHANNEL_GROUP (weak_object);
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gboolean can_add;

  _ti_page_channel_group_log_group_flags_changed (self, added,
      removed);

  priv->group_flags = (priv->group_flags | added) & (~removed);

  _ti_page_channel_group_refresh_group_flags_checkboxes (self);

  can_add = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_CAN_ADD) != 0;
  gtk_widget_set_sensitive (GTK_WIDGET (priv->button_remove_members), can_add);

  _ti_page_channel_group_update_selection_permissions (self);
  _ti_page_channel_group_update_button_remove_sensitivity (self);
}


static void
_ti_page_channel_group_self_handle_changed (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gchar *str;
  GtkTextIter iter;

  str = g_strdup_printf ("SelfHandle = %u",
     tp_channel_group_get_self_handle (priv->channel));
  gtk_text_buffer_get_end_iter (priv->textbuffer_event_log, &iter);
  gtk_text_buffer_insert (priv->textbuffer_event_log, &iter, str, -1);
  g_free (str);
}


TIPageChannelGroup *
ti_page_channel_group_new (GtkNotebook *parent_notebook,
                           GtkWindow *parent_window,
                           TpChannel *channel,
                           TIHandleMapper *handle_mapper)
{
  TIPage *self = NULL;
  TIPageChannelGroupPrivate *priv = NULL;

  g_assert (channel != NULL);
  g_return_val_if_fail (tp_channel_is_ready (channel), NULL);

  self = g_object_new (TI_TYPE_PAGE_CHANNEL_GROUP, NULL);
  priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);

  priv->channel = g_object_ref (channel);
  priv->handle_mapper = g_object_ref (handle_mapper);

  tp_cli_channel_interface_group_connect_to_members_changed (priv->channel,
      _ti_page_channel_group_members_changed, NULL, NULL,
      (GObject *) self, NULL);

  tp_cli_channel_interface_group_connect_to_group_flags_changed (priv->channel,
      _ti_page_channel_group_group_flags_changed, NULL, NULL,
      (GObject *) self, NULL);

  g_signal_connect_swapped (priv->preferences, "handle-display-mode-changed",
      G_CALLBACK (_ti_page_channel_group_handle_display_mode_changed),
      self);

  priv->dialog_add_members = ti_dlg_add_members_new (parent_window,
      handle_mapper);

  if (priv->dialog_add_members == NULL)
    {
      g_object_unref (self);
      self = NULL;
      goto CLEAN_UP;
    }

  priv->dialog_get_text = ti_dlg_get_text_new (parent_window,
      "Enter message:");

  if (priv->dialog_get_text == NULL)
    {
      g_object_unref (self);
      self = NULL;
      goto CLEAN_UP;
    }

  _ti_page_new (&self, parent_notebook,
      "page-channel-group.xml");

CLEAN_UP:
    return (TIPageChannelGroup *) self;
}

/**
* Setup Page - Helper Function
*/
static void
_ti_page_channel_group_setup_page (TIPage *page, GladeXML *glade_xml)
{
  TIPageChannelGroup *self = TI_PAGE_CHANNEL_GROUP (page);
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gboolean can_add;
  GtkWidget *widget;

  _ti_page_channel_group_setup_group_flags (self, glade_xml);

  _ti_page_channel_group_build_members_treeview (self, glade_xml);
  _ti_page_channel_group_fill_members_list (self);

  priv->button_remove_members = GTK_BUTTON (glade_xml_get_widget (glade_xml,
        "button_remove_members"));
  g_assert (GTK_IS_BUTTON (priv->button_remove_members));
  g_signal_connect_swapped (priv->button_remove_members, "clicked",
      G_CALLBACK (_ti_page_channel_group_remove_selected_members), self);

  priv->button_add_members = GTK_BUTTON (glade_xml_get_widget (glade_xml,
        "button_add_members"));
  g_assert (GTK_IS_BUTTON (priv->button_add_members));
  g_signal_connect_swapped (priv->button_add_members, "clicked",
      G_CALLBACK (_ti_page_channel_group_add_members), self);

  can_add = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_CAN_ADD) != 0;
  gtk_widget_set_sensitive (GTK_WIDGET (priv->button_remove_members), can_add);

  widget = glade_xml_get_widget (glade_xml, "button_refresh_members");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (GTK_BUTTON (widget), "clicked",
      G_CALLBACK (_ti_page_channel_group_fill_members_list), self);

  widget = glade_xml_get_widget (glade_xml, "textview_event_log");
  g_assert (GTK_IS_TEXT_VIEW (widget));
  priv->textbuffer_event_log = gtk_text_view_get_buffer (
      GTK_TEXT_VIEW (widget));

  widget = glade_xml_get_widget (glade_xml, "button_event_log_clear");
  g_assert (GTK_IS_BUTTON (widget));
  g_signal_connect_swapped (GTK_BUTTON (widget), "clicked",
      G_CALLBACK (_ti_page_channel_group_clear_log), self);

  g_signal_connect_swapped (priv->channel, "notify::group-self-handle",
      G_CALLBACK (_ti_page_channel_group_self_handle_changed), self);
  _ti_page_channel_group_self_handle_changed ((TIPageChannelGroup *) self);
}

static void
_ti_page_channel_group_build_members_treeview (TIPageChannelGroup *self,
                                               GladeXML *glade_xml)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;

  priv->members_tree_view = GTK_TREE_VIEW (glade_xml_get_widget (glade_xml,
        "treeview_members"));
  g_assert (priv->members_tree_view != NULL &&
      GTK_IS_TREE_VIEW (priv->members_tree_view));

  priv->members_list = gtk_list_store_new (5,
      G_TYPE_UINT /*handle*/,
      G_TYPE_STRING /*name*/,
      G_TYPE_STRING /*status*/,
      G_TYPE_UINT /*handle owner handle*/,
      G_TYPE_STRING /*handle owner name*/);
  gtk_tree_view_set_model (priv->members_tree_view,
      GTK_TREE_MODEL (priv->members_list));

  renderer = gtk_cell_renderer_text_new ();

  if (ti_preferences_get_handle_display_mode (priv->preferences) ==
      TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      column = gtk_tree_view_column_new_with_attributes ("Handle", renderer,
          "text", 0,
          NULL);
      gtk_tree_view_append_column (priv->members_tree_view, column);
    }
  else
    {
      column = gtk_tree_view_column_new_with_attributes ("Name", renderer,
          "text", 1,
          NULL);
      gtk_tree_view_append_column (priv->members_tree_view, column);
    }

  /* Status column. Displays something like "local pending", "remote pending"
   * and "regular member". */
  column = gtk_tree_view_column_new_with_attributes ("Status", renderer,
      "text", 2,
      NULL);
  gtk_tree_view_append_column (priv->members_tree_view, column);

  if (ti_preferences_get_handle_display_mode (priv->preferences) ==
      TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      column = gtk_tree_view_column_new_with_attributes ("Owner handle",
          renderer,
          "text", 3,
          NULL);
      gtk_tree_view_append_column (priv->members_tree_view, column);
    }
  else
    {
      column = gtk_tree_view_column_new_with_attributes ("Owner", renderer,
          "text", 4,
          NULL);
      gtk_tree_view_append_column (priv->members_tree_view, column);
    }

  priv->members_selection = gtk_tree_view_get_selection (
      priv->members_tree_view);

  g_signal_connect_swapped (priv->members_selection, "changed",
      G_CALLBACK (_ti_page_channel_group_update_button_remove_sensitivity),
      self);

  gtk_tree_selection_set_mode (priv->members_selection,
      GTK_SELECTION_MULTIPLE);
}

static void
_ti_page_channel_group_fill_members_list (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv =
      TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GArray *members, *lp, *rp;

  gtk_list_store_clear (priv->members_list);

  members = tp_intset_to_array (tp_channel_group_get_members (priv->channel));
  lp = tp_intset_to_array (tp_channel_group_get_local_pending (priv->channel));
  rp = tp_intset_to_array (tp_channel_group_get_remote_pending (priv->channel));

  _ti_page_channel_group_append_members_to_list (self, members,
      TI_GROUP_MEMBER_STATUS_REGULAR);
  _ti_page_channel_group_append_members_to_list (self, lp,
      TI_GROUP_MEMBER_STATUS_LOCAL_PENDING);
  _ti_page_channel_group_append_members_to_list (self, rp,
      TI_GROUP_MEMBER_STATUS_REMOTE_PENDING);

  g_array_free (members, TRUE);
  g_array_free (lp, TRUE);
  g_array_free (rp, TRUE);
}

static void
_ti_page_channel_group_append_members_to_list (TIPageChannelGroup *self,
                                               const GArray *handles,
                                               const gchar *status)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  guint i;

  for (i = 0; i < handles->len; i++)
    {
      GtkTreeIter iter;
      guint handle, owner;
      gchar *name, *oname;
      gchar *specific_status;
      TpHandle actor;
      TpChannelGroupChangeReason reason;
      const gchar *message;

      handle = g_array_index (handles, guint, i);
      owner = tp_channel_group_get_handle_owner (priv->channel, handle);

      name = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper,
          handle);
      oname = ti_handle_mapper_get_contact_handle_name (priv->handle_mapper,
          owner);

      if (name == NULL)
        name = g_strdup_printf ("%u", handle);

      if (oname == NULL)
        oname = g_strdup_printf ("%u", owner);

      if (tp_channel_group_get_local_pending_info (priv->channel, handle,
            &actor, &reason, &message))
        {
          specific_status = g_strdup_printf ("%s: actor %u, reason %u, \"%s\"",
              status, actor, reason, message);
        }
      else
        {
          specific_status = g_strdup (status);
        }


      /* If there's not already a row for that contact, we create one. */
      if (!_ti_page_channel_get_contact_row (self, handle, &iter))
        {
          gtk_list_store_append (priv->members_list, &iter);
        }

      gtk_list_store_set (priv->members_list, &iter,
          TI_COLUMN_MEMBERS_LIST_HANDLE, handle,
          TI_COLUMN_MEMBERS_LIST_NAME, name,
          TI_COLUMN_MEMBERS_LIST_STATUS, specific_status,
          TI_COLUMN_MEMBERS_LIST_OWNER, owner,
          TI_COLUMN_MEMBERS_LIST_OWNER_NAME, oname,
          -1);

      g_free (name);
      g_free (oname);
      g_free (specific_status);
    }
}

static void
_ti_page_channel_group_setup_group_flags (TIPageChannelGroup *self,
                                          GladeXML *glade_xml)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);

  priv->checkbutton_can_add = glade_xml_get_widget (glade_xml,
      "checkbutton_can_add");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_can_add));

  priv->checkbutton_can_remove = glade_xml_get_widget (glade_xml,
      "checkbutton_can_remove");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_can_remove));

  priv->checkbutton_can_rescind = glade_xml_get_widget (glade_xml,
      "checkbutton_can_rescind");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_can_rescind));

  priv->checkbutton_message_add = glade_xml_get_widget (glade_xml,
      "checkbutton_message_add");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_message_add));

  priv->checkbutton_message_remove = glade_xml_get_widget (glade_xml,
      "checkbutton_message_remove");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_message_remove));

  priv->checkbutton_message_accept = glade_xml_get_widget (glade_xml,
      "checkbutton_message_accept");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_message_accept));

  priv->checkbutton_message_reject = glade_xml_get_widget (glade_xml,
      "checkbutton_message_reject");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_message_reject));

  priv->checkbutton_message_rescind = glade_xml_get_widget (glade_xml,
      "checkbutton_message_rescind");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_message_rescind));

  priv->checkbutton_channel_specific_handles = glade_xml_get_widget (glade_xml,
      "checkbutton_channel_specific_handles");
  g_assert (GTK_IS_CHECK_BUTTON (priv->checkbutton_channel_specific_handles));

  priv->group_flags = tp_channel_group_get_flags (priv->channel);

  _ti_page_channel_group_refresh_group_flags_checkboxes (self);
}

static void
_ti_page_channel_group_remove_selected_members (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GError *error = NULL;
  GArray *members = NULL;
  gchar *message = NULL;

  members = ti_get_selected_elements (priv->members_selection,
      TI_COLUMN_MEMBERS_LIST_HANDLE, G_TYPE_UINT);
  g_assert (members != NULL);

  if ((priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_REMOVE) != 0)
    {
        if (!ti_dlg_get_text_run (priv->dialog_get_text, &message))
          goto CLEAN_UP;
    }

  if (!tp_cli_channel_interface_group_run_remove_members (priv->channel,
        -1, members, message, &error, NULL))
    {
      g_printerr ("RemoveMembers(): %s\n", error->message);
      g_error_free (error);
    }

CLEAN_UP:
  g_free (message);
  g_array_free (members, TRUE);
}

static void
_ti_page_channel_group_add_members (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GArray *handles = NULL;
  GError *error = NULL;
  gchar *message = NULL;
  GArray *selected_handles = NULL;

  selected_handles = ti_get_selected_elements (priv->members_selection,
      TI_COLUMN_MEMBERS_LIST_HANDLE, G_TYPE_UINT);

  if (!ti_dlg_add_members_run (priv->dialog_add_members, selected_handles,
        &handles, &message))
    goto CLEAN_UP;

  if (!tp_cli_channel_interface_group_run_add_members (priv->channel,
        -1, handles, message, &error, NULL))
    {
      g_printerr ("AddMembers(): %s\n", error->message);
      g_error_free (error);
    }

CLEAN_UP:
  g_free (message);
  g_array_free (selected_handles, TRUE);

  if (handles != NULL)
    g_array_free (handles, TRUE);
}


static void
_ti_page_channel_group_remove_members (TIPageChannelGroup *self,
                                       const GArray *removed_handles)
{
  guint i;
  guint handle;

  for (i = 0; i < removed_handles->len; i++)
    {
      handle = g_array_index (removed_handles, guint, i);
      _ti_page_channel_group_remove_member (self, handle);
    }
}

static void
_ti_page_channel_group_remove_member (TIPageChannelGroup *self, guint handle)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gboolean ok;
  GtkTreeIter iter;
  gboolean removed = FALSE;
  guint member_handle;

  ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->members_list),
      &iter);

  while (ok && !removed)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->members_list), &iter,
          TI_COLUMN_MEMBERS_LIST_HANDLE, &member_handle,
          -1);

      if (member_handle == handle)
        {
          gtk_list_store_remove (priv->members_list, &iter);
          removed = TRUE;
        }
        else
        {
          ok = gtk_tree_model_iter_next  (GTK_TREE_MODEL (priv->members_list),
              &iter);
        }
    }
}

static void
_ti_page_channel_group_update_button_remove_sensitivity (
    TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gint selection_count;

  selection_count = gtk_tree_selection_count_selected_rows (
      priv->members_selection);

  gtk_widget_set_sensitive (GTK_WIDGET (priv->button_remove_members),
      selection_count > 0);
}

static void
_ti_page_channel_group_log_members_changed (TIPageChannelGroup *self,
                                            const gchar *message,
                                            const GArray *added,
                                            const GArray *removed,
                                            const GArray *local_pending,
                                            const GArray *remote_pending,
                                            guint actor,
                                            guint reason)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GtkTextIter iter;
  GString *text = NULL;
  guint i;
  guint handle;

  text = g_string_new ("=== MembersChanged ===\n");

  g_string_append (text, "  Added:");
  for (i = 0; i < added->len; i++)
    {
      handle = g_array_index (added, guint, i);
      g_string_append_printf (text, " %u", handle);
    }
  g_string_append (text, "\n");

  g_string_append (text, "  Removed:");
  for (i = 0; i < removed->len; i++)
    {
      handle = g_array_index (removed, guint, i);
      g_string_append_printf (text, " %u", handle);
    }
  g_string_append (text, "\n");

  g_string_append (text, "  Local pending:");
  for (i = 0; i < local_pending->len; i++)
    {
      handle = g_array_index (local_pending, guint, i);
      g_string_append_printf (text, " %u", handle);
    }
  g_string_append (text, "\n");

  g_string_append (text, "  Remote pending:");
  for (i = 0; i < remote_pending->len; i++)
    {
      handle = g_array_index (remote_pending, guint, i);
      g_string_append_printf (text, " %u", handle);
    }
  g_string_append (text, "\n");

  g_string_append_printf (text, "  Actor: %u\n", actor);

  g_string_append_printf (text, "  Reason: %u", reason);
  switch (reason)
    {
    case TP_CHANNEL_GROUP_CHANGE_REASON_NONE:
      g_string_append (text, " (None)\n");
      break;

    case TP_CHANNEL_GROUP_CHANGE_REASON_OFFLINE:
      g_string_append (text, " (Offline)\n");
      break;

    case TP_CHANNEL_GROUP_CHANGE_REASON_KICKED:
      g_string_append (text, " (Kicked)\n");
      break;

    case TP_CHANNEL_GROUP_CHANGE_REASON_BUSY:
      g_string_append (text, "(Busy)\n");
      break;

    case TP_CHANNEL_GROUP_CHANGE_REASON_INVITED:
      g_string_append (text, " (Invited)\n");
      break;

    default:
      g_string_append (text, " (Unknown/Invalid)\n");
    }

  /* Space before next entry */
  g_string_append (text, "\n");

  gtk_text_buffer_get_end_iter (priv->textbuffer_event_log, &iter);
  gtk_text_buffer_insert (priv->textbuffer_event_log, &iter, text->str, -1);

  if (text != NULL)
    g_string_free (text, TRUE);
}

static void
_ti_page_channel_group_clear_log (TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GtkTextIter start_iter;
  GtkTextIter end_iter;

  gtk_text_buffer_get_start_iter (priv->textbuffer_event_log, &start_iter);
  gtk_text_buffer_get_end_iter (priv->textbuffer_event_log, &end_iter);

  gtk_text_buffer_delete (priv->textbuffer_event_log, &start_iter, &end_iter);
}


static void
_ti_page_channel_group_refresh_group_flags_checkboxes (
    TIPageChannelGroup *self)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gboolean active;

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_CAN_ADD) != 0;
  gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (priv->checkbutton_can_add),
      active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_CAN_REMOVE) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_can_remove), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_CAN_RESCIND) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_can_rescind), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_ADD) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_message_add), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_REMOVE) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_message_remove), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_ACCEPT) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_message_accept), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_REJECT) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_message_reject), active);

  active = (priv->group_flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_RESCIND) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_message_rescind), active);

  active = (priv->group_flags &
      TP_CHANNEL_GROUP_FLAG_CHANNEL_SPECIFIC_HANDLES) != 0;
  gtk_toggle_button_set_active (
      GTK_TOGGLE_BUTTON (priv->checkbutton_channel_specific_handles), active);
}

static void
_ti_page_channel_group_update_selection_permissions (TIPageChannelGroup *self)
{
#if 0
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);

  if ((priv->group_flags & TI_CHANNEL_GROUP_FLAG_CAN_REMOVE) != 0)
    {
      gtk_tree_selection_set_mode (priv->members_selection,
          GTK_SELECTION_MULTIPLE);
    }
  else
    {
      gtk_tree_selection_set_mode (priv->members_selection,
          GTK_SELECTION_NONE);
    }

  if ((priv->group_flags & TI_CHANNEL_GROUP_FLAG_CAN_RESCIND) != 0)
    {
      gtk_tree_selection_set_mode (priv->remote_pending_selection,
          GTK_SELECTION_MULTIPLE);
    }
  else
    {
      gtk_tree_selection_set_mode (priv->remote_pending_selection,
          GTK_SELECTION_NONE);
    }
#endif
}

static void
_ti_page_channel_group_log_group_flags_changed (TIPageChannelGroup *self,
                                                guint added,
                                                guint removed)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GtkTextIter iter;
  GString *text;

  text = g_string_new ("=== GroupFlagsChanged ===\n");

  g_string_append (text, "  Added:");
  _group_flags_to_string (text, added);
  g_string_append (text, "\n");

  g_string_append (text, "  Removed:");
  _group_flags_to_string (text, removed);
  g_string_append (text, "\n");

  /* Space before the next entry */
  g_string_append (text, "\n");

  gtk_text_buffer_get_end_iter (priv->textbuffer_event_log, &iter);
  gtk_text_buffer_insert (priv->textbuffer_event_log, &iter, text->str, -1);

  if (text != NULL)
      g_string_free (text, TRUE);
}

static void
_group_flags_to_string (GString *text,
                        guint flags)
{
  if ((flags & TP_CHANNEL_GROUP_FLAG_CAN_ADD) != 0)
    g_string_append_printf (text, " Can Add (%u)",
        TP_CHANNEL_GROUP_FLAG_CAN_ADD);

  if ((flags & TP_CHANNEL_GROUP_FLAG_CAN_REMOVE) != 0)
    g_string_append_printf (text, " Can Remove (%u)",
        TP_CHANNEL_GROUP_FLAG_CAN_REMOVE);

  if ((flags & TP_CHANNEL_GROUP_FLAG_CAN_RESCIND) != 0)
    g_string_append_printf (text, " Can Rescind (%u)",
        TP_CHANNEL_GROUP_FLAG_CAN_RESCIND);

  if ((flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_ADD) != 0)
    g_string_append_printf (text, " Message Add (%u)",
        TP_CHANNEL_GROUP_FLAG_MESSAGE_ADD);

  if ((flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_REMOVE) != 0)
    g_string_append_printf (text, " Message Remove (%u)",
        TP_CHANNEL_GROUP_FLAG_MESSAGE_REMOVE);

  if ((flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_ACCEPT) != 0)
    g_string_append_printf (text, " Message Accept (%u)",
        TP_CHANNEL_GROUP_FLAG_MESSAGE_ACCEPT);

  if ((flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_REJECT) != 0)
    g_string_append_printf (text, " Message Reject (%u)",
        TP_CHANNEL_GROUP_FLAG_MESSAGE_REJECT);

  if ((flags & TP_CHANNEL_GROUP_FLAG_MESSAGE_RESCIND) != 0)
    g_string_append_printf (text, " Message Rescind (%u)",
        TP_CHANNEL_GROUP_FLAG_MESSAGE_RESCIND);

  if ((flags & TP_CHANNEL_GROUP_FLAG_CHANNEL_SPECIFIC_HANDLES) != 0)
    g_string_append_printf (text, " Channel Specific Handles (%u)",
        TP_CHANNEL_GROUP_FLAG_CHANNEL_SPECIFIC_HANDLES);
}

static void
_ti_page_channel_group_handle_display_mode_changed (TIPageChannelGroup *self,
                                                    guint handle_display_mode)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  GtkTreeViewColumn *contact_handle_column;
  GtkCellRenderer *renderer;
  GList *renderers_list = NULL;

  g_assert (TI_IS_PAGE_CHANNEL_GROUP (self));

  contact_handle_column = gtk_tree_view_get_column (priv->members_tree_view,
      0);

  renderers_list = gtk_tree_view_column_get_cell_renderers (
      contact_handle_column);
  g_assert (g_list_length (renderers_list) == 1);

  renderer = GTK_CELL_RENDERER (renderers_list->data);

  if (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      gtk_tree_view_column_set_title (contact_handle_column, "Handle");
      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_MEMBERS_LIST_HANDLE,
          NULL);
    }
  else
    {
      g_assert (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_NAME);

      gtk_tree_view_column_set_title (contact_handle_column, "Name");
      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_MEMBERS_LIST_NAME,
          NULL);
    }

  contact_handle_column = gtk_tree_view_get_column (priv->members_tree_view,
      2);

  if (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_HANDLE)
    {
      gtk_tree_view_column_set_title (contact_handle_column, "Owner handle");
      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_MEMBERS_LIST_OWNER,
          NULL);
    }
  else
    {
      g_assert (handle_display_mode == TI_PREFERENCES_HANDLE_DISPLAY_NAME);

      gtk_tree_view_column_set_title (contact_handle_column, "Owner");
      gtk_tree_view_column_set_attributes (contact_handle_column, renderer,
          "text", TI_COLUMN_MEMBERS_LIST_OWNER_NAME,
          NULL);
    }

  g_list_free (renderers_list);
}

static gboolean
_ti_page_channel_get_contact_row (TIPageChannelGroup *self,
                                  guint contact_handle,
                                  GtkTreeIter *iter)
{
  TIPageChannelGroupPrivate *priv = TI_PAGE_CHANNEL_GROUP_GET_PRIVATE (self);
  gboolean ok;
  gboolean found = FALSE;
  guint curr_handle;

  ok = gtk_tree_model_get_iter_first (GTK_TREE_MODEL (priv->members_list),
      iter);

  while (ok && !found)
    {
      gtk_tree_model_get (GTK_TREE_MODEL (priv->members_list), iter,
          TI_COLUMN_MEMBERS_LIST_HANDLE, &curr_handle,
          -1);

      if (curr_handle == contact_handle)
        {
          found = TRUE;
        }
      else
        {
          ok = gtk_tree_model_iter_next (GTK_TREE_MODEL (priv->members_list),
              iter);
        }
  }

  return found;
}
