/*
 * Telepathy Inspector - A Telepathy client which exposes Telepathy interfaces.
 *                       Meant to inspect and/or test connection managers.
 *
 * ti-util.c:
 * General purpose utility functions.
 *
 * Copyright (C) 2008 Collabora Ltd. <http://www.collabora.co.uk/>
 * Copyright (C) 2008 Nokia Corporation
 * Copyright (C) 2006-2007 INdT - Instituto Nokia de Tecnologia
 * Originally by Daniel d'Andrada T. de Carvalho <daniel.carvalho@indt.org.br>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA
 */

#include "util.h"

#include <stdio.h>

#define TI_MAX_NUMBERS 100

/**
 * Parses a string containing a comma-separated uint list.
 * e.g.:  "134,3423,343,343,434"
 */
GArray *
ti_parse_uint_list (const gchar *uint_list_str)
{
  GArray *numbers = NULL;
  gchar **number_str_array = NULL;
  guint number;
  int result;
  gboolean failed = FALSE;
  guint i;

  numbers = g_array_new (FALSE, FALSE, sizeof (guint));

  number_str_array = g_strsplit (uint_list_str, ",", TI_MAX_NUMBERS);

  for (i = 0; number_str_array[i] != NULL && !failed; i++)
    {
      result = sscanf (number_str_array[i], "%u", &number);
      failed = result != 1;

      g_array_append_val (numbers, number);
    }

  g_strfreev (number_str_array);

  if (failed)
    {
      g_array_free (numbers, TRUE);
      numbers = NULL;
    }

  return numbers;
}

static void
_ti_hash_table_to_array_iterator (gpointer key,
                                  gpointer value,
                                  GArray *array)
{
  TIHashEntry hash_entry;

  hash_entry.key = key;
  hash_entry.value = value;

  g_array_append_val (array, hash_entry);
}

GArray *
ti_hash_table_to_array (GHashTable *hash_table)
{
  GArray *array;

  g_assert (hash_table != NULL);

  array = g_array_sized_new (FALSE, FALSE, sizeof (TIHashEntry),
      g_hash_table_size (hash_table));

  g_hash_table_foreach (hash_table,
      (GHFunc) _ti_hash_table_to_array_iterator, array);

  return array;
}

gchar *
ti_value_to_string (const GValue *value)
{
  gchar *string = NULL;
  g_assert (G_IS_VALUE (value));

  switch (G_VALUE_TYPE (value))
    {
    case G_TYPE_UINT:
      string = g_strdup_printf ("%u", g_value_get_uint (value));
      break;

    case G_TYPE_INT:
      string = g_strdup_printf ("%u", g_value_get_int (value));
      break;

    case G_TYPE_STRING:
      string = g_value_dup_string (value);
      break;

    case G_TYPE_BOOLEAN:
      if (g_value_get_boolean (value) == TRUE)
        {
          string = g_strdup ("True");
        }
      else
        {
          string = g_strdup ("False");
        }

      break;

    default:
      g_critical ("ti_value_to_string: Unknown/unhandled GValue type.");
      string = g_strdup ("ti_value_to_string: Unknown/unhandled GValue type.");
    }

  return string;
}

void
ti_string_to_value (GValue *value,
                    const gchar *dbus_sig,
                    const gchar *value_str)
{
  guint value_uint;
  guint value_int;

  if (g_str_equal (dbus_sig, "s"))
    {
      g_value_init (value, G_TYPE_STRING);
      g_value_set_string (value, value_str);
    }
  else if (g_str_equal (dbus_sig, "u") || g_str_equal (dbus_sig, "q"))
    {
      g_value_init (value, G_TYPE_UINT);
      sscanf (value_str, "%u", &value_uint);
      g_value_set_uint (value, value_uint);
    }
  else if (g_str_equal (dbus_sig, "i") || g_str_equal (dbus_sig, "n"))
    {
      g_value_init (value, G_TYPE_INT);
      sscanf (value_str, "%d", &value_int);
      g_value_set_int (value, value_int);
    }
  else if (g_str_equal (dbus_sig, "b"))
    {
      gchar *value_str_low = NULL;

      g_value_init (value, G_TYPE_BOOLEAN);

      value_str_low = g_ascii_strdown (value_str, -1);

      if (g_str_equal (value_str, "0"))
        {
          g_value_set_boolean (value, FALSE);
        }
      else if (g_str_equal (value_str_low, "true") ||
               g_str_equal (value_str, "yes"))
        {
          g_value_set_boolean (value, TRUE);
        }
      else if (g_str_equal (value_str_low, "false") ||
               g_str_equal (value_str, "no"))
        {
          g_value_set_boolean (value, FALSE);
        }
      else
        {
          /* Assumed to be a nonzero number */
          g_value_set_boolean (value, TRUE);
        }

      g_free (value_str_low);
  }
}

void
ti_value_destroy (gpointer data)
{
  GValue *value = (GValue *) data;

  g_assert (value != NULL);
  g_assert (G_IS_VALUE (value));

  g_value_unset (value);
  g_free (value);
}

void
ti_remove_selected_elements (GtkTreeSelection *tree_selection)
{
  GList *selected_rows_list = NULL;
  GtkTreeModel *tree_model = NULL;
  GList *list_item = NULL;
  GtkTreePath *tree_path = NULL;
  GPtrArray *row_references_array = g_ptr_array_new ();
  GtkTreeRowReference *row_reference;
  guint i;
  GtkTreeIter iter;

  g_assert (tree_selection != NULL);
  g_assert (GTK_IS_TREE_SELECTION (tree_selection));

  selected_rows_list = gtk_tree_selection_get_selected_rows (tree_selection,
      &tree_model);

  list_item = selected_rows_list;
  while (list_item != NULL)
    {
      tree_path = (GtkTreePath *) list_item->data;

      row_reference = gtk_tree_row_reference_new (tree_model, tree_path);
      g_ptr_array_add (row_references_array, row_reference);

      list_item = list_item->next;
    }

  for (i = 0; i < row_references_array->len; i++)
    {
      row_reference = (GtkTreeRowReference *)
          g_ptr_array_index (row_references_array, i);

      tree_path = gtk_tree_row_reference_get_path (row_reference);

      gtk_tree_model_get_iter (tree_model, &iter, tree_path);

      if (GTK_IS_LIST_STORE (tree_model))
        {
          gtk_list_store_remove (GTK_LIST_STORE (tree_model), &iter);
        }
      else
        {
          g_assert (GTK_IS_TREE_STORE (tree_model));
          gtk_tree_store_remove (GTK_TREE_STORE (tree_model), &iter);
        }
    }

  if (selected_rows_list != NULL)
    {
      g_list_foreach (selected_rows_list, (GFunc) gtk_tree_path_free, NULL);
      g_list_free (selected_rows_list);
    }

  if (row_references_array != NULL)
    g_ptr_array_free (row_references_array, TRUE);
}

GArray *
ti_get_selected_elements (GtkTreeSelection *tree_selection,
                          guint column,
                          GType type)
{
  GArray *elements_array = NULL;
  GList *selected_rows_list = NULL;
  GtkTreeModel *tree_model = NULL;
  GtkTreePath *tree_path = NULL;
  GList *list_item = NULL;
  GtkTreeIter iter;
  guint elem_uint;
  gint elem_int;
  gchar *elem_string;

  g_assert (tree_selection != NULL);
  g_assert (GTK_IS_TREE_SELECTION (tree_selection));

  switch (type)
    {
    case G_TYPE_STRING:
      elements_array = g_array_new (FALSE, FALSE, sizeof (gchar *));
      break;

    case G_TYPE_UINT:
      elements_array = g_array_new (FALSE, FALSE, sizeof (guint));
      break;

    case G_TYPE_INT:
      elements_array = g_array_new (FALSE, FALSE, sizeof (gint));
      break;

    default:
      /* TODO: implement the missing type, if that's the case */
      g_assert_not_reached ();
      return NULL;
    }

  selected_rows_list = gtk_tree_selection_get_selected_rows (tree_selection,
      &tree_model);

  list_item = selected_rows_list;

  while (list_item != NULL)
    {
      tree_path = (GtkTreePath *) list_item->data;

      gtk_tree_model_get_iter (tree_model, &iter, tree_path);

      switch (type)
        {
        case G_TYPE_STRING:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_string,
                              -1);
          g_array_append_val (elements_array, elem_string);
          break;

        case G_TYPE_UINT:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_uint,
                              -1);
          g_array_append_val (elements_array, elem_uint);
          break;

        case G_TYPE_INT:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_int,
                              -1);
          g_array_append_val (elements_array, elem_int);
          break;
        }

      list_item = list_item->next;
    }

  if (selected_rows_list != NULL)
    {
      g_list_foreach (selected_rows_list, (GFunc) gtk_tree_path_free, NULL);
      g_list_free (selected_rows_list);
    }

  return elements_array;
}

GArray *
ti_get_tree_model_elements (GtkTreeModel *tree_model,
                            guint column,
                            GType type)
{
  GtkTreeIter iter;
  gboolean ok;
  guint elem_uint;
  gint elem_int;
  gchar *elem_string;
  GArray *elements_array = NULL;

  g_assert (tree_model != NULL);
  g_assert (GTK_IS_TREE_MODEL (tree_model));

  switch (type)
    {
    case G_TYPE_STRING:
      elements_array = g_array_new (FALSE, FALSE, sizeof (gchar*));
      break;

    case G_TYPE_UINT:
      elements_array = g_array_new (FALSE, FALSE, sizeof (guint));
      break;

    case G_TYPE_INT:
      elements_array = g_array_new (FALSE, FALSE, sizeof (gint));
      break;

    default:
      /* TODO: implement the missing type, if that's the case. */
      g_assert_not_reached ();
      return NULL;
    }

  ok = gtk_tree_model_get_iter_first (tree_model, &iter);

  while (ok)
    {
      switch (type)
        {
        case G_TYPE_STRING:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_string,
                              -1);
          g_array_append_val (elements_array, elem_string);
          break;

        case G_TYPE_UINT:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_uint,
                              -1);
          g_array_append_val (elements_array, elem_uint);
          break;

        case G_TYPE_INT:
          gtk_tree_model_get (tree_model, &iter,
                              column, &elem_int,
                              -1);
          g_array_append_val (elements_array, elem_int);
          break;
        }

        ok = gtk_tree_model_iter_next (tree_model, &iter);
    }

  return elements_array;
}

static gchar *ti_argv0_dir = NULL;
static gchar *ti_glade_path = NULL;

void
ti_set_executable (const gchar *argv0)
{
  gchar *found = g_find_program_in_path (argv0);

  if (found != NULL)
    {
      if (!g_path_is_absolute (found))
        {
          gchar *cwd = g_get_current_dir ();
          gchar *abs = g_build_filename (cwd, found, NULL);

          g_free (cwd);
          g_free (found);
          found = abs;
        }

      g_free (ti_argv0_dir);
      ti_argv0_dir = g_path_get_dirname (found);
    }

  g_free (found);
}

/* Like Python's os.pardir.
 * FIXME: this is OS-specific, e.g. on Mac OS 9 this should be "::" - but I
 * doubt this program will work on platforms that un-Unixy anyway */
#define PARDIR ".."

gchar *
ti_get_glade_path (const gchar *filename)
{
  const gchar *dir;
  const gchar * const *dirs;
  gchar *abs_path;

  if (ti_argv0_dir != NULL && ti_glade_path == NULL)
    {
      gchar *candidate;

      g_assert (g_path_is_absolute (ti_argv0_dir));

      if (g_str_has_suffix (ti_argv0_dir, "/.libs"))
        {
          /* built against an uninstalled shared library with libtool */
          candidate = g_build_filename (ti_argv0_dir, PARDIR, PARDIR, "data",
              NULL);
        }
      else
        {
          candidate = g_build_filename (ti_argv0_dir, PARDIR, "data", NULL);
        }

      abs_path = g_build_filename (candidate, filename, NULL);

      if (g_file_test (abs_path, G_FILE_TEST_EXISTS))
        {
          g_message ("Using %s as Glade path", candidate);
          ti_glade_path = candidate;
          return abs_path;
        }

      g_free (candidate);
      g_free (abs_path);
    }

  if (ti_glade_path != NULL)
    return g_build_filename (ti_glade_path, filename, NULL);

  dir = g_get_user_data_dir ();
  abs_path = g_build_filename (dir, "telepathy-inspector", filename, NULL);

  if (g_file_test (abs_path, G_FILE_TEST_EXISTS))
    {
      ti_glade_path = g_build_filename (dir, "telepathy-inspector", NULL);
      return abs_path;
    }

  g_free (abs_path);

  for (dirs = g_get_system_data_dirs (); *dirs != NULL; dirs++)
    {
      dir = *dirs;
      abs_path = g_build_filename (dir, "telepathy-inspector", filename, NULL);

      if (g_file_test (abs_path, G_FILE_TEST_EXISTS))
        {
          ti_glade_path = g_build_filename (dir, "telepathy-inspector", NULL);
          return abs_path;
        }

      g_free (abs_path);
    }

  /* eek */
  g_error ("Unable to find telepathy-inspector Glade XML files");
  g_assert_not_reached ();
  return NULL;
}

void
ti_ok_button_clicked (GtkButton *button,
                      GtkDialog *dialog)
{
  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}

void
ti_cancel_button_clicked (GtkButton *button,
                          GtkDialog *dialog)
{
  gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_CANCEL);
}
