/* ====================================================================
 * Copyright (c) 2009  Martin Hauner
 *                     http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "ProjectWriter.h"


ProjectWriter::ProjectWriter( ConfigData* cfg, const Project* prj )
: _cfg(cfg), _prj(prj)
{
  _keyRoot = QString("project.%1").arg( _prj->getIndex() );
}

ProjectWriter::~ProjectWriter()
{
}

void ProjectWriter::write()
{
  writeRoot();
  writeTrunkItem();
  writeBranchesItem();
  writeTagsItem();
  writeWorkingCopyItems();
  writeRepositoryItems();

  _cfg->setValues( sc::String(_keyRoot.utf8()), _values ); 
}

void ProjectWriter::writeRoot()
{
  addValue( _keyRoot + ".name", _prj->getName() );
  addValue( _keyRoot + ".guid", _prj->getId().toString() );
  addValue( _keyRoot + ".sortpos", _prj->getSortPos() );
}

void ProjectWriter::writeTrunkItem()
{
  writeRepositoryItem( _keyRoot + ".trunk", _prj->getTrunkItem() );
}

void ProjectWriter::writeBranchesItem()
{
  writeRepositoryItem( _keyRoot + ".branches", _prj->getBranchesItem()
    );
}

void ProjectWriter::writeTagsItem()
{
  writeRepositoryItem( _keyRoot + ".tags", _prj->getTagsItem() );
}

void ProjectWriter::writeWorkingCopyItems()
{
  Project::Items items;
  _prj->getItems(items);

  long wcCur = 0;
  long cnt = 0;
  for( Project::Items::iterator it = items.begin(); it != items.end();
    it++ )
  {
    const Project::Item& item = *it;

    if( ! item.isWorkingCopy() )
      continue;

    if( item.getId() == _prj->getCurWorkingCopyId() )
      wcCur = cnt;

    QString key = _keyRoot + QString(".wc.%1").arg(cnt);
    writeWorkingCopyItem(key,item);
    writeWorkingCopySubItems(key,item);

    cnt++;
  }

  addValue( _keyRoot + ".wc.current", wcCur );
}

void ProjectWriter::writeWorkingCopySubItems( const QString& key, const
  Project::Item& item )
{
  Project::SubItems items;
  item.getSubItems(items);

  long cnt = 0;
  for( Project::SubItems::iterator it = items.begin(); it !=
    items.end(); it++ )
  {
    const Project::SubItem& item = *it;

    QString subkey = key + QString(".subitem.%1").arg(cnt);
    writeWorkingCopySubItem(subkey,item);

    cnt++;
  }
}

void ProjectWriter::writeRepositoryItems()
{
  Project::Items items;
  _prj->getItems(items);

  long cnt = 0;
  for( Project::Items::iterator it = items.begin(); it != items.end();
    it++ )
  {
    const Project::Item& item = *it;

    if( ! item.isRepository() )
      continue;

    QString key = _keyRoot + QString(".rp.%1").arg(cnt);
    writeRepositoryItem(key,item);
    cnt++;
  }
}

void ProjectWriter::writeRepositoryItem( const QString& key, const
  Project::Item& item )
{
  addValue( key + ".name", item.getName() );
  addValue( key + ".url", item.getSource() );
  addValue( key + ".sortpos", item.getSortPos() );
  addValue( key + ".recursive", item.getRecursive() );
  addValue( key + ".revision", item.getRevision()->toString() );
  addValue( key + ".viewheader", item.getViewHeader() );
}

void ProjectWriter::writeWorkingCopyItem( const QString& key, const
  Project::Item& item )
{
  addValue( key + ".name", item.getName() );
  addValue( key + ".path", item.getSource() );
  addValue( key + ".sortpos", item.getSortPos() );
  addValue( key + ".recursive", item.getRecursive() );
  addValue( key + ".update", item.getUpdate() );
  addValue( key + ".autoupdate", item.getAutoUpdate() );
  addValue( key + ".autorefresh", item.getAutoRefresh() );
  addValue( key + ".viewheader", item.getViewHeader() );
}

void ProjectWriter::writeWorkingCopySubItem( const QString& key, const
  Project::SubItem& item )
{
  addValue( key + ".path", item.getPath() );

  if( item.getType() == Project::SubItem_Flat )
    addValue( key + ".type", sc::String("flat") );
  else
    addValue( key + ".type", sc::String("tree") );
}

void ProjectWriter::addValue( const QString& key, const sc::String&
  value )
{
  QString fullkey = _keyRoot + key;
  _values.push_back( ConfigValue(sc::String(key.utf8()), value) );
}

void ProjectWriter::addValue( const QString& key, sc::String& value )
{
  QString fullkey = _keyRoot + key;
  _values.push_back( ConfigValue(sc::String(key.utf8()), value) );
}

void ProjectWriter::addValue( const QString& key, long value )
{
  QString fullkey = _keyRoot + key;
  _values.push_back( ConfigValue(sc::String(key.utf8()), value) );
}

void ProjectWriter::addValue( const QString& key, bool value )
{
  QString fullkey = _keyRoot + key;
  _values.push_back( ConfigValue(sc::String(key.utf8()), value) );
}
