/*
    Application.vala
    Copyright (C) 2010 Maia Kozheva <sikon@ubuntu.com>

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

using Gtk;
using Unique;
using Steadyflow.Core;
using Steadyflow.UI;

namespace Steadyflow {

private static const string COMMAND_LINE_HELP = """
Syntax: steadyflow [command [args...] [-option1=value1] [-option2=value2]...]

Commands:
  steadyflow add [url]  - Add a file for download. If url is omitted, populates
                          the add dialog with the clipboard content if it is a
                          valid URL.
  steadyflow show       - Show the main window
  steadyflow hide       - Hide the main window
  steadyflow help       - Show this help screen
  steadyflow version    - Show version information
""";

public class Application : Unique.App {
	private NotificationController notifications;
	private OptionParser parser;
	
	private enum CommandId {
		ADD = 1,
		SHOW,
		HIDE
	}
	
	private enum ShowMainWindow {
		DEFAULT,
		SHOW,
		HIDE
	}

	public Application (string[] args) {
		GLib.Object (name: "net.launchpad.steadyflow");
		
		string domain = "steadyflow";
		Intl.textdomain (domain);
		Intl.bindtextdomain (domain, Util.is_local_install () ? "bin/locale" : AppConfig.DATA_DIR + "/locale");
		Intl.bind_textdomain_codeset (domain, "UTF-8");
		
		Notify.init (domain);

		parser = new OptionParser (args);
		parser.unhandled_command.connect (show_help);
		parser.register_command ("help", (args, kwargs) => { show_help (); });
		parser.register_command ("version", (args, kwargs) => { show_version (); });
		parser.register_command ("add", (args, kwargs) => {
			if (args.length > 1) {
				show_help ();
			} else if (!is_running) {
				run_default (args.length == 1 ? args[0] : "", ShowMainWindow.DEFAULT);
			} else {
				MessageData msg = new MessageData();
				string data = args.length == 1 ? args[0] : "";
				msg.set_text(data, data.length);
				send_message (CommandId.ADD, msg);
			}
		});
		parser.register_command ("show", (args, kwargs) => {
			if (!is_running) {
				run_default (null, ShowMainWindow.SHOW);
			} else {
				send_message (CommandId.SHOW, null);
			}
		});
		parser.register_command ("hide", (args, kwargs) => {
			if (!is_running) {
				run_default (null, ShowMainWindow.HIDE);
			} else {
				send_message (CommandId.HIDE, null);
			}
		});
		parser.register_command ("@default", (args, kwargs) => {
			run_default (null, ShowMainWindow.DEFAULT);
		});
		
		add_command ("add", CommandId.ADD);
		add_command ("show", CommandId.SHOW);
		add_command ("hide", CommandId.HIDE);
	}
	
	private Response on_unique_callback (int command, MessageData message_data, uint time) {
		switch ((CommandId) command) {
		case CommandId.ADD:
			add_file (message_data.get_text());
			return Response.OK;
		case CommandId.SHOW:
			set_visible (true);
			return Response.OK;
		case CommandId.HIDE:
			set_visible (false);
			return Response.OK;
		default:
			return Response.INVALID;
		}
	}
	
	private void run () {
		parser.run ();
	}
	
	private void show_version () {
		stdout.printf("Steadyflow version %s\n", AppConfig.APP_VERSION);
	}
	
	private void show_help () {
		show_version ();
		stdout.printf("%s", COMMAND_LINE_HELP);
	}
	
	private void run_default (string? add_url, ShowMainWindow show) {
		if (is_running) {
			stderr.printf ("%s\n", _("Another instance of Steadyflow is already running."));
			Process.exit (1);
		}
		
		ulong sigid_add_file = -1;
		ulong sigid_set_visible = -1;
		
		try {
			Services.init ();
			IconTheme.get_default ().append_search_path (Util.resolve_path ("img"));

			MainWindow main_wnd = new MainWindow ();
			if (!Services.settings.get_boolean("show-indicator")) {
				main_wnd.show ();
			} else {
				main_wnd.realize (); // To correctly initialize size even if initially hidden
				bool start_minimized = Services.settings.get_boolean ("start-minimized");
				
				if (show == ShowMainWindow.SHOW || (show == ShowMainWindow.DEFAULT && !start_minimized)) {
					main_wnd.show ();
				} else {
					// We need to clear startup notify status manually if we start hidden
					Gdk.notify_startup_complete ();
				}
			}
			
			if (add_url != null) {
				main_wnd.add_download (add_url);
			}
			
			sigid_add_file = add_file.connect ((url) => { main_wnd.add_download (url); });
			sigid_set_visible = set_visible.connect ((visible) => { main_wnd.set_visible (visible); });
			message_received.connect (on_unique_callback);
			notifications = new NotificationController ();
			dbus_register ();
			Gtk.main ();
		} catch (Error e) {
			Util.fatal_error (e);
		} finally {
			if (sigid_add_file != -1)
				disconnect (sigid_add_file);
			
			if (sigid_set_visible != -1)
				disconnect (sigid_set_visible);
			
			Services.done ();
		}
	}
	
	private void dbus_register () {
		Bus.own_name (BusType.SESSION, "net.launchpad.steadyflow.App", BusNameOwnerFlags.NONE,
					  on_bus_acquired, null, null);
	}
	
	private void on_bus_acquired (DBusConnection conn, string name) {
		try {
			conn.register_object ("/net/launchpad/steadyflow/app", new AppService(this));
		} catch (IOError e) {
			stderr.printf ("%s\n",
						   _("Cannot start a D-Bus service. Another instance of Steadyflow may be already running."));
			Process.exit (1);
		}
	}

	public static int main (string[] args) {
		if (Util.is_local_install ()) {
			// Hack to make GSettings work locally, there should really be a better way...
			string xdg_data_dirs = Environment.get_variable ("XDG_DATA_DIRS");
		
			if (xdg_data_dirs == null)
				xdg_data_dirs = "/usr/local/share/:/usr/share/";
	
			Environment.set_variable ("XDG_DATA_DIRS", "bin/:" + xdg_data_dirs, true);
		}
		
		Gtk.init (ref args);
		Application app = new Application (args);
		app.run ();
		return 0;
	}
	
	internal signal void add_file (string url);
	internal signal void set_visible (bool visible);
}

[DBus (name = "net.launchpad.steadyflow.App")]
protected class AppService: GLib.Object {
	private Application app;
	
	public AppService (Application app) {
		this.app = app;
	}
	
	// D-Bus public API
	
	public void add_file (string url) {
		app.add_file (url);
	}
	
	public void set_visible (bool visible) {
		app.set_visible (visible);
	}
}

}
