/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/
#include <vdb/extern.h>

#include <sra/sradb.h>
#include <vdb/xform.h>
#include <vdb/table.h>
#include <klib/data-buffer.h>
#include <klib/text.h>
#include <klib/rc.h>
#include <kdb/meta.h>
#include <sysalloc.h>

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <assert.h>

const int signature_no_spotgroup = 3;
const int signature_spotgroup = 4;


typedef struct sra_meta_stats_data_struct {
    SRATableStats* stats;
    KMDataNode* run;
    KMDataNode* spot_groups;
} sra_meta_stats_data;


static
void CC sra_meta_stats_whack( void* self )
{
    if( self != NULL ) {
        sra_meta_stats_data* data = self;
        KMDataNodeRelease(data->run);
        KMDataNodeRelease(data->spot_groups);
        free(data);
    }
}

static
rc_t sra_meta_stats_read(KMDataNode* node, void* value, size_t sz)
{
    rc_t rc = 0;
    size_t num_read, remaining;

    if( (rc = KMDataNodeRead(node, 0, value, sz, &num_read, &remaining)) == 0 &&
        num_read == sz && remaining == 0 ) {
    } else if( rc == 0 && num_read == 0 && remaining == 0 ) {
        memset(value, 0, sz);
    } else {
        rc = rc ? rc : RC(rcVDB, rcFunction, rcUpdating, rcMetadata, rcCorrupt);
    }
    return rc;
}

static
rc_t sra_meta_stats_write(KMDataNode* root, SRATableStats* stats, const int64_t spot_id,
                          const uint32_t spot_len, const uint32_t bio_spot_len)
{
    rc_t rc = 0;
    KMDataNode* node;
    uint64_t u64;
    int64_t i64;

    if( (rc = KMDataNodeOpenNodeUpdate(root, &node, "SPOT_MIN")) == 0 ) {
        if( (rc = sra_meta_stats_read(node, &i64, sizeof(i64))) == 0 ) {
            if( i64 == 0 || i64 > spot_id ) {
                rc = KMDataNodeWriteB64(node, &spot_id);
                if( stats != NULL ) {
                    stats->min_spot_id = i64;
                }
            }
        }
        KMDataNodeRelease(node);
    }
    if( (rc = KMDataNodeOpenNodeUpdate(root, &node, "SPOT_MAX")) == 0 ) {
        if( (rc = sra_meta_stats_read(node, &i64, sizeof(i64))) == 0 ) {
            if( i64 == 0 || i64 < spot_id ) {
                rc = KMDataNodeWriteB64(node, &spot_id);
                if( stats != NULL ) {
                    stats->max_spot_id = i64;
                }
            }
        }
        KMDataNodeRelease(node);
    }
    if( (rc = KMDataNodeOpenNodeUpdate(root, &node, "SPOT_COUNT")) == 0 ) {
        if( (rc = sra_meta_stats_read(node, &u64, sizeof(u64))) == 0 ) {
            if( u64 + 1 < u64 ) {
                rc = RC(rcVDB, rcFunction, rcUpdating, rcMetadata, rcOutofrange);
            } else {
                ++u64;
                rc = KMDataNodeWriteB64(node, &u64);
                if( stats != NULL ) {
                    stats->spot_count = u64;
                }
            }
        }
        KMDataNodeRelease(node);
    }
    if( (rc = KMDataNodeOpenNodeUpdate(root, &node, "BASE_COUNT")) == 0 ) {
        if( (rc = sra_meta_stats_read(node, &u64, sizeof(u64))) == 0 ) {
            if( u64 + 1 < u64 ) {
                rc = RC(rcVDB, rcFunction, rcUpdating, rcMetadata, rcOutofrange);
            } else {
                u64 += spot_len;
                rc = KMDataNodeWriteB64(node, &u64);
                if( stats != NULL ) {
                    stats->base_count = u64;
                }
            }
        }
        KMDataNodeRelease(node);
    }
    if( (rc = KMDataNodeOpenNodeUpdate(root, &node, "BIO_BASE_COUNT")) == 0 ) {
        if( (rc = sra_meta_stats_read(node, &u64, sizeof(u64))) == 0 ) {
            if( u64 + bio_spot_len < u64 ) {
                rc = RC(rcVDB, rcFunction, rcUpdating, rcMetadata, rcOutofrange);
            } else {
                u64 += bio_spot_len;
                rc = KMDataNodeWriteB64(node, &u64);
            }
        }
        KMDataNodeRelease(node);
    }
    return rc;
}

static
rc_t CC sra_meta_stats( void *self, const VXformInfo *info, int64_t row_id,
                             VRowResult *rslt, uint32_t argc, const VRowData argv [] )
{
    rc_t rc = 0;
    sra_meta_stats_data* data = self;
    uint32_t i, bio_spot_len = 0, spot_len = argv[0].u.data.elem_count;
    uint8_t nreads = argv[1].u.data.elem_count;
    const uint32_t* read_len = &((uint32_t*)argv[1].u.data.base)[argv[1].u.data.first_elem];
    const uint8_t* read_type = &((uint8_t*)argv[2].u.data.base)[argv[2].u.data.first_elem];

    assert(data != NULL);
    assert(argc >= signature_no_spotgroup && argc <= signature_spotgroup);
    assert(nreads == argv[2].u.data.elem_count);

    for(i = 0; i < nreads; i++) {
        if( ( read_type[i] & SRA_READ_TYPE_BIOLOGICAL ) != 0 ) {
            bio_spot_len += read_len[i];
        }
    }

    if( (rc = sra_meta_stats_write(data->run, data->stats, row_id, spot_len, bio_spot_len)) == 0 &&
        argc == signature_spotgroup ) {
        KMDataNode* node;
        const char* grp = &((const char*)argv[3].u.data.base)[argv[3].u.data.first_elem];
        int len = argv[3].u.data.elem_count;
        if( len == 0 || grp == NULL || grp[0] == '\0' ) {
            grp = "default";
            len = 7;
        }
        if( (rc = KMDataNodeOpenNodeUpdate(data->spot_groups, &node, "%.*s", len, grp)) == 0 ) {
            rc = sra_meta_stats_write(node, NULL, row_id, spot_len, bio_spot_len);
            KMDataNodeRelease(node);
        }
    }
    return rc;
}

VTRANSFACT_IMPL ( NCBI_SRA_stats_trigger, 1, 0, 0 ) ( const void *self, const VXfactInfo *info, VFuncDesc *rslt,
                                                      const VFactoryParams *cp, const VFunctionParams *dp )
{
    rc_t rc = 0;
    SRATableStats* stats = NULL;

    if( dp->argc < signature_no_spotgroup || dp->argc > signature_spotgroup ) {
        rc = RC(rcVDB, rcFunction, rcConstructing, rcParam, rcInvalid);
    } else if( (rc = VTableGetUserData(info->tbl, (void**)&stats)) != 0) {
/*    } else if( cp->argc < 1 || cp->argc > 2 ) {
        rc = RC(rcVDB, rcFunction, rcConstructing, rcParam, rcInvalid);*/
    } else {
        sra_meta_stats_data* data;
        if( (data = calloc(1, sizeof(sra_meta_stats_data))) == NULL ) {
            rc = RC(rcVDB, rcFunction, rcConstructing, rcMemory, rcExhausted);
        } else {
            KMetadata *meta;
            if( (rc = VTableOpenMetadataUpdate((VTable*)info->tbl, &meta)) == 0 ) {
                data->stats = stats;
                rc = KMetadataOpenNodeUpdate(meta, &data->run, "STATS/TABLE");
                if( rc == 0 && dp->argc == signature_spotgroup ) {
                    rc = KMetadataOpenNodeUpdate(meta, &data->spot_groups, "STATS/SPOT_GROUP");
                }
                /*rc = KMetadataOpenNodeUpdate(meta, &data->run, "%.*s", cp->argv[0].count, cp->argv[0].data.ascii);
                if( rc == 0 && dp->argc == signature_spotgroup ) {
                    rc = KMetadataOpenNodeUpdate(meta, &data->spot_groups, "%.*s", cp->argv[1].count, cp->argv[1].data.ascii);
                }*/
                KMetadataRelease(meta);
            }
            if( rc == 0 ) {
                rslt->self = data;
                rslt->whack = sra_meta_stats_whack;
                rslt->variant = vftNonDetRow;
                rslt->u.rf = sra_meta_stats;
            } else {
                sra_meta_stats_whack(data);
            }
        }
    }
    return rc;
}
