/*===========================================================================
*
*                            PUBLIC DOMAIN NOTICE
*               National Center for Biotechnology Information
*
*  This software/database is a "United States Government Work" under the
*  terms of the United States Copyright Act.  It was written as part of
*  the author's official duties as a United States Government employee and
*  thus cannot be copyrighted.  This software/database is freely available
*  to the public for use. The National Library of Medicine and the U.S.
*  Government have not placed any restriction on its use or reproduction.
*
*  Although all reasonable efforts have been taken to ensure the accuracy
*  and reliability of the software and data, the NLM and the U.S.
*  Government do not and cannot warrant the performance or results that
*  may be obtained by using this software or data. The NLM and the U.S.
*  Government disclaim all warranties, express or implied, including
*  warranties of performance, merchantability or fitness for any particular
*  purpose.
*
*  Please cite the author in any work or product based on this material.
*
* ===========================================================================
*
*/

#ifndef _h_klib_writer_
#define _h_klib_writer_

#ifndef _h_klib_extern_
#include <klib/extern.h>
#endif

#ifndef _h_klib_defs_
#include <klib/defs.h>
#endif

#ifndef _h_klib_logfmt_
#include <klib/logfmt.h>
#endif

#include <stdarg.h>

#ifdef __cplusplus
extern "C" {
#endif


/* ----------
 * The writer module (which contains the log, status, out, and debug sub-modules)
 * is an ecapsulation and virtualization of standard output.
 *
 *  this logging API - like most others - attempts to remain isolated
 *  from any actual logging implementation, such that output may be
 *  routed in several ways.
 *
 *  the default handler
 *
 *  the path from unstructured text to formatted XML is difficult in that
 *  it requires parsing, and is therefore error prone.
 *
 *  the path from structured reports to XML as well as less or unstructured
 *  text is simple to implement, but generally more of a burden to use.
 *
 *  the challenge is therefore to present an API that makes structured
 *  logging less painful than it might be.
 *
 *
 * There are two parts with an unfortunate legacy quirky combination.
 *  There is a formater that will be shared by all sub-modules.
 *  There is the actual stream writer.
 *
 * Each submodule will have one or more calls that call the formatter and writer functions.
 */
KLIB_EXTERN rc_t CC KWrtInit ( const char * appname, uint32_t vers );

/* -----
 * The writer module core file/stream output function:
 *
 * The function's prototype matches as much as possible the KFS file writer prototype.
 *
 * If the self structure contained a KFile * and a position it could use the KFS file write
 */

typedef rc_t ( CC * KWrtWriter ) ( void * self, const char * buffer, size_t bufsize, size_t * num_writ );

/* ----
 * a Writer handler combines the function and its self object
 */
typedef struct KWrtHandler
{
    KWrtWriter writer;
    void *     data;    /* the actual type depends upon the writer function */
} KWrtHandler;


/*--------------------------------------------------------------------------
 * nvp - name/value pair
 */
typedef struct wrt_nvp_t_struct {
    const char *name;
    const char *value;
} wrt_nvp_t;

KLIB_EXTERN void CC wrt_nvp_sort(size_t argc, wrt_nvp_t argv[]);
KLIB_EXTERN const wrt_nvp_t* CC wrt_nvp_find( size_t argc, const wrt_nvp_t argv[], const char* key );
KLIB_EXTERN const char* CC wrt_nvp_find_value( size_t argc, const wrt_nvp_t argv[], const char* key );

/* -----
 * The formatter module core file/stream output function:
 *
 */

typedef rc_t ( CC * KFmtWriter ) ( void* self, KWrtHandler* writer,
                                   size_t argc, const wrt_nvp_t args[],
                                   size_t envc, const wrt_nvp_t envs[]);
/* ----
 * a formatter handler combines the function and custom data
 */
typedef struct KFmtHandler
{
    KFmtWriter formatter;
    void* data;    /* the actual type depends upon the writer function */
} KFmtHandler;


/*
 * For an extensible printf we've added a formatter subcall ability
 *
 * "%()" will call a function provided externally.  This allows
 * adding formats without rewriting the knprintf itself.
 *
 * The characters between the '%' and the '(' are used as normal "%s" c-string modifiers.
 *    This is the full standard printf width and size modifiers for string including
 *    '*'.
 * The characters between the '(' and the ')' are passed to the function. 
 *    This can be anything desired by the function writer.  An '*' is not interpreted as
 *    a call for another parameter at this time.
 *
 * The characters after the ) determine the type passed to the function.
 *    That is what is popped off the stack/arg_list as [v]knprintf is called.
 *    supported are:
 *        p: a pointer is retrieved
 *        i,o,u,x,X: an integer (32 bit)
 *        ji,jo,ju,jx,jX,li,lo,lu,lx,lX: a long integer (64 bit)
 *        e,f,g,E,F,G: a double
 *        Le,Lf,Lg,LE,LF,LG: a long double
 *    This character is not passed to the called function and no differention is
 *    made amongst the integer types nor the float types.  Only the number of bits
 *    matters really.
 */

typedef size_t ( CC * pfunc_t ) ( char * buffer, size_t bufsize, const char * fmt, ... );


KLIB_EXTERN size_t CC vknprintf ( char * buf, size_t bufsize, const char * str, va_list ap );
KLIB_EXTERN size_t CC knprintf ( char * buf, size_t bufsize, const char * fmt, ... );

/* RCExplain
 *  explains rc code in "English'ish" language
 */
KLIB_EXTERN rc_t CC RCExplain ( rc_t rc, char *buffer, size_t bsize, size_t *num_writ );

KLIB_EXTERN rc_t CC LogInsertSpace(const char *spacer, char *buffer, size_t bsize, size_t *num_writ);

KLIB_EXTERN rc_t CC LogTimestamp ( char *buffer, size_t bsize, size_t *num_writ );
KLIB_EXTERN rc_t CC LogSimpleTimestamp ( char *buffer, size_t bsize, size_t *num_writ );
KLIB_EXTERN rc_t CC LogPID ( char *buffer, size_t bsize, size_t *num_writ );
KLIB_EXTERN rc_t CC LogAppName ( char *buffer, size_t bsize, size_t *num_writ );
KLIB_EXTERN rc_t CC LogAppVersion ( char *buffer, size_t bsize, size_t *num_writ );

/* FlushLine
 *  flushes a buffer through output handler
 */
KLIB_EXTERN rc_t CC LogFlush(KWrtHandler * handler, const char *buffer, const size_t bsize);

#ifdef __cplusplus
}
#endif

#endif /* _h_klib_writer_ */
