/* Somaplayer - Copyright (C) 2003-5 bakunin - Andrea Marchesini 
 *                                     <bakunin@autistici.org>
 *
 * This source code is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Public License as published 
 * by the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 *
 * This source code is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * Please refer to the GNU Public License for more details.
 *
 * You should have received a copy of the GNU Public License along with
 * this source code; if not, write to:
 * Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This program is released under the GPL with the additional exemption that
 * compiling, linking, and/or using OpenSSL is allowed.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#else
# error Use configure; make; make install
#endif
#ifdef ENABLE_OGG

#include "../../player.h"
#include "../../output.h"
#include "../../other.h"
#include "../../util.h"
#include "../../buffer.h"

size_t
ogg_read (void *ptr, size_t size, size_t nmemb, void *arg)
{
  int ret;
  int r, bytes;
  char *rptr = (char *) ptr;
  format_data_ogg *ogg = (format_data_ogg *) play->format_current->data;

  if (!ogg)
    return 0;

  ret = size * nmemb;
  bytes = 0;

  do
    {
      if (events.skip || events.quit)
	break;

      if (!(r = buffer_pop (rptr + bytes, ret)))
	return bytes;

      ret -= r;
      bytes += r;

    }
  while (ret);

  if (events.skip || events.quit)
    return -1;

  ogg->bytes += bytes;

  return bytes;
}

int
ogg_seek (void *arg, ogg_int64_t offset, int whence)
{
  if (whence == SEEK_CUR)
    {
      if (offset > 0)
	return buffer_pop (NULL, offset);
      else if (!offset)
	return 0;
    }

  return -1;
}

int
ogg_close (void *arg)
{
  return -1;
}

long
ogg_tell (void *arg)
{
  format_data_ogg *ogg = (format_data_ogg *) play->format_current->data;

  if (!ogg)
    return 0;

  return ogg->bytes;
}

void
ogg_run (void)
{
  unsigned char buf[4096];
  vorbis_comment *vc;
  vorbis_info *vi;
  int bitstream_1 = -1, bitstream_2 = -1;
  int ret;
  char **c;

  ov_callbacks ogg_callback = {
    &ogg_read,
    &ogg_seek,
    &ogg_close,
    &ogg_tell
  };

  format_data_ogg *ogg = (format_data_ogg *) play->format_current->data;

  if (!ogg)
    return;

  play->play = 0;

  ogg->time = 0;
  ogg->duration = 0;
  ogg->bytes = 0;

  if (ov_open_callbacks (play, &ogg->ogg, NULL, 0, ogg_callback) < 0)
    {
      msg (_("Error ogg init."));
      return;
    }

  vi = ov_info (&ogg->ogg, -1);
  vc = ov_comment (&ogg->ogg, -1);

  if (!vi || !vc)
    fatal (_("Error: memory."));

  if (vi->channels > 2)
    {
      msg (_("No > 2 channels."));
      return;
    }

  fprintf (stderr, _("\n* Info File Audio:\n"));

  if ((c = vc->user_comments))
    {
      while (*c)
	{
	  fprintf (stderr, "\t%s\n", *c);
	  ++c;
	}
    }

  fprintf (stderr, _("\tLayer %d - %ld Hz - Channels %d\n"), vi->version,
	   vi->rate, vi->channels);
  fprintf (stderr,
	   _("\tBitrate hints: upper %ld - " "nominal %ld - " "lower %ld - "
	     "window %ld\n"), vi->bitrate_upper, vi->bitrate_nominal,
	   vi->bitrate_lower, vi->bitrate_window);
  fprintf (stderr, _("\tEncoder %s\n"), vc->vendor);

  msg (_("Play file audio: %s\n"), playing_name ());

  ogg->duration = (long) ov_pcm_total (&ogg->ogg, -1);

  while (1)
    {
      play->play = 1;

      if (events.skip || events.quit)
	break;

      ogg->time = (double) ov_pcm_tell (&ogg->ogg) / (double) vi->rate;

      ret =
	ov_read (&ogg->ogg, (void *) buf, 4096, big_endian (), 2, 1,
		 &bitstream_1);

      if (ret == 0)
	break;
      else if (ret < 0)
	continue;

      if ((bitstream_2 == -1 || bitstream_2 != bitstream_1)
	  || !output_check ())
	{

	  output_close ();
	  output_open (vi->rate, vi->channels, 16);

	  bitstream_2 = bitstream_1;
	}

      output_write (buf, ret);

    }

  output_close ();
  play->play = 0;

  ov_clear (&ogg->ogg);
}

char *
ogg_get_time (void)
{
  static char output[13];
  long min;
  double sec;

  format_data_ogg *ogg = (format_data_ogg *) play->format_current->data;
  if (!ogg)
    return NULL;

  min = (long) ogg->time / (long) 60;
  sec = ogg->time - 60.0f * min;

  snprintf (output, 13, "T: %02li:%05.2f ", min, sec);

  return output;
}

#endif
