/***************************************************************************
    smb4kmounter.h  -  The core class that mounts the shares.
                             -------------------
    begin                : Die Jun 10 2003
    copyright            : (C) 2003-2008 by Alexander Reinholdt
    email                : dustpuppy@users.berlios.de
 ***************************************************************************/

/***************************************************************************
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful, but   *
 *   WITHOUT ANY WARRANTY; without even the implied warranty of            *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   General Public License for more details.                              *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,   *
 *   MA  02111-1307 USA                                                    *
 ***************************************************************************/

#ifndef SMB4KMOUNTER_H
#define SMB4KMOUNTER_H

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

// Qt includes
#include <QObject>
#include <QQueue>
#include <QList>
#include <QFile>
#include <QString>

// KDE includes
#include <kprocess.h>
#include <kdemacros.h>

// forward declarations
class Smb4KMounterPrivate;
class Smb4KMounterQueueContainer;
class Smb4KShare;


/**
 * This is one of the core classes of Smb4K. It manages the mounting
 * and unmounting of remote Samba/Windows shares. Additionally it maintains a
 * list of all mounts with SMBFS and CIFS file system that are present
 * on the system.
 *
 * @author Alexander Reinholdt <dustpuppy@users.berlios.de>
 */

class KDE_EXPORT Smb4KMounter : public QObject
{
  Q_OBJECT

  public:
    /**
     * The constructor.
     */
    Smb4KMounter( QObject *parent = 0 );
    /**
     * The destructor.
     */
    ~Smb4KMounter();

    /**
     * Aborts a running process.
     */
    void abort();

    /**
     * Enumeration for the state of the process.
     */
    enum State{ TriggerRemounts,
                Remount,
                Import,
                Mount,
                Unmount,
                UnmountAll,
                Idle };

    /**
     * Unmounts a share. This can either be done the "normal" way, or you may
     * force it. If you decide the force the unmounting by setting @p force to TRUE,
     * a lazy unmount will be initiated. With the parameter @p noMessage you can
     * suppress any error messages.
     *
     * @param share       The share object that should be unmounted.
     *
     * @param force       Force the unmounting of the share.
     *
     * @param noMessage   Determines whether this function should emit an error code in
     *                    case of an error. The default value is FALSE.
     */
    void unmountShare( Smb4KShare *share,
                       bool force = false,
                       bool noMessage = false );

    /**
     * Unmounts all shares at once.
     */
    void unmountAllShares();

    /**
     * Mounts a share.
     *
     * @param share       The Smb4KShare item that is representing the share.
     */
    void mountShare( Smb4KShare *share );

    /**
     * This function reports if the mounter is running or not.
     *
     * @returns TRUE if the mounter is running and FALSE otherwise.
     */
    bool isRunning() { return m_working; }

    /**
     * This function executes Smb4KMounter::slotAboutToQuit(). Under normal circumstances,
     * there is no need to use this function, because everything that is done here is
     * also done when QApplication::aboutToQuit() is emitted. However, it comes in handy
     * when you need to perform last actions in a plugin.
     */
    void prepareForShutdown();

    /**
     * This function initializes import of mounted shares and the remounting of recently
     * used shares.
     */
    void init();

  signals:
    /**
     * This signal emits the run state.
     *
     * @param state           The so-called run state. There are several defined
     *                        in the smb4kdefs.h header file.
     */
    void state( int state );

    /**
     * Tells the program, that the shares list has been updated.
     */
    void updated();

    /**
     * This signal is emitted when a share has successfully been mounted or
     * if it has already been mounted.
     *
     * @param share            The share that was just mounted.
     */
    void mounted( Smb4KShare *share );

    /**
     * This signal is emitted just before a share is unmounted.
     *
     * @param share            The share that is going to be unmounted.
     */
    void aboutToUnmount( Smb4KShare *share );

  protected:
    /**
     * Starts a process.
     */
    void startProcess();

    /**
     * Is called, when the process ended. It either calls one of the
     * process*() functions if the process exited normally or throws
     * an error if a problem occurred.
     *
     * @param exitCode           The exit code of the process
     *
     * @param exitStatus         The exit status of the process (@see QProcess::ExitStatus)
     */
    void endProcess( int exitCode,
                     QProcess::ExitStatus exitStatus );

    /**
     * Finishes the mounting of shares.
     */
    void processMount();

    /**
     * Finishes the unmounting of a single share.
     */
    void processUnmount();

    /**
     * Reimplemented from QObject to process the queue.
     *
     * @param event             The QTimerEvent event
     */
    void timerEvent( QTimerEvent *event );

  protected slots:
    /**
     * This slot is called, when the process exited. It invokes endProcess().
     *
     * @param exitCode           The exit code of the process
     *
     * @param exitStatus         The exit status of the process (@see QProcess::ExitStatus)
     */
    void slotProcessFinished( int exitCode,
                              QProcess::ExitStatus exitStatus );

    /**
     * This slot is invoked when an error occurred with the process.
     *
     * @param errorCode       The error code returned by the process
     */
    void slotProcessError( QProcess::ProcessError errorCode );

    /**
     * This slot is called by the QApplication::aboutToQuit() signal.
     * Is does everything that has to be done before the program
     * really exits.
     */
    void slotAboutToQuit();

  private:
    /**
     * Trigger the remounting of shares that were used
     * in the last session.
     */
    void triggerRemounts();

    /**
     * Imports mounted shares.
     */
    void import();

    /**
     * Mounts a selected share.
     */
    void mount( const Smb4KShare &share );

    /**
     * Unmounts a share.
     */
    void unmount( const Smb4KShare &share,
                  bool noMessage,
                  bool force = false );

    /**
     * Unmounts all shares at once.
     */
    void unmountAll();

    /**
     * Checks the file system of the share and its accessibility and set
     * the appropriate values accordingly. Additionally, the free and total
     * disk space are determined.
     *
     * @param share           The share that should be checked.
     */
    void check( Smb4KShare *share );

    /**
     * The KProcess object.
     */
    KProcess *m_proc;

    /**
     * The queue, where the incoming requests are stored.
     */
    QQueue<Smb4KMounterQueueContainer> m_queue;

    /**
     * Determines, whether the mounter is running or not.
     */
    bool m_working;

    /**
     * The internal state of the process. Do not mix this up with
     * the state that's emitted to notify the application about changes.
     */
    int m_state;

    /**
     * If the process was aborted, this property is TRUE and FALSE
     * otherwise
     */
    bool m_aborted;

    /**
     * The process error code
     */
    QProcess::ProcessError m_process_error;

#ifndef __FreeBSD__
    /**
     * Makes sure that the error message concerning the missing of
     * the file /proc/mounts is only shown once.
     */
    bool m_proc_error;

    /**
     * This file object points to /proc/mounts.
     */
    QFile m_proc_mounts;
#endif

    /**
     * This is the pointer to the private helper class.
     */
    Smb4KMounterPrivate *m_priv;
};

#endif
