//  Copyright (c) CNES  2008
//
//  This software is part of CelestLab, a CNES toolbox for Scilab
//
//  This software is governed by the CeCILL  license under French law and
//  abiding by the rules of distribution of free software.  You can  use,
//  modify and/ or redistribute the software under the terms of the CeCILL
//  license as circulated by CEA, CNRS and INRIA at the following URL
//  'http://www.cecill.info'.

function [betaa] = CL_gm_raan2beta(inc,raan,alpha_sun,delta_sun)
// Right ascension of ascending node to beta angle
//
// Calling Sequence
// [betaa] = CL_gm_raan2beta(inc,raan,alpha_sun,delta_sun)
//
// Description
// <itemizedlist><listitem>
// <p>Computes the beta angle (<b>betaa</b>), which is the angle between 
// the orbit plane and the Sun direction. </p>
// <p> The beta angle is a function of the orbit's right ascension of ascending node
// (<b>raan</b>) and inclination (<b>inc</b>) 
// and the Sun's right ascension (<b>alpha_sun</b>) and
// declination (<b>delta_sun</b>). </p>
// <p> <inlinemediaobject><imageobject><imagedata fileref="beta.gif"/></imageobject></inlinemediaobject></p>
// <p></p></listitem>
// <listitem>
// <p><b>Note:</b></p>
// <p>Beta is a signed quantity. It is positive when the Sun direction is less than 90 degrees
// from the orbit's angular momentum vector. </p>
// </listitem>
// </itemizedlist>
//
// Parameters
// inc: Orbit inclination [rad] (1xN)
// raan: Orbit right ascension of ascending node [rad] (1xN)
// alpha_sun: Sun right ascension [rad] (1xN)
// delta_sun: Sun declination [rad] (1xN)
// betaa: Beta angle [rad] (1xN)
//
// See also
// CL_gm_beta2raan
//
// Authors
// CNES - DCT/SB
//
// Examples
// // Get alpha_sun and delta_sun
// cjd = 20050;
// [r_sun,rs] = CL_mod_moonSunG50(cjd,'s');
// r_sph = CL_co_car2sph(r_sun);
// alpha_sun = r_sph(1);
// delta_sun = r_sph(2);
// raan = %pi/4;
// inc = CL_deg2rad(98.7);
// betaa = CL_gm_raan2beta(inc,raan,alpha_sun,delta_sun)

// Declarations:


// Code:

betaa = asin( cos(delta_sun).*sin(inc).*sin(raan-alpha_sun) + sin(delta_sun).*cos(inc) );

endfunction
